<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaksi;
use App\Models\TiketWisata;
use App\Models\ProdukUmkm;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TransaksiController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Transaksi::with(['transactable']);
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('reference_id', 'like', "%{$search}%")
                  ->orWhere('merchant_ref', 'like', "%{$search}%")
                  ->orWhere('customer_name', 'like', "%{$search}%")
                  ->orWhere('customer_email', 'like', "%{$search}%");
            });
        }
        
        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        // Filter by payment method
        if ($request->filled('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }
        
        // Filter by type
        if ($request->filled('transactable_type')) {
            $query->where('transactable_type', $request->transactable_type);
        }
        
        // Filter by date range
        if ($request->filled('tanggal_dari') && $request->filled('tanggal_sampai')) {
            $query->whereBetween('created_at', [$request->tanggal_dari, $request->tanggal_sampai]);
        }
        
        $transaksi = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();
        
        // Statistics
        $stats = [
            'total' => Transaksi::count(),
            'paid' => Transaksi::where('status', 'PAID')->count(),
            'unpaid' => Transaksi::where('status', 'UNPAID')->count(),
            'expired' => Transaksi::where('status', 'EXPIRED')->count(),
            'failed' => Transaksi::where('status', 'FAILED')->count(),
            'refunded' => Transaksi::where('status', 'refunded')->count(),
        ];
        
        $revenueStats = [
            'total_amount' => Transaksi::where('status', 'PAID')->sum('amount'),
            'total_fee' => Transaksi::where('status', 'PAID')->sum('total_fee'),
            'net_amount' => Transaksi::where('status', 'PAID')->sum('amount') - 
                           Transaksi::where('status', 'PAID')->sum('total_fee'),
        ];
        
        return view('admin.transaksi.index', compact('transaksi', 'stats', 'revenueStats'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(Transaksi $transaksi)
    {
        $transaksi->load(['transactable']);
        
        return view('admin.transaksi.show', compact('transaksi'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    public function refund(Transaksi $transaksi)
    {
        if ($transaksi->status !== 'PAID') {
            return response()->json([
                'success' => false,
                'message' => 'Hanya transaksi yang sudah dibayar yang dapat direfund.'
            ], 422);
        }
        
        if ($transaksi->status === 'refunded') {
            return response()->json([
                'success' => false,
                'message' => 'Transaksi sudah direfund sebelumnya.'
            ], 422);
        }
        
        // Process refund
        $transaksi->update([
            'status' => 'refunded',
            'refund_date' => now(),
            'refund_amount' => $transaksi->amount,
            'refund_reason' => request('reason', 'Refund by admin')
        ]);
        
        // Update related transactable if needed
        if ($transaksi->transactable_type === TiketWisata::class) {
            $tiket = $transaksi->transactable;
            $tiket->update([
                'status_tiket' => 'refunded',
                'status_tiket' => 'dibatalkan',
                'tanggal_pembatalan' => now(),
                'alasan_pembatalan' => 'Refund oleh admin'
            ]);
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Refund berhasil diproses.',
            'refund_amount' => $transaksi->refund_amount
        ]);
    }
    
    public function callback(Request $request)
    {
        // Handle Tripay callback
        $callbackData = $request->all();
        
        // Verify callback signature
        $callbackSignature = $request->header('X-Callback-Signature');
        
        if (!$this->verifyCallbackSignature($callbackData, $callbackSignature)) {
            return response()->json(['message' => 'Invalid signature'], 400);
        }
        
        $referenceId = $callbackData['reference'];
        $status = $callbackData['status'];
        
        // Find transaction
        $transaksi = Transaksi::where('reference_id', $referenceId)->first();
        
        if (!$transaksi) {
            return response()->json(['message' => 'Transaction not found'], 404);
        }
        
        // Update transaction status
        $transaksi->update([
            'status' => $status,
            'paid_at' => $status === 'PAID' ? now() : null,
            'payment_method' => $callbackData['payment_method'] ?? $transaksi->payment_method,
            'callback_data' => $callbackData
        ]);
        
        // Update related model based on transaction type
        if ($status === 'PAID') {
            if ($transaksi->transactable_type === TiketWisata::class) {
                $tiket = $transaksi->transactable;
                $tiket->update(['status_tiket' => 'paid']);
                
                // Generate QR code for ticket
                $tiket->generateQRCode();
            } elseif ($transaksi->transactable_type === ProdukUmkm::class) {
                // Handle product order
                $produk = $transaksi->transactable;
                // Update stock or process order
            }
        } elseif (in_array($status, ['EXPIRED', 'FAILED'])) {
            if ($transaksi->transactable_type === TiketWisata::class) {
                $tiket = $transaksi->transactable;
                $tiket->update([
                    'status_tiket' => 'cancelled',
                    'status_tiket' => 'expired'
                ]);
            }
        }
        
        return response()->json(['message' => 'Callback processed successfully']);
    }
    
    public function laporan(Request $request)
    {
        $periode = $request->get('periode', 'bulan');
        $tahun = $request->get('tahun', now()->year);
        $bulan = $request->get('bulan', now()->month);
        
        $query = Transaksi::where('status', 'PAID');
        
        if ($periode === 'harian') {
            $query->whereYear('paid_at', $tahun)
                  ->whereMonth('paid_at', $bulan);
        } elseif ($periode === 'bulanan') {
            $query->whereYear('paid_at', $tahun);
        }
        
        $transaksiData = $query->get();
        
        // Process statistics
        $statistik = [
            'total_transaksi' => $transaksiData->count(),
            'total_amount' => $transaksiData->sum('amount'),
            'total_fee' => $transaksiData->sum('total_fee'),
            'net_revenue' => $transaksiData->sum('amount') - $transaksiData->sum('total_fee'),
        ];
        
        // Group by payment method
        $perMetode = $transaksiData->groupBy('payment_method')->map(function($transactions, $method) {
            return [
                'metode' => $method,
                'total_transaksi' => $transactions->count(),
                'total_amount' => $transactions->sum('amount'),
                'total_fee' => $transactions->sum('total_fee'),
            ];
        });
        
        // Group by transaction type
        $perTipe = $transaksiData->groupBy('transactable_type')->map(function($transactions, $type) {
            $typeName = $type === TiketWisata::class ? 'Tiket Wisata' : 'Produk UMKM';
            return [
                'tipe' => $typeName,
                'total_transaksi' => $transactions->count(),
                'total_amount' => $transactions->sum('amount'),
            ];
        });
        
        return view('admin.transaksi.laporan', compact('statistik', 'perMetode', 'perTipe', 'periode', 'tahun', 'bulan'));
    }
    
    public function export(Request $request)
    {
        $periode = $request->get('periode', 'bulan');
        $tahun = $request->get('tahun', now()->year);
        $bulan = $request->get('bulan', now()->month);
        
        $query = Transaksi::with(['transactable']);
        
        if ($periode === 'harian') {
            $query->whereYear('created_at', $tahun)
                  ->whereMonth('created_at', $bulan);
        } elseif ($periode === 'bulanan') {
            $query->whereYear('created_at', $tahun);
        }
        
        $transaksi = $query->orderBy('created_at', 'desc')->get();
        
        // Generate CSV
        $filename = "transaksi_{$periode}_{$tahun}" . ($periode === 'harian' ? "_{$bulan}" : '') . ".csv";
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];
        
        $callback = function() use ($transaksi) {
            $file = fopen('php://output', 'w');
            
            // CSV headers
            fputcsv($file, [
                'Reference ID',
                'Merchant Ref',
                'Customer Name',
                'Customer Email',
                'Amount',
                'Fee',
                'Net Amount',
                'Payment Method',
                'Status',
                'Type',
                'Created At',
                'Paid At'
            ]);
            
            foreach ($transaksi as $t) {
                fputcsv($file, [
                    $t->reference_id,
                    $t->merchant_ref,
                    $t->customer_name,
                    $t->customer_email,
                    $t->amount,
                    $t->total_fee,
                    $t->amount - $t->total_fee,
                    $t->payment_method,
                    $t->status,
                    $t->transactable_type === TiketWisata::class ? 'Tiket Wisata' : 'Produk UMKM',
                    $t->created_at->format('Y-m-d H:i:s'),
                    $t->paid_at ? $t->paid_at->format('Y-m-d H:i:s') : ''
                ]);
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    public function bulkAction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'action' => 'required|in:refund,delete',
            'ids' => 'required|array',
            'ids.*' => 'exists:transaksi,id'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid.',
                'errors' => $validator->errors()
            ], 422);
        }
        
        $transaksiList = Transaksi::whereIn('id', $request->ids);
        $count = $transaksiList->count();
        
        switch ($request->action) {
            case 'refunded':
                $updated = 0;
                foreach ($transaksiList->where('status', 'PAID')->get() as $transaksi) {
                    $transaksi->update([
                        'status' => 'refunded',
                        'refund_date' => now(),
                        'refund_amount' => $transaksi->amount,
                        'refund_reason' => 'Bulk refund by admin'
                    ]);
                    
                    // Update related transactable
                    if ($transaksi->transactable_type === TiketWisata::class) {
                        $tiket = $transaksi->transactable;
                        $tiket->update([
                            'status_tiket' => 'refunded',
                            'status_tiket' => 'dibatalkan',
                            'tanggal_pembatalan' => now(),
                            'alasan_pembatalan' => 'Bulk refund by admin'
                        ]);
                    }
                    $updated++;
                }
                $message = "{$updated} transaksi berhasil direfund.";
                break;
                
            case 'delete':
                $transaksiList->whereIn('status', ['EXPIRED', 'FAILED', 'CANCELLED'])->delete();
                $message = "Transaksi yang dapat dihapus telah berhasil dihapus.";
                break;
        }
        
        return response()->json([
            'success' => true,
            'message' => $message
        ]);
    }
    
    public function sync(Request $request)
    {
        $syncCount = 0;
        $errorCount = 0;
        $errors = [];
        
        // Get pending transactions to sync
        $pendingTransactions = Transaksi::whereIn('status', ['UNPAID', 'PENDING'])
                                       ->where('created_at', '>=', now()->subDays(7)) // Only sync last 7 days
                                       ->get();
        
        foreach ($pendingTransactions as $transaksi) {
            try {
                // Call Tripay API to check transaction status
                $response = $this->checkTransactionStatus($transaksi->reference_id);
                
                if ($response && isset($response['data']['status'])) {
                    $newStatus = $response['data']['status'];
                    $oldStatus = $transaksi->status;
                    
                    if ($newStatus !== $oldStatus) {
                        // Update transaction status
                        $transaksi->update([
                            'status' => $newStatus,
                            'paid_at' => $newStatus === 'PAID' ? now() : null,
                            'payment_method' => $response['data']['payment_method'] ?? $transaksi->payment_method,
                            'last_synced_at' => now()
                        ]);
                        
                        // Update related model based on transaction type
                        if ($newStatus === 'PAID') {
                            if ($transaksi->transactable_type === TiketWisata::class) {
                                $tiket = $transaksi->transactable;
                                $tiket->update(['status_tiket' => 'paid']);
                            }
                        } elseif (in_array($newStatus, ['EXPIRED', 'FAILED'])) {
                            if ($transaksi->transactable_type === TiketWisata::class) {
                                $tiket = $transaksi->transactable;
                                $tiket->update(['status_tiket' => 'expired']);
                            }
                        }
                        
                        $syncCount++;
                    }
                }
            } catch (\Exception $e) {
                $errorCount++;
                $errors[] = "Error syncing transaction {$transaksi->reference_id}: " . $e->getMessage();
            }
        }
        
        $message = "Sinkronisasi selesai. {$syncCount} transaksi berhasil disinkronkan.";
        if ($errorCount > 0) {
            $message .= " {$errorCount} transaksi gagal disinkronkan.";
        }
        
        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => $message,
                'synced_count' => $syncCount,
                'error_count' => $errorCount,
                'errors' => $errors
            ]);
        }
        
        return redirect()->route('admin.transaksi.index')
                        ->with('success', $message);
    }

    private function checkTransactionStatus($referenceId)
    {
        try {
            $tripayConfig = config('tripay');
            $apiKey = $tripayConfig['api_key'];
            $privateKey = $tripayConfig['private_key'];
            $merchantCode = $tripayConfig['merchant_code'];
            $baseUrl = $tripayConfig['base_url'];
            
            // Create signature
            $payload = $merchantCode . $referenceId . $privateKey;
            $signature = hash('sha256', $payload);
            
            // Make API request to Tripay
            $curl = curl_init();
            
            curl_setopt_array($curl, [
                CURLOPT_FRESH_CONNECT => true,
                CURLOPT_URL => $baseUrl . '/transaction/detail?reference=' . $referenceId,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HEADER => false,
                CURLOPT_HTTPHEADER => [
                    'Authorization: Bearer ' . $apiKey,
                    'X-Signature: ' . $signature
                ],
                CURLOPT_FAILONERROR => false,
                CURLOPT_IPRESOLVE => CURL_IPRESOLVE_V4
            ]);
            
            $response = curl_exec($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);
            
            if ($httpCode === 200) {
                return json_decode($response, true);
            }
            
            return null;
        } catch (\Exception $e) {
            \Log::error('Tripay sync error: ' . $e->getMessage());
            return null;
        }
    }
    
    private function verifyCallbackSignature($data, $signature)
    {
        // Implement Tripay signature verification
        $privateKey = config('tripay.private_key');
        
        if (!$privateKey) {
            return false;
        }
        
        $callbackSignature = hash_hmac('sha256', json_encode($data), $privateKey);
        
        return hash_equals($signature, $callbackSignature);
    }
}
