<?php

namespace App\Http\Controllers;

use App\Models\AgendaDesa;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class AgendaDesaController extends Controller
{
    public function index(Request $request)
    {
        $query = AgendaDesa::with('creator');

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter berdasarkan jenis kegiatan
        if ($request->filled('jenis')) {
            $query->where('jenis_kegiatan', $request->jenis);
        }

        // Filter berdasarkan bulan
        if ($request->filled('bulan')) {
            $query->whereMonth('tanggal_mulai', $request->bulan);
        }

        // Pencarian
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('judul', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhere('tempat', 'like', "%{$search}%");
            });
        }

        $agenda = $query->latest('tanggal_mulai')->paginate(10);
        
        return view('admin.agenda-desa.index', compact('agenda'));
    }

    public function create()
    {
        $jenisKegiatan = [
            'Rapat Desa', 'Musyawarah Desa', 'Gotong Royong', 'Pelatihan',
            'Sosialisasi', 'Acara Budaya', 'Olahraga', 'Keagamaan',
            'Pembangunan', 'Kesehatan', 'Pendidikan', 'Lainnya'
        ];
        
        $targetPeserta = [
            'Semua Warga', 'RT Tertentu', 'RW Tertentu', 'Dusun Tertentu', 'Kelompok Usia'
        ];
        
        return view('admin.agenda-desa.create', compact('jenisKegiatan', 'targetPeserta'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'judul' => 'required|string|max:255',
            'deskripsi' => 'required|string',
            'jenis_kegiatan' => 'required|string|max:100',
            'tanggal_mulai' => 'required|date|after_or_equal:today',
            'tanggal_selesai' => 'required|date|after_or_equal:tanggal_mulai',
            'tempat' => 'required|string|max:255',
            'alamat_lengkap' => 'nullable|string|max:500',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'penyelenggara' => 'required|string|max:255',
            'penanggung_jawab' => 'required|string|max:255',
            'kontak_person' => 'nullable|string|max:20',
            'target_peserta' => 'required|string|max:100',
            'filter_peserta' => 'nullable|array',
            'kapasitas' => 'nullable|integer|min:1',
            'perlu_daftar' => 'boolean',
            'status' => 'required|in:Draft,Terjadwal,Berlangsung,Selesai,Dibatalkan',
            'tampil_public' => 'boolean',
            'banner' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'kirim_reminder' => 'boolean',
            'waktu_reminder' => 'nullable|array'
        ]);

        $data = $request->except(['banner']);
        $data['created_by'] = auth()->id();

        // Handle banner upload
        if ($request->hasFile('banner')) {
            $banner = $request->file('banner');
            $bannerName = time() . '_banner_' . $banner->getClientOriginalName();
            $data['banner'] = $banner->storeAs('agenda/banner', $bannerName, 'public');
        }

        $agenda = AgendaDesa::create($data);

        // Log activity
        AuditLog::logDataChange('create', $agenda, null, $data);

        return redirect()->route('admin.agenda-desa.index')
                        ->with('success', 'Agenda berhasil dibuat');
    }

    public function show(AgendaDesa $agendaDesa)
    {
        $agendaDesa->load('creator');
        $targetPeserta = $agendaDesa->getTargetPeserta();
        
        return view('admin.agenda-desa.show', compact('agendaDesa', 'targetPeserta'));
    }

    public function edit(AgendaDesa $agendaDesa)
    {
        $jenisKegiatan = [
            'Rapat Desa', 'Musyawarah Desa', 'Gotong Royong', 'Pelatihan',
            'Sosialisasi', 'Acara Budaya', 'Olahraga', 'Keagamaan',
            'Pembangunan', 'Kesehatan', 'Pendidikan', 'Lainnya'
        ];
        
        $targetPeserta = [
            'Semua Warga', 'RT Tertentu', 'RW Tertentu', 'Dusun Tertentu', 'Kelompok Usia'
        ];
        
        return view('admin.agenda-desa.edit', compact('agendaDesa', 'jenisKegiatan', 'targetPeserta'));
    }

    public function update(Request $request, AgendaDesa $agendaDesa)
    {
        $request->validate([
            'judul' => 'required|string|max:255',
            'deskripsi' => 'required|string',
            'jenis_kegiatan' => 'required|string|max:100',
            'tanggal_mulai' => 'required|date',
            'tanggal_selesai' => 'required|date|after_or_equal:tanggal_mulai',
            'tempat' => 'required|string|max:255',
            'alamat_lengkap' => 'nullable|string|max:500',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'penyelenggara' => 'required|string|max:255',
            'penanggung_jawab' => 'required|string|max:255',
            'kontak_person' => 'nullable|string|max:20',
            'target_peserta' => 'required|string|max:100',
            'filter_peserta' => 'nullable|array',
            'kapasitas' => 'nullable|integer|min:1',
            'perlu_daftar' => 'boolean',
            'status' => 'required|in:Draft,Terjadwal,Berlangsung,Selesai,Dibatalkan',
            'tampil_public' => 'boolean',
            'banner' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'kirim_reminder' => 'boolean',
            'waktu_reminder' => 'nullable|array'
        ]);

        $oldData = $agendaDesa->toArray();
        $data = $request->except(['banner']);

        // Handle banner upload
        if ($request->hasFile('banner')) {
            // Delete old banner
            if ($agendaDesa->banner) {
                Storage::disk('public')->delete($agendaDesa->banner);
            }
            
            $banner = $request->file('banner');
            $bannerName = time() . '_banner_' . $banner->getClientOriginalName();
            $data['banner'] = $banner->storeAs('agenda/banner', $bannerName, 'public');
        }

        $agendaDesa->update($data);

        // Log activity
        AuditLog::logDataChange('update', $agendaDesa, $oldData, $data);

        return redirect()->route('admin.agenda-desa.index')
                        ->with('success', 'Agenda berhasil diperbarui');
    }

    public function destroy(AgendaDesa $agendaDesa)
    {
        $oldData = $agendaDesa->toArray();

        // Delete banner if exists
        if ($agendaDesa->banner) {
            Storage::disk('public')->delete($agendaDesa->banner);
        }

        $agendaDesa->delete();

        // Log activity
        AuditLog::logDataChange('delete', $agendaDesa, $oldData, null);

        return redirect()->route('admin.agenda-desa.index')
                        ->with('success', 'Agenda berhasil dihapus');
    }

    public function updateStatus(Request $request, AgendaDesa $agendaDesa)
    {
        $request->validate([
            'status' => 'required|in:Draft,Terjadwal,Berlangsung,Selesai,Dibatalkan'
        ]);

        $oldStatus = $agendaDesa->status;
        $agendaDesa->update(['status' => $request->status]);

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'UPDATE_STATUS',
            'module' => 'AgendaDesa',
            'table_name' => 'agenda_desa',
            'record_id' => $agendaDesa->id,
            'description' => "Update status agenda from {$oldStatus} to {$request->status}",
            'metadata' => [
                'old_status' => $oldStatus,
                'new_status' => $request->status
            ]
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Status agenda berhasil diperbarui'
        ]);
    }

    public function calendar(Request $request)
    {
        $month = $request->get('month', now()->month);
        $year = $request->get('year', now()->year);
        
        $agenda = AgendaDesa::whereMonth('tanggal_mulai', $month)
                           ->whereYear('tanggal_mulai', $year)
                           ->whereIn('status', ['Terjadwal', 'Berlangsung'])
                           ->get();

        return view('admin.agenda-desa.calendar', compact('agenda', 'month', 'year'));
    }

    public function apiCalendar(Request $request)
    {
        $start = $request->get('start');
        $end = $request->get('end');
        
        $agenda = AgendaDesa::whereBetween('tanggal_mulai', [$start, $end])
                           ->orWhereBetween('tanggal_selesai', [$start, $end])
                           ->get();

        $events = $agenda->map(function($item) {
            return [
                'id' => $item->id,
                'title' => $item->judul,
                'start' => $item->tanggal_mulai->format('Y-m-d H:i:s'),
                'end' => $item->tanggal_selesai->format('Y-m-d H:i:s'),
                'description' => $item->deskripsi,
                'location' => $item->tempat,
                'color' => $this->getStatusColor($item->status),
                'url' => route('admin.agenda-desa.show', $item->id)
            ];
        });

        return response()->json($events);
    }

    public function reminder(AgendaDesa $agendaDesa)
    {
        if (!$agendaDesa->shouldSendReminder()) {
            return response()->json([
                'success' => false,
                'message' => 'Reminder tidak perlu dikirim'
            ]);
        }

        $targetPeserta = $agendaDesa->getTargetPeserta();
        $sentCount = 0;

        foreach ($targetPeserta as $peserta) {
            // Send reminder logic here (WhatsApp, SMS, Email)
            // Implementation depends on notification service
            $sentCount++;
        }

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'SEND_REMINDER',
            'module' => 'AgendaDesa',
            'description' => "Send reminder untuk agenda: {$agendaDesa->judul}",
            'metadata' => [
                'agenda_id' => $agendaDesa->id,
                'sent_count' => $sentCount
            ]
        ]);

        return response()->json([
            'success' => true,
            'message' => "Reminder berhasil dikirim ke {$sentCount} peserta"
        ]);
    }

    public function hasilKegiatan(Request $request, AgendaDesa $agendaDesa)
    {
        $request->validate([
            'hasil_kegiatan' => 'required|string',
            'dokumentasi' => 'nullable|array',
            'dokumentasi.*' => 'image|mimes:jpeg,png,jpg|max:5120',
            'jumlah_peserta' => 'required|integer|min:0'
        ]);

        $data = $request->only(['hasil_kegiatan', 'jumlah_peserta']);
        
        // Handle dokumentasi upload
        if ($request->hasFile('dokumentasi')) {
            $dokumentasiPaths = [];
            foreach ($request->file('dokumentasi') as $file) {
                $fileName = time() . '_' . uniqid() . '_' . $file->getClientOriginalName();
                $path = $file->storeAs('agenda/dokumentasi', $fileName, 'public');
                $dokumentasiPaths[] = $path;
            }
            $data['dokumentasi'] = json_encode($dokumentasiPaths);
        }

        $data['status'] = 'Selesai';
        $agendaDesa->update($data);

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'UPDATE_HASIL',
            'module' => 'AgendaDesa',
            'table_name' => 'agenda_desa',
            'record_id' => $agendaDesa->id,
            'description' => "Update hasil kegiatan: {$agendaDesa->judul}",
            'metadata' => [
                'jumlah_peserta' => $data['jumlah_peserta']
            ]
        ]);

        return redirect()->back()
                        ->with('success', 'Hasil kegiatan berhasil disimpan');
    }

    public function publicIndex()
    {
        $agendaBulanIni = AgendaDesa::getAgendaBulanIni();
        $agendaMingguIni = AgendaDesa::getAgendaMingguIni();
        
        return view('public.agenda.index', compact('agendaBulanIni', 'agendaMingguIni'));
    }

    public function publicShow(AgendaDesa $agendaDesa)
    {
        if (!$agendaDesa->tampil_public || $agendaDesa->status === 'Draft') {
            abort(404);
        }

        return view('public.agenda.show', compact('agendaDesa'));
    }

    private function getStatusColor($status)
    {
        return match($status) {
            'Draft' => '#6c757d',
            'Terjadwal' => '#007bff',
            'Berlangsung' => '#ffc107',
            'Selesai' => '#28a745',
            'Dibatalkan' => '#dc3545',
            default => '#6c757d'
        };
    }
} 