<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Umkm;
use App\Models\ProdukUmkm;
use Illuminate\Http\Request;

class UmkmController extends Controller
{
    /**
     * Display a listing of UMKM
     */
    public function index(Request $request)
    {
        try {
            $query = Umkm::where('status', 'aktif');

            // Filter by category
            if ($request->has('kategori') && $request->kategori) {
                $query->where('kategori', $request->kategori);
            }

            // Search functionality
            if ($request->has('search') && $request->search) {
                $query->where(function($q) use ($request) {
                    $q->where('nama_umkm', 'like', '%' . $request->search . '%')
                      ->orWhere('nama_pemilik', 'like', '%' . $request->search . '%')
                      ->orWhere('deskripsi', 'like', '%' . $request->search . '%');
                });
            }

            $page = $request->get('page', 1);
            $limit = $request->get('limit', 10);

            $umkm = $query->orderBy('created_at', 'desc')
                         ->paginate($limit, ['*'], 'page', $page);

            return response()->json([
                'success' => true,
                'data' => $umkm->items(),
                'meta' => [
                    'current_page' => $umkm->currentPage(),
                    'last_page' => $umkm->lastPage(),
                    'per_page' => $umkm->perPage(),
                    'total' => $umkm->total(),
                ],
                'message' => 'Data UMKM berhasil diambil'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data UMKM',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified UMKM
     */
    public function show($id)
    {
        try {
            $umkm = Umkm::where('status', 'aktif')->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $umkm,
                'message' => 'Detail UMKM berhasil diambil'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'UMKM tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Display a listing of products
     */
    public function produkIndex(Request $request)
    {
        try {
            $query = ProdukUmkm::with('umkm')->where('status', 'aktif');

            // Filter by category
            if ($request->has('kategori') && $request->kategori) {
                $query->where('kategori', $request->kategori);
            }

            // Filter by UMKM
            if ($request->has('umkm_id') && $request->umkm_id) {
                $query->where('umkm_id', $request->umkm_id);
            }

            // Search functionality
            if ($request->has('search') && $request->search) {
                $query->where(function($q) use ($request) {
                    $q->where('nama_produk', 'like', '%' . $request->search . '%')
                      ->orWhere('deskripsi', 'like', '%' . $request->search . '%');
                });
            }

            $page = $request->get('page', 1);
            $limit = $request->get('limit', 10);

            $produk = $query->orderBy('created_at', 'desc')
                           ->paginate($limit, ['*'], 'page', $page);

            return response()->json([
                'success' => true,
                'data' => $produk->items(),
                'meta' => [
                    'current_page' => $produk->currentPage(),
                    'last_page' => $produk->lastPage(),
                    'per_page' => $produk->perPage(),
                    'total' => $produk->total(),
                ],
                'message' => 'Data produk berhasil diambil'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data produk',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified product
     */
    public function produkShow($id)
    {
        try {
            $produk = ProdukUmkm::with('umkm')->where('status', 'aktif')->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $produk,
                'message' => 'Detail produk berhasil diambil'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Produk tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }
}