<?php

namespace App\Http\Controllers;

use App\Models\InventarisDesa;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class InventarisDesaController extends Controller
{
    public function index(Request $request)
    {
        $query = InventarisDesa::with('creator');

        if ($request->filled('kategori')) {
            $query->byKategori($request->kategori);
        }

        if ($request->filled('kondisi')) {
            $query->byKondisi($request->kondisi);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_barang', 'like', "%{$search}%")
                  ->orWhere('kode_inventaris', 'like', "%{$search}%");
            });
        }

        $inventaris = $query->latest()->paginate(15);
        
        return view('admin.inventaris-desa.index', compact('inventaris'));
    }

    public function create()
    {
        $kategori = [
            'Tanah', 'Bangunan', 'Kendaraan', 'Elektronik', 'Furniture',
            'Alat Tulis Kantor', 'Alat Kesehatan', 'Alat Pertanian'
        ];
        
        return view('admin.inventaris-desa.create', compact('kategori'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nama_barang' => 'required|string|max:255',
            'kategori' => 'required|string|max:100',
            'harga_perolehan' => 'required|numeric|min:0',
            'kondisi' => 'required|string',
            'lokasi_penyimpanan' => 'required|string|max:255'
        ]);

        $data = $request->all();
        $data['created_by'] = Auth::id();
        $data['kode_inventaris'] = InventarisDesa::generateKodeInventaris($request->kategori);
        $data['nilai_sekarang'] = $request->harga_perolehan;

        $inventaris = InventarisDesa::create($data);

        AuditLog::logDataChange('create', $inventaris, null, $data);

        return redirect()->route('admin.inventaris-desa.index')
                        ->with('success', 'Data inventaris berhasil ditambahkan');
    }

    public function show(InventarisDesa $inventarisDesa)
    {
        return view('admin.inventaris-desa.show', compact('inventarisDesa'));
    }

    public function edit(InventarisDesa $inventarisDesa)
    {
        $kategori = [
            'Tanah', 'Bangunan', 'Kendaraan', 'Elektronik', 'Furniture',
            'Alat Tulis Kantor', 'Alat Kesehatan', 'Alat Pertanian'
        ];
        
        return view('admin.inventaris-desa.edit', compact('inventarisDesa', 'kategori'));
    }

    public function update(Request $request, InventarisDesa $inventarisDesa)
    {
        $request->validate([
            'nama_barang' => 'required|string|max:255',
            'kategori' => 'required|string|max:100',
            'harga_perolehan' => 'required|numeric|min:0',
            'kondisi' => 'required|string',
            'lokasi_penyimpanan' => 'required|string|max:255'
        ]);

        $oldData = $inventarisDesa->toArray();
        $inventarisDesa->update($request->all());

        AuditLog::logDataChange('update', $inventarisDesa, $oldData, $request->all());

        return redirect()->route('admin.inventaris-desa.index')
                        ->with('success', 'Data inventaris berhasil diperbarui');
    }

    public function destroy(InventarisDesa $inventarisDesa)
    {
        $oldData = $inventarisDesa->toArray();
        $inventarisDesa->delete();

        AuditLog::logDataChange('delete', $inventarisDesa, $oldData, null);

        return redirect()->route('admin.inventaris-desa.index')
                        ->with('success', 'Data inventaris berhasil dihapus');
    }

    public function qrCode(InventarisDesa $inventarisDesa)
    {
        $qrCode = $inventarisDesa->generateQrCode();
        
        return view('admin.inventaris-desa.qr-code', compact('inventarisDesa', 'qrCode'));
    }

    public function statistik()
    {
        $stats = [
            'total_nilai' => InventarisDesa::getTotalNilai(),
            'kondisi' => InventarisDesa::getStatistikKondisi(),
            'per_kategori' => InventarisDesa::selectRaw('kategori, COUNT(*) as total')
                                           ->groupBy('kategori')
                                           ->get()
        ];
        
        return view('admin.inventaris-desa.statistik', compact('stats'));
    }

    public function scanQr(Request $request)
    {
        $request->validate([
            'qr_data' => 'required|string'
        ]);

        try {
            $qrData = json_decode(base64_decode($request->qr_data), true);
            
            if (!$qrData || !isset($qrData['kode'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'QR Code tidak valid'
                ], 400);
            }

            $inventaris = InventarisDesa::where('kode_inventaris', $qrData['kode'])->first();
            
            if (!$inventaris) {
                return response()->json([
                    'success' => false,
                    'message' => 'Inventaris tidak ditemukan'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $inventaris
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal memproses QR Code: ' . $e->getMessage()
            ], 500);
        }
    }

    public function maintenance(Request $request, InventarisDesa $inventarisDesa)
    {
        $request->validate([
            'kondisi' => 'required|string',
            'keterangan' => 'nullable|string'
        ]);

        try {
            $oldKondisi = $inventarisDesa->kondisi;
            $inventarisDesa->update([
                'kondisi' => $request->kondisi,
                'terakhir_maintenance' => now(),
                'keterangan' => $request->keterangan
            ]);

            AuditLog::logDataChange('maintenance', $inventarisDesa, ['kondisi' => $oldKondisi], ['kondisi' => $request->kondisi]);

            return response()->json([
                'success' => true,
                'message' => 'Kondisi inventaris berhasil diperbarui'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal update kondisi: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateNilai(Request $request, InventarisDesa $inventarisDesa)
    {
        $request->validate([
            'nilai_sekarang' => 'required|numeric|min:0',
            'keterangan' => 'nullable|string'
        ]);

        try {
            $oldNilai = $inventarisDesa->nilai_sekarang;
            $inventarisDesa->update([
                'nilai_sekarang' => $request->nilai_sekarang,
                'keterangan' => $request->keterangan
            ]);

            AuditLog::logDataChange('update_nilai', $inventarisDesa, ['nilai_sekarang' => $oldNilai], ['nilai_sekarang' => $request->nilai_sekarang]);

            return response()->json([
                'success' => true,
                'message' => 'Nilai inventaris berhasil diperbarui'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal update nilai: ' . $e->getMessage()
            ], 500);
        }
    }

    public function bulkUpdateKondisi(Request $request)
    {
        $request->validate([
            'inventaris_ids' => 'required|array',
            'kondisi' => 'required|string',
            'keterangan' => 'nullable|string'
        ]);

        try {
            $updated = 0;
            foreach ($request->inventaris_ids as $id) {
                $inventaris = InventarisDesa::find($id);
                if ($inventaris) {
                    $inventaris->update([
                        'kondisi' => $request->kondisi,
                        'keterangan' => $request->keterangan
                    ]);
                    $updated++;
                }
            }

            return response()->json([
                'success' => true,
                'message' => "Berhasil update kondisi {$updated} inventaris"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal update kondisi: ' . $e->getMessage()
            ], 500);
        }
    }

    public function export($format = 'excel')
    {
        try {
            $inventaris = InventarisDesa::with('creator')->get();
            
            if ($format === 'excel') {
                // Implement Excel export
                return response()->json([
                    'success' => true,
                    'message' => 'Export Excel berhasil',
                    'data' => $inventaris
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Format export tidak didukung'
                ], 400);
            }

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal export: ' . $e->getMessage()
            ], 500);
        }
    }

    // API Methods
    public function apiIndex(Request $request)
    {
        try {
            $query = InventarisDesa::with('creator');

            if ($request->filled('kategori')) {
                $query->byKategori($request->kategori);
            }

            if ($request->filled('kondisi')) {
                $query->byKondisi($request->kondisi);
            }

            $inventaris = $query->latest()->paginate(15);

            return response()->json([
                'success' => true,
                'data' => $inventaris
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data inventaris: ' . $e->getMessage()
            ], 500);
        }
    }

    public function apiShow(InventarisDesa $inventarisDesa)
    {
        try {
            $inventarisDesa->load('creator');

            return response()->json([
                'success' => true,
                'data' => $inventarisDesa
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil detail inventaris: ' . $e->getMessage()
            ], 500);
        }
    }

    public function apiScanQr(Request $request)
    {
        return $this->scanQr($request);
    }
} 