<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\PpobService;
use App\Models\PpobTransaction;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class PpobController extends Controller
{
    protected $ppobService;

    public function __construct(PpobService $ppobService)
    {
        $this->ppobService = $ppobService;
    }

    /**
     * Display PPOB main page
     */
    public function index()
    {
        $products = $this->ppobService->getProducts();
        $recentTransactions = [];
        
        if (Auth::check()) {
            $recentTransactions = $this->ppobService->getUserTransactions(Auth::id(), 5);
        }

        return view('frontend.ppob.index', compact('products', 'recentTransactions'));
    }

    /**
     * Show product detail page
     */
    public function product($type)
    {
        $products = $this->ppobService->getProducts();
        
        if (!isset($products[$type])) {
            abort(404, 'Produk tidak ditemukan');
        }

        $productType = $type;
        $productInfo = $products[$type];
        $user = Auth::user();

        return view('frontend.ppob.product', compact('productType', 'productInfo', 'user'));
    }

    /**
     * Get denominations for specific provider/product type
     */
    public function getDenominations($productType, Request $request)
    {
        $provider = $request->input('provider');
        $denominations = [];
        
        try {
            if ($productType === 'pulsa') {
                $denominations = $this->ppobService->getPulsaDenominations($provider ?: 'TELKOMSEL');
            } elseif ($productType === 'paket_data') {
                $denominations = $this->ppobService->getDataPackages($provider ?: 'TELKOMSEL');
            } elseif ($productType === 'pln') {
                // PLN denominations
                $denominations = [
                    ['amount' => 20000, 'price' => 20500, 'description' => 'Token PLN 20K'],
                    ['amount' => 50000, 'price' => 50500, 'description' => 'Token PLN 50K'],
                    ['amount' => 100000, 'price' => 100500, 'description' => 'Token PLN 100K'],
                    ['amount' => 200000, 'price' => 200500, 'description' => 'Token PLN 200K'],
                    ['amount' => 500000, 'price' => 500500, 'description' => 'Token PLN 500K'],
                ];
            } elseif ($productType === 'pbb') {
                // PBB denominations (flexible amounts)
                $denominations = [
                    ['amount' => 50000, 'price' => 51000, 'description' => 'PBB Rp 50.000'],
                    ['amount' => 100000, 'price' => 101000, 'description' => 'PBB Rp 100.000'],
                    ['amount' => 200000, 'price' => 202000, 'description' => 'PBB Rp 200.000'],
                    ['amount' => 500000, 'price' => 505000, 'description' => 'PBB Rp 500.000'],
                    ['amount' => 1000000, 'price' => 1010000, 'description' => 'PBB Rp 1.000.000'],
                ];
            } elseif ($productType === 'bpjs') {
                // BPJS denominations
                $denominations = [
                    ['amount' => 42000, 'price' => 43000, 'description' => 'BPJS Kelas III'],
                    ['amount' => 110000, 'price' => 111000, 'description' => 'BPJS Kelas II'],
                    ['amount' => 160000, 'price' => 161000, 'description' => 'BPJS Kelas I'],
                ];
            }

            return response()->json([
                'success' => true,
                'data' => $denominations
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal memuat denominations: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Process transaction
     */
    public function processTransaction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_type' => 'required|string',
            'provider' => 'nullable|string',
            'target_number' => 'required|string',
            'amount' => 'required|numeric|min:1000',
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email|max:255',
            'customer_phone' => 'nullable|string|max:20'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $request->all();
            $data['description'] = $this->generateDescription($data);
            
            $result = $this->ppobService->processTransaction($data);

            if ($result['success']) {
                return response()->json([
                    'success' => true,
                    'message' => 'Transaksi berhasil dibuat',
                    'data' => $result['data']
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => $result['message']
            ], 400);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Check transaction status
     */
    public function checkStatus($orderId)
    {
        try {
            $result = $this->ppobService->checkTransactionStatus($orderId);

            if ($result['success']) {
                return response()->json([
                    'success' => true,
                    'data' => $result['data']
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => $result['message']
            ], 400);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Payment return handler
     */
    public function paymentReturn(Request $request)
    {
        $orderId = $request->input('order_id');
        $status = $request->input('status');

        if ($orderId) {
            // Check transaction status
            $result = $this->ppobService->checkTransactionStatus($orderId);
            
            if ($result['success']) {
                $transaction = $result['data']['transaction'];
                
                return view('frontend.ppob.payment-result', [
                    'transaction' => $transaction,
                    'status' => $status,
                    'message' => $this->getStatusMessage($status)
                ]);
            }
        }

        return view('frontend.ppob.payment-result', [
            'transaction' => null,
            'status' => 'failed',
            'message' => 'Transaksi tidak ditemukan'
        ]);
    }

    /**
     * Payment callback handler
     */
    public function paymentCallback(Request $request)
    {
        try {
            $data = $request->all();
            $signature = $request->header('X-Signature');
            
            // Verify signature
            $ottopayService = app(\App\Services\OttopayService::class);
            if (!$ottopayService->verifyCallback($data, $signature)) {
                return response()->json(['message' => 'Invalid signature'], 400);
            }

            $orderId = $data['order_id'] ?? null;
            $status = $data['status'] ?? 'failed';

            if ($orderId) {
                $transaction = PpobTransaction::where('order_id', $orderId)->first();
                
                if ($transaction) {
                    $transaction->update([
                        'status' => $status,
                        'payment_response' => $data,
                        'processed_at' => $status === 'success' ? now() : null
                    ]);

                    return response()->json(['message' => 'Callback processed successfully']);
                }
            }

            return response()->json(['message' => 'Transaction not found'], 404);

        } catch (\Exception $e) {
            Log::error('PPOB callback error: ' . $e->getMessage());
            return response()->json(['message' => 'Callback processing failed'], 500);
        }
    }

    /**
     * Show transaction history
     */
    public function history()
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $transactions = PpobTransaction::where('user_id', Auth::id())
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('frontend.ppob.history', compact('transactions'));
    }

    /**
     * Show transaction detail
     */
    public function detail($orderId)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $transaction = PpobTransaction::where('order_id', $orderId)
            ->where('user_id', Auth::id())
            ->firstOrFail();

        return view('frontend.ppob.detail', compact('transaction'));
    }

    /**
     * Generate transaction description
     */
    private function generateDescription($data)
    {
        $productType = $data['product_type'];
        $provider = $data['provider'] ?? '';
        $targetNumber = $data['target_number'];
        $amount = number_format($data['amount'], 0, ',', '.');

        switch ($productType) {
            case 'pulsa':
                return "Pulsa {$provider} Rp {$amount} untuk {$targetNumber}";
            case 'paket_data':
                return "Paket Data {$provider} Rp {$amount} untuk {$targetNumber}";
            case 'pln':
                return "Token PLN Rp {$amount} untuk {$targetNumber}";
            case 'pdam':
                return "Pembayaran PDAM Rp {$amount} untuk {$targetNumber}";
            case 'bpjs':
                return "Pembayaran BPJS Rp {$amount} untuk {$targetNumber}";
            case 'tv_kabel':
                return "Pembayaran TV Kabel {$provider} Rp {$amount} untuk {$targetNumber}";
            case 'internet':
                return "Pembayaran Internet {$provider} Rp {$amount} untuk {$targetNumber}";
            case 'finance':
                return "Pembayaran {$provider} Rp {$amount} untuk {$targetNumber}";
            default:
                return "Pembayaran PPOB Rp {$amount} untuk {$targetNumber}";
        }
    }

    /**
     * Get status message
     */
    private function getStatusMessage($status)
    {
        return match($status) {
            'pending' => 'Pembayaran sedang diproses',
            'paid' => 'Pembayaran berhasil',
            'success' => 'Transaksi berhasil',
            'failed' => 'Transaksi gagal',
            'cancelled' => 'Transaksi dibatalkan',
            default => 'Status tidak diketahui'
        };
    }
}
