<?php

namespace App\Http\Controllers;

use App\Models\Penduduk;
use App\Models\Keluarga;
use App\Models\AnggotaKeluarga;
use App\Models\PelayananSurat;
use App\Models\Pengaduan;
use App\Models\Bantuan;
use App\Models\Berita;
use App\Models\Setting;
use App\Models\ProdukUmkm;
use App\Models\Umkm;
use App\Models\TiketWisata;

use App\Models\PerangkatDesa;
use Illuminate\Support\Facades\Log;
use App\Models\ObjekWisata;
use App\Services\PaymentMethodSyncService;
use App\Services\FileUploadService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class PublicController extends Controller
{
    protected $paymentMethodSyncService;

    public function __construct(?PaymentMethodSyncService $paymentMethodSyncService = null)
    {
        $this->paymentMethodSyncService = $paymentMethodSyncService;
    }

    public function index()
    {
        // Ambil data statistik real-time dari database
        $stats = [
            'total_penduduk' => Penduduk::count(),
            'total_keluarga' => Penduduk::distinct('no_kk')->count('no_kk'),
            'total_surat' => PelayananSurat::count(),
            'total_pengaduan' => Pengaduan::count(),
        ];

        $berita_terbaru = Berita::where('status', 'Terbit')
                               ->orderBy('created_at', 'desc')
                               ->limit(3)
                               ->get();

        // Ambil data perangkat desa
        $perangkat_desa = PerangkatDesa::with('penduduk')
                                     ->where('status', 'Aktif')
                                     ->orderByRaw("
                                         CASE jabatan
                                             WHEN 'Kepala Desa' THEN 1
                                             WHEN 'Sekretaris Desa' THEN 2
                                             WHEN 'Kaur Tata Usaha dan Umum' THEN 3
                                             WHEN 'Kaur Keuangan' THEN 4
                                             WHEN 'Kaur Perencanaan' THEN 5
                                             WHEN 'Kasi Pemerintahan' THEN 6
                                             WHEN 'Kasi Kesejahteraan' THEN 7
                                             WHEN 'Kasi Pelayanan' THEN 8
                                             WHEN 'Kepala Dusun' THEN 9
                                             ELSE 99
                                         END, nama
                                     ")
                                     ->limit(6)
                                     ->get();

        // Path ke file HTML frontend
        $htmlPath = resource_path('views/frontend/index.php');
        
        // Load dan proses HTML dengan data backend lengkap
        if (file_exists($htmlPath)) {
            $html = file_get_contents($htmlPath);
            
            // 1. Replace data statistik dengan data real dari database
            $html = str_replace('1,234', number_format($stats['total_penduduk']), $html);
            $html = str_replace('456', number_format($stats['total_keluarga']), $html);
            

            
            // 3. Update judul halaman dengan nama desa dinamis
            $setting = Setting::first();
            if ($setting) {
                $html = str_replace('Sistem Informasi Desa', 'Portal ' . ($setting->nama_desa ?? 'Desa Digital'), $html);
                $html = str_replace('Desa Digital', $setting->nama_desa ?? 'Desa Digital', $html);
            }
            
            // 4. Integrasi berita terbaru dengan data real
            if ($berita_terbaru->count() > 0) {
                $berita_html = '';
                foreach ($berita_terbaru->take(2) as $berita) {
                    // Use the correct image field and make it clickable
                    $imageTag = $berita->gambar ? 
                        '<img src="/storage/' . $berita->gambar . '" alt="' . htmlspecialchars($berita->judul) . '" class="w-12 h-12 rounded-lg flex-shrink-0 object-cover">' :
                        '<div class="w-12 h-12 bg-gray-200 rounded-lg flex-shrink-0 flex items-center justify-center">
                            <svg class="w-6 h-6 text-gray-400" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M4 3a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V5a2 2 0 00-2-2H4zm12 12H4l4-8 3 6 2-4 3 6z"/>
                            </svg>
                         </div>';
                    
                    // Make the entire card clickable
                    $berita_html .= '
                <a href="/berita/' . $berita->slug . '" class="block">
                    <div class="mobile-card py-2 px-3 hover:bg-gray-50 transition-colors">
                        <div class="flex space-x-3">
                            ' . $imageTag . '
                            <div class="flex-1">
                                <h4 class="font-semibold text-gray-800 text-xs mb-1">' . htmlspecialchars($berita->judul) . '</h4>
                                <p class="text-gray-600 text-xs mb-1">' . Str::limit(strip_tags($berita->konten ?? $berita->isi), 60) . '</p>
                                <div class="flex justify-between items-center">
                                    <div class="text-xs text-gray-400">' . $berita->created_at->diffForHumans() . '</div>
                                    ' . ($berita->views > 0 ? '<div class="text-xs text-gray-400">' . number_format($berita->views) . ' views</div>' : '') . '
                                </div>
                            </div>
                        </div>
                    </div>
                </a>';
                }
                
                // Replace section berita statis dengan berita real dari database
                $html = preg_replace(
                    '/(<div class="space-y-3">).*?(<\/div>\s*<\/div>\s*<!-- Bottom Navigation)/s',
                    '$1' . $berita_html . '$2',
                    $html
                );
            }
            
            // 4. Pastikan navigasi service buttons berfungsi ke endpoint yang benar
            $html = str_replace("'/ajukan-surat'", "'/ajukan-surat'", $html);
            $html = str_replace("'/pengaduan'", "'/pengaduan'", $html);
            $html = str_replace("'/peta'", "'/peta'", $html);
            $html = str_replace("'/info-bantuan'", "'/info-bantuan'", $html);
            
            // Update tombol "Lihat Semua" berita untuk mengarah ke halaman berita
            $html = str_replace('<button class="text-blue-600 text-xs font-medium">Lihat Semua</button>', 
                               '<a href="/berita" class="text-blue-600 text-xs font-medium hover:text-blue-800 transition-colors">Lihat Semua</a>', $html);
            
            // 5. Perangkat Desa section removed as requested
            // Village officials cards have been removed from homepage

            // 6. Tambahkan meta data dinamis untuk SEO
            $description = $setting ? 'Portal Digital ' . $setting->nama_desa . ' - Akses layanan administrasi desa dengan mudah dan cepat' : 'Portal Digital Desa - Sistem Informasi Desa Modern';
            $html = str_replace('<meta name="viewport" content="width=device-width, initial-scale=1.0">', 
                '<meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="description" content="' . $description . '">
    <meta name="keywords" content="desa digital, pelayanan publik, surat online, pengaduan, ' . ($setting->nama_desa ?? 'desa') . '">
    <meta property="og:title" content="Portal ' . ($setting->nama_desa ?? 'Desa Digital') . '">
    <meta property="og:description" content="' . $description . '">
    <meta property="og:type" content="website">', $html);
            
            return response($html)->header('Content-Type', 'text/html');
        }
        
        // Fallback jika file HTML tidak ada
        return response()->json([
            'status' => 'success',
            'message' => 'Sistem Informasi Desa - Backend Active',
            'data' => $stats,
            'berita' => $berita_terbaru
        ]);
    }

    // Method untuk mengakses tampilan Bootstrap lama
    public function publicBootstrap()
    {
        $stats = [
            'total_penduduk' => Penduduk::count(),
            'total_keluarga' => Penduduk::distinct('no_kk')->count(),
            'total_surat' => PelayananSurat::count(),
            'total_pengaduan' => Pengaduan::count(),
        ];

        $berita_terbaru = Berita::where('status', 'Terbit')
                               ->orderBy('created_at', 'desc')
                               ->limit(4)
                               ->get();

        $pengumuman = Berita::where('status', 'Terbit')
                           ->where('kategori', 'Pengumuman')
                           ->orderBy('created_at', 'desc')
                           ->limit(3)
                           ->get();

        return view('public.index', compact('stats', 'berita_terbaru', 'pengumuman'));
    }

    public function tentang()
    {
        $setting = Setting::first();
        
        // Ambil data perangkat desa dari model PerangkatDesa
        $perangkat_desa = PerangkatDesa::with('penduduk')
                                     ->where('status', 'Aktif')
                                     ->orderByRaw("
                                         CASE jabatan
                                             WHEN 'Kepala Desa' THEN 1
                                             WHEN 'Sekretaris Desa' THEN 2
                                             WHEN 'Kaur Tata Usaha dan Umum' THEN 3
                                             WHEN 'Kaur Keuangan' THEN 4
                                             WHEN 'Kaur Perencanaan' THEN 5
                                             WHEN 'Kasi Pemerintahan' THEN 6
                                             WHEN 'Kasi Kesejahteraan' THEN 7
                                             WHEN 'Kasi Pelayanan' THEN 8
                                             WHEN 'Kepala Dusun' THEN 9
                                             ELSE 99
                                         END, nama
                                     ")
                                     ->get();
        
        return view('public.tentang', compact('setting', 'perangkat_desa'));
    }

    public function layanan()
    {
        $jenis_surat = ['Surat Domisili', 'Surat Keterangan Tidak Mampu', 'Surat Usaha', 'SKCK', 'Surat Keterangan Penghasilan'];
        return view('public.layanan', compact('jenis_surat'));
    }

    public function pengaduanForm()
    {
        $kategori_pengaduan = [
            'Infrastruktur',
            'Pelayanan Publik',
            'Kebersihan',
            'Keamanan',
            'Ekonomi',
            'Pendidikan',
            'Kesehatan',
            'Sosial',
            'Lainnya'
        ];
        
        return $this->renderServicePage('pengaduan', compact('kategori_pengaduan'));
    }

    public function storePengaduan(Request $request)
    {
        try {
            $validated = $request->validate([
                'nama' => 'required|string|max:255',
                'email' => 'nullable|email',
                'no_hp' => 'nullable|string|max:15',
                'kategori' => 'required|string',
                'judul' => 'required|string|max:255',
                'isi_pengaduan' => 'required|string',
                'lampiran' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120'
            ]);
            
            // Upload lampiran jika ada
            if ($request->hasFile('lampiran')) {
                $validated['lampiran'] = $request->file('lampiran')
                                                ->store('pengaduan', 'public');
            }
            
            $validated['nomor_tiket'] = 'TKT-' . date('Ymd') . '-' . str_pad(Pengaduan::whereDate('created_at', today())->count() + 1, 3, '0', STR_PAD_LEFT);
            $validated['status'] = 'Baru';
            $validated['tanggal_pengaduan'] = now();
            
            $pengaduan = Pengaduan::create($validated);
            
            // Return JSON response for AJAX
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Pengaduan berhasil dikirim!',
                    'data' => [
                        'nomor_tiket' => $pengaduan->nomor_tiket,
                        'judul' => $pengaduan->judul,
                        'status' => $pengaduan->status
                    ]
                ]);
            }
            
            return redirect()->route('public.pengaduan.success')
                            ->with('success', 'Pengaduan berhasil dikirim! Nomor tiket: ' . $pengaduan->nomor_tiket);
                            
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data tidak valid',
                    'errors' => $e->errors()
                ], 422);
            }
            
            return back()->withErrors($e->errors())->withInput();
            
        } catch (\Exception $e) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->withErrors(['error' => 'Terjadi kesalahan sistem'])->withInput();
        }
    }

    public function pengaduanSuccess()
    {
        return view('public.pengaduan-success');
    }

    public function cekLayanan()
    {
        return $this->renderServicePage('cek-layanan');
    }
    
    public function checkStatus(Request $request)
    {
        try {
            $validated = $request->validate([
                'tipe' => 'required|in:surat,pengaduan',
                'nomor_antrian' => 'required|string'
            ]);
            
            $data = null;
            
            if ($validated['tipe'] === 'surat') {
                $data = PelayananSurat::where('nomor_antrian', $validated['nomor_antrian'])->first();
            } else {
                $data = Pengaduan::where('nomor_tiket', $validated['nomor_antrian'])->first();
            }
            
            if ($data) {
                return response()->json([
                    'success' => true,
                    'data' => $data
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Data tidak ditemukan'
                ]);
            }
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $e->errors()
            ], 422);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan sistem'
            ], 500);
        }
    }

    public function ajukanSurat()
    {
        $jenis_surat = [
            'Surat Domisili',
            'Surat Keterangan Tidak Mampu', 
            'Surat Usaha',
            'SKCK',
            'Surat Kematian',
            'Surat Keterangan Belum Menikah',
            'Surat Pengantar Nikah',
            'Surat Keterangan Penghasilan',
            'Lainnya'
        ];
        
        return $this->renderServicePage('ajukan-surat', compact('jenis_surat'));
    }
    
    private function renderServicePage($pageName, $data = [])
    {
        // Path ke file HTML untuk service page
        $htmlPath = resource_path("views/frontend/{$pageName}.php");
        
        if (file_exists($htmlPath)) {
            $html = file_get_contents($htmlPath);
            
            // Inject CSRF token
            $html = str_replace('<meta name="viewport"', 
                '<meta name="csrf-token" content="' . csrf_token() . '">
    <meta name="viewport"', $html);
            
            // Update title dengan nama desa jika ada setting
            $setting = Setting::first();
            if ($setting) {
                $html = str_replace('Sistem Informasi Desa', 'Portal ' . ($setting->nama_desa ?? 'Desa Digital'), $html);
            }

            // Handle dynamic data replacement based on page type
            if ($pageName === 'berita' && isset($data['statistik_berita'])) {
                // Replace statistics
                $stats = $data['statistik_berita'];
                $html = str_replace('{{total_berita}}', number_format($stats['total_berita']), $html);
                $html = str_replace('{{berita_bulan_ini}}', number_format($stats['berita_bulan_ini']), $html);
                $html = str_replace('{{total_views}}', 
                    $stats['total_views'] >= 1000 ? number_format($stats['total_views']/1000, 1) . 'K' : $stats['total_views'], $html);

                // Generate featured news content
                $featuredContent = '';
                if (isset($data['berita_unggulan']) && $data['berita_unggulan']) {
                    $berita = $data['berita_unggulan'];
                    $imageTag = $berita->featured_image ? 
                        '<img src="/storage/' . $berita->featured_image . '" alt="' . htmlspecialchars($berita->judul) . '" class="w-full h-48 object-cover rounded-lg">' :
                        '<img src="https://via.placeholder.com/400x200/3B82F6/FFFFFF?text=Berita+Utama" alt="Berita Utama" class="w-full h-48 object-cover rounded-lg">';
                    
                    $featuredContent = '
                    <div class="mobile-card article-card transition-all duration-300 cursor-pointer">
                        <div class="relative mb-4">
                            ' . $imageTag . '
                            <div class="absolute top-3 left-3">
                                <span class="category-badge category-' . strtolower($berita->kategori) . '">' . $berita->kategori . '</span>
                            </div>
                            <div class="absolute bottom-3 right-3 bg-black/50 text-white px-2 py-1 rounded text-xs">
                                ' . $berita->created_at->diffForHumans() . '
                            </div>
                        </div>
                        <h2 class="text-lg font-bold text-gray-800 mb-2 line-clamp-2">
                            ' . htmlspecialchars($berita->judul) . '
                        </h2>
                        <p class="text-gray-600 text-sm mb-3 line-clamp-3">
                            ' . Str::limit(strip_tags($berita->isi), 120) . '
                        </p>
                        <div class="flex items-center justify-between text-xs text-gray-500">
                            <div class="flex items-center space-x-3">
                                <span class="flex items-center space-x-1">
                                    <svg class="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
                                        <path d="M10 12a2 2 0 100-4 2 2 0 000 4z"/>
                                        <path fill-rule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z"/>
                                    </svg>
                                    <span>' . $berita->views . '</span>
                                </span>
                            </div>
                            <span>Admin Desa</span>
                        </div>
                    </div>';
                }
                $html = str_replace('{{featured_news_content}}', $featuredContent, $html);

                // Generate news list content
                $newsListContent = '';
                if (isset($data['berita_terbaru'])) {
                    foreach ($data['berita_terbaru'] as $berita) {
                        $imageTag = $berita->featured_image ? 
                            '<img src="/storage/' . $berita->featured_image . '" alt="' . htmlspecialchars($berita->judul) . '" class="w-20 h-16 object-cover rounded-lg flex-shrink-0">' :
                            '<img src="https://via.placeholder.com/100x80/10B981/FFFFFF?text=News" alt="Berita" class="w-20 h-16 object-cover rounded-lg flex-shrink-0">';
                        
                        $categoryClass = match(strtolower($berita->kategori)) {
                            'berita' => 'category-berita',
                            'pengumuman' => 'category-pengumuman',
                            'kegiatan' => 'category-kegiatan',
                            default => 'category-berita'
                        };

                        $newsListContent .= '
                        <div class="mobile-card article-card transition-all duration-300 cursor-pointer news-item" data-category="' . strtolower($berita->kategori) . '">
                            <div class="flex space-x-4">
                                ' . $imageTag . '
                                <div class="flex-1">
                                    <div class="flex items-start justify-between mb-2">
                                        <span class="category-badge ' . $categoryClass . '">' . $berita->kategori . '</span>
                                        <span class="text-xs text-gray-400">' . $berita->created_at->diffForHumans() . '</span>
                                    </div>
                                    <h4 class="font-semibold text-gray-800 text-sm mb-1 line-clamp-2">
                                        ' . htmlspecialchars($berita->judul) . '
                                    </h4>
                                    <p class="text-gray-600 text-xs mb-2 line-clamp-2">
                                        ' . Str::limit(strip_tags($berita->isi), 80) . '
                                    </p>
                                    <div class="flex items-center justify-between text-xs text-gray-500">
                                        <div class="flex items-center space-x-2">
                                            <span>👁 ' . $berita->views . '</span>
                                        </div>
                                        <span>Admin Desa</span>
                                    </div>
                                </div>
                            </div>
                        </div>';
                    }
                }
                $html = str_replace('{{news_list_content}}', $newsListContent, $html);
            }

            // Handle info-bantuan dynamic content
            if ($pageName === 'info-bantuan' && isset($data['statistik_bantuan'])) {
                $stats = $data['statistik_bantuan'];
                $html = str_replace('{{total_penerima}}', number_format($stats['total_penerima']), $html);
                $html = str_replace('{{total_program}}', number_format($stats['total_program']), $html);
                $html = str_replace('{{bantuan_aktif}}', number_format($stats['bantuan_aktif']), $html);
                $html = str_replace('{{nilai_total}}', 'Rp ' . number_format($stats['nilai_total'], 0, ',', '.'), $html);
            }

            // Handle peta dynamic content
            if ($pageName === 'peta' && isset($data['stats'])) {
                $stats = $data['stats'];
                $html = str_replace('{{total_fasilitas}}', number_format($stats['total_fasilitas']), $html);
                $html = str_replace('{{total_umkm}}', number_format($stats['total_umkm']), $html);
                $html = str_replace('{{total_wisata}}', number_format($stats['total_wisata']), $html);
                $html = str_replace('{{total_lokasi}}', number_format(count($data['umkm_locations'] ?? [])), $html);
            }
            
            return response($html)->header('Content-Type', 'text/html');
        }
        
        // Fallback ke blade view jika HTML file tidak ada
        return view("public.{$pageName}", $data);
    }

    public function checkNik($nik)
    {
        $penduduk = Penduduk::where('nik', $nik)->first();
        
        if ($penduduk) {
            return response()->json([
                'exists' => true,
                'nama' => $penduduk->nama,
                'alamat' => $penduduk->alamat
            ]);
        }
        
        return response()->json([
            'exists' => false,
            'message' => 'NIK tidak ditemukan di database'
        ]);
    }

    public function storeSurat(Request $request)
    {
        try {
            $validated = $request->validate([
                'nik' => 'required|string|size:16|exists:penduduk,nik',
                'nama' => 'required|string|max:255', // Only for validation, not stored
                'jenis_surat' => 'required|string|in:Surat Domisili,SKCK,Surat Kematian,Surat Usaha,Surat Keterangan Tidak Mampu,Surat Keterangan Belum Menikah,Surat Pengantar Nikah,Surat Keterangan Penghasilan,Lainnya',
                'keperluan' => 'required|string',
                'berkas_pendukung' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048'
            ]);
            
            // Remove 'nama' from validated data as it's not in the table
            unset($validated['nama']);
            
            // Upload berkas jika ada
            if ($request->hasFile('berkas_pendukung')) {
                $validated['berkas_pendukung'] = $request->file('berkas_pendukung')
                                                       ->store('berkas-surat', 'public');
            }
            
            $validated['nomor_antrian'] = 'ANT-' . date('Ymd') . '-' . str_pad(PelayananSurat::whereDate('created_at', today())->count() + 1, 3, '0', STR_PAD_LEFT);
            $validated['status'] = 'Menunggu';
            $validated['tanggal_pengajuan'] = now();
            
            $surat = PelayananSurat::create($validated);
            
            // Return JSON response for AJAX
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Pengajuan surat berhasil dikirim!',
                    'data' => [
                        'nomor_antrian' => $surat->nomor_antrian,
                        'jenis_surat' => $surat->jenis_surat,
                        'status' => $surat->status
                    ]
                ]);
            }
            
            return redirect()->route('public.surat.success')
                            ->with('success', 'Pengajuan surat berhasil! Nomor antrian: ' . $surat->nomor_antrian);
                            
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data tidak valid',
                    'errors' => $e->errors()
                ], 422);
            }
            
            return back()->withErrors($e->errors())->withInput();
            
        } catch (\Exception $e) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->withErrors(['error' => 'Terjadi kesalahan sistem'])->withInput();
        }
    }

    public function suratSuccess()
    {
        return view('public.surat-success');
    }

    public function infoBantuan()
    {
        // Ambil program bantuan yang unik tanpa GROUP BY yang bermasalah
        $program_bantuan = Bantuan::select('jenis_bantuan', 'deskripsi', 'periode', 'nominal')
                                 ->distinct()
                                 ->limit(10)
                                 ->get();
        
        $statistik_bantuan = [
            'total_penerima' => Bantuan::distinct('nik')->count(),
            'total_program' => Bantuan::distinct('jenis_bantuan')->count(),
            'bantuan_aktif' => Bantuan::where('status', 'Diterima')->count(),
            'nilai_total' => Bantuan::where('status', 'Diterima')->sum('nominal')
        ];
        
        return $this->renderServicePage('info-bantuan', compact('program_bantuan', 'statistik_bantuan'));
    }

    public function kontak()
    {
        return view('public.kontak');
    }

    public function produkUmkm(Request $request)
    {
        // Cache key based on request parameters
        $cacheKey = 'umkm_page_' . md5(json_encode($request->all()));
        
        // Check cache first
        $cachedData = cache()->get($cacheKey);
        if ($cachedData && !$request->filled('search')) {
            return $this->renderProductPage($cachedData);
        }
        
        // Optimized query with proper indexes and selected columns only
        $query = ProdukUmkm::select([
            'id', 'umkm_id', 'nama_produk', 'slug', 'deskripsi', 'kategori_produk',
            'foto_utama', 'harga_jual', 'harga_diskon', 'rating_rata_rata', 
            'jumlah_review', 'total_terjual', 'is_featured', 'created_at'
        ])->with(['umkm:id,nama_umkm,slug'])
          ->aktif()
          ->available();
        
        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_produk', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhereHas('umkm', function($qq) use ($search) {
                      $qq->where('nama_umkm', 'like', "%{$search}%");
                  });
            });
        }
        
        // Filter by category
        if ($request->filled('kategori')) {
            $query->where('kategori_produk', $request->kategori);
        }
        
        // Optimized sorting with proper indexes
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        
        if (in_array($sortBy, ['nama_produk', 'harga_jual', 'rating_rata_rata', 'total_terjual', 'created_at'])) {
            $query->orderBy($sortBy, $sortOrder);
        }
        
        $produk = $query->paginate(12)->withQueryString();
        
        // Optimized featured products query with selected columns
        $produkUnggulan = ProdukUmkm::select([
            'id', 'umkm_id', 'nama_produk', 'slug', 'foto_utama', 
            'harga_jual', 'harga_diskon', 'rating_rata_rata', 'jumlah_review'
        ])->with(['umkm:id,nama_umkm'])
          ->aktif()
          ->available()
          ->featured()
          ->limit(8)
          ->get();
        
        // Cache categories query result
        $kategoris = cache()->remember('umkm_kategoris', 3600, function() {
            return ProdukUmkm::select('kategori_produk')
                              ->distinct()
                              ->whereNotNull('kategori_produk')
                              ->pluck('kategori_produk');
        });
        
        // Cache UMKM list
        $umkmList = cache()->remember('umkm_list', 1800, function() {
            return Umkm::select(['id', 'nama_umkm', 'slug', 'logo'])
                        ->aktif()
                        ->verified()
                        ->has('produk')
                        ->get();
        });
        
        // Cache statistics with longer TTL
        $stats = cache()->remember('umkm_stats', 1800, function() {
            return [
                'total_produk' => ProdukUmkm::aktif()->available()->count(),
                'total_umkm' => Umkm::aktif()->verified()->has('produk')->count(),
                'total_kategori' => ProdukUmkm::distinct('kategori_produk')->count('kategori_produk'),
            ];
        });
        
        $data = compact('produk', 'produkUnggulan', 'kategoris', 'umkmList', 'stats');
        
        // Cache the result for 5 minutes (except for search results)
        if (!$request->filled('search')) {
            cache()->put($cacheKey, $data, 300);
        }
        
        return $this->renderProductPage($data);
    }

    public function detailProduk($slug)
    {
        $produk = ProdukUmkm::with(['umkm'])
                            ->where('slug', $slug)
                            ->aktif()
                            ->firstOrFail();
        
        // Increment view count
        $produk->increment('view_count');
        
        // Related products
        $produkTerkait = ProdukUmkm::with(['umkm'])
                                  ->where('kategori_produk', $produk->kategori_produk)
                                  ->where('id', '!=', $produk->id)
                                  ->aktif()
                                  ->available()
                                  ->limit(4)
                                  ->get();
        
        return $this->renderProductDetailPage(compact('produk', 'produkTerkait'));
    }

    private function renderProductPage($data = [])
    {
        // Path ke file HTML frontend untuk produk
        $htmlPath = resource_path('views/frontend/produk-umkm.php');
        
        if (file_exists($htmlPath)) {
            $html = file_get_contents($htmlPath);
            
            // Extract data
            $stats = $data['stats'] ?? [];
            $produk = $data['produk'] ?? collect();
            $produkUnggulan = $data['produkUnggulan'] ?? collect();
            $kategoris = $data['kategoris'] ?? collect();
            
            // 1. Update statistics with placeholders
            $html = str_replace('{{total_produk}}', number_format($stats['total_produk'] ?? 0), $html);
            $html = str_replace('{{total_umkm}}', number_format($stats['total_umkm'] ?? 0), $html);
            $html = str_replace('{{total_kategori}}', number_format($stats['total_kategori'] ?? 0), $html);
            
            // 2. Generate featured products content
            $featuredContent = '';
            if ($produkUnggulan->count() > 0) {
                foreach ($produkUnggulan->take(4) as $item) {
                    $discountBadge = '';
                    $originalPrice = '';
                    
                    if ($item->harga_diskon && $item->harga_diskon < $item->harga_jual) {
                        $discountPercent = round((($item->harga_jual - $item->harga_diskon) / $item->harga_jual) * 100);
                        $discountBadge = '<div class="absolute top-2 left-2 bg-red-500 text-white text-xs px-2 py-1 rounded-full">-' . $discountPercent . '%</div>';
                        $originalPrice = '<span class="text-xs text-gray-400 line-through ml-1">Rp ' . number_format($item->harga_jual, 0, ',', '.') . '</span>';
                    }
                    
                    $harga = $item->harga_diskon ?? $item->harga_jual;
                    $imageUrl = $item->foto_utama ? '/storage/produk/' . $item->foto_utama : 'https://via.placeholder.com/160x120/f1f5f9/64748b?text=' . urlencode($item->nama_produk);
                    
                    $featuredContent .= '
                    <div class="product-card flex-shrink-0 w-40">
                        <div class="relative">
                            <img src="' . $imageUrl . '" alt="' . htmlspecialchars($item->nama_produk) . '" class="w-full h-24 object-cover">
                            ' . $discountBadge . '
                        </div>
                        <div class="p-3">
                            <h4 class="font-semibold text-sm text-gray-800 mb-1">' . htmlspecialchars($item->nama_produk) . '</h4>
                            <p class="text-xs text-gray-500 mb-2">' . htmlspecialchars($item->umkm->nama_umkm ?? 'UMKM') . '</p>
                            <div class="mb-2">
                                <span class="text-sm font-bold text-green-600">Rp ' . number_format($harga, 0, ',', '.') . '</span>
                                ' . $originalPrice . '
                            </div>
                            <div class="flex space-x-1">
                                <button onclick="quickCheckout(' . $item->id . ', \'' . addslashes($item->nama_produk) . '\', ' . $harga . ')" class="flex-1 bg-green-500 text-white px-1 py-1 rounded text-xs">Beli</button>
                                <button onclick="beliProduk(\'' . $item->slug . '\')" class="w-6 h-6 bg-blue-500 text-white rounded flex items-center justify-center">
                                    <i class="fas fa-eye text-xs"></i>
                                </button>
                            </div>
                        </div>
                    </div>';
                }
            }
            $html = str_replace('{{featured_products_content}}', $featuredContent, $html);
            
            // 3. Generate products grid content
            $productsContent = '';
            if ($produk->count() > 0) {
                foreach ($produk as $item) {
                    $harga = $item->harga_diskon ?? $item->harga_jual;
                    $imageUrl = $item->foto_utama ? '/storage/produk/' . $item->foto_utama : 'https://via.placeholder.com/160x120/f1f5f9/64748b?text=' . urlencode($item->nama_produk);
                    $stars = str_repeat('★', floor($item->rating_rata_rata ?? 0)) . str_repeat('☆', 5 - floor($item->rating_rata_rata ?? 0));
                    
                    $productsContent .= '
                    <div class="product-card">
                        <div class="relative">
                            <img src="' . $imageUrl . '" alt="' . htmlspecialchars($item->nama_produk) . '" class="w-full h-32 object-cover">
                            <button class="absolute top-2 right-2 w-8 h-8 bg-white/80 rounded-full flex items-center justify-center">
                                <i class="fas fa-heart text-gray-600"></i>
                            </button>
                        </div>
                        <div class="p-3">
                            <h4 class="font-semibold text-sm text-gray-800 mb-1">' . htmlspecialchars($item->nama_produk) . '</h4>
                            <p class="text-xs text-gray-500 mb-2">' . htmlspecialchars($item->umkm->nama_umkm ?? 'UMKM') . '</p>
                            <div class="flex items-center mb-2">
                                <div class="flex text-yellow-400 text-xs">
                                    <span>' . $stars . '</span>
                                </div>
                                <span class="text-xs text-gray-500 ml-1">(' . ($item->jumlah_review ?? 0) . ')</span>
                            </div>
                            <div class="flex items-center justify-between mb-2">
                                <span class="text-sm font-bold text-green-600">Rp ' . number_format($harga, 0, ',', '.') . '</span>
                            </div>
                            <div class="flex space-x-1">
                                <button onclick="quickCheckout(' . $item->id . ', \'' . addslashes($item->nama_produk) . '\', ' . $harga . ')" class="flex-1 bg-green-500 text-white px-2 py-1 rounded text-xs font-medium">Beli Cepat</button>
                                <button onclick="beliProduk(\'' . $item->slug . '\')" class="flex-1 bg-blue-500 text-white px-2 py-1 rounded text-xs font-medium">Detail</button>
                            </div>
                        </div>
                    </div>';
                }
            }
            $html = str_replace('{{products_grid_content}}', $productsContent, $html);
            
            // 4. Generate filter categories
            $categoriesContent = '<button onclick="filterByCategory(\'semua\')" class="px-3 py-1 bg-blue-100 text-blue-700 rounded-full text-sm font-medium">Semua</button>';
            if ($kategoris->count() > 0) {
                foreach ($kategoris as $kategori) {
                    $categoriesContent .= '<button onclick="filterByCategory(\'' . $kategori . '\')" class="px-3 py-1 bg-gray-100 text-gray-700 rounded-full text-sm">' . htmlspecialchars($kategori) . '</button>';
                }
            }
            $html = str_replace('{{filter_categories_buttons}}', $categoriesContent, $html);
            
            // 5. Generate pagination
            $paginationContent = '';
            if ($produk->hasPages()) {
                if ($produk->hasMorePages()) {
                    $paginationContent = '<button onclick="loadPage(' . ($produk->currentPage() + 1) . ')" class="bg-blue-500 text-white px-6 py-3 rounded-xl font-medium">Lihat Produk Lainnya</button>';
                } else {
                    $paginationContent = '<button onclick="loadPage(1)" class="bg-gray-100 text-gray-700 px-6 py-3 rounded-xl font-medium">Kembali ke Halaman Pertama</button>';
                }
            }
            $html = str_replace('{{pagination_content}}', $paginationContent, $html);
            
            return response($html)->header('Content-Type', 'text/html');
        }
        
        // Fallback ke JSON
        return response()->json([
            'status' => 'success',
            'message' => 'Produk UMKM',
            'data' => $data
        ]);
    }

    private function renderProductDetailPage($data = [])
    {
        // Path ke file HTML frontend untuk detail produk
        $htmlPath = resource_path('views/frontend/detail-produk.php');
        
        if (file_exists($htmlPath)) {
            $html = file_get_contents($htmlPath);
            
            // Process with data if needed
            return response($html)->header('Content-Type', 'text/html');
        }
        
        // Fallback ke JSON
        return response()->json([
            'status' => 'success',
            'message' => 'Detail Produk',
            'data' => $data
        ]);
    }

    public function wisata(Request $request)
    {
        // Cache key based on request parameters
        $cacheKey = 'wisata_page_' . md5(json_encode($request->all()));
        
        // Check cache first
        $cachedData = cache()->get($cacheKey);
        if ($cachedData && !$request->filled('search')) {
            return $this->renderTourismPage($cachedData);
        }
        
        // Optimized query with proper indexes and selected columns only
        $query = ObjekWisata::select([
            'id', 'nama_wisata', 'slug', 'deskripsi', 'kategori_wisata',
            'alamat', 'foto_utama', 'rating_rata_rata', 'jumlah_review',
            'total_pengunjung', 'harga_tiket_dewasa', 'is_featured',
            'created_at'
        ])->where('status', 'aktif');
        
        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_wisata', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhere('alamat', 'like', "%{$search}%");
            });
        }
        
        // Filter by category
        if ($request->filled('kategori')) {
            $query->where('kategori_wisata', $request->kategori);
        }

        // Filter by featured
        if ($request->filled('featured') && $request->featured == '1') {
            $query->where('is_featured', true);
        }

        // Optimized sorting with proper indexes
        $sortBy = $request->get('sort', 'created_at');
        $sortOrder = $request->get('order', 'desc');
        
        if (in_array($sortBy, ['nama_wisata', 'rating_rata_rata', 'total_pengunjung', 'harga_tiket_dewasa', 'created_at'])) {
            $query->orderBy($sortBy, $sortOrder);
        }
        
        $destinasi = $query->paginate(6)->withQueryString();
        
        // Optimized featured destinations query with selected columns
        $destinasiFeatured = ObjekWisata::select([
            'id', 'nama_wisata', 'slug', 'deskripsi', 'foto_utama',
            'rating_rata_rata', 'jumlah_review', 'total_pengunjung',
            'harga_tiket_dewasa', 'alamat'
        ])->where('status', 'aktif')
          ->where('is_featured', true)
          ->orderBy('rating_rata_rata', 'desc')
          ->limit(8)
          ->get();
        
        // Cache categories query result
        $kategoris = cache()->remember('wisata_kategoris', 3600, function() {
            return ObjekWisata::where('status', 'aktif')
                              ->select('kategori_wisata')
                              ->distinct()
                              ->pluck('kategori_wisata');
        });
        
        // Cache statistics with longer TTL
        $stats = cache()->remember('wisata_stats', 1800, function() {
            return [
                'total_destinasi' => ObjekWisata::where('status', 'aktif')->count(),
                'total_kategori' => ObjekWisata::where('status', 'aktif')->distinct('kategori_wisata')->count(),
                'total_pengunjung' => ObjekWisata::where('status', 'aktif')->sum('total_pengunjung'),
                'rata_rating' => ObjekWisata::where('status', 'aktif')->avg('rating_rata_rata')
            ];
        });

        // Check if showing featured only
        $showingFeatured = $request->filled('featured') && $request->featured == '1';
        
        $data = compact('destinasi', 'destinasiFeatured', 'kategoris', 'stats', 'showingFeatured');
        
        // Cache the result for 5 minutes (except for search results)
        if (!$request->filled('search')) {
            cache()->put($cacheKey, $data, 300);
        }
        
        return $this->renderTourismPage($data);
    }

    public function detailWisata($slug)
    {
        // Ambil destinasi wisata dari database berdasarkan slug
        $destinasi = ObjekWisata::where('slug', $slug)
                                  ->where('status', 'aktif')
                                  ->firstOrFail();
        
        // Increment view count
        $destinasi->increment('total_pengunjung');
        
        // Destinasi terkait (kategori sama)
        $destinasiTerkait = ObjekWisata::where('kategori_wisata', $destinasi->kategori_wisata)
                                         ->where('id', '!=', $destinasi->id)
                                         ->where('status', 'aktif')
                                         ->limit(4)
                                         ->get();

        return $this->renderTourismDetailPage(compact('destinasi', 'destinasiTerkait'));
    }

    private function renderTourismPage($data = [])
    {
        // Path ke file HTML frontend untuk wisata
        $htmlPath = resource_path('views/frontend/wisata.php');
        
        if (file_exists($htmlPath)) {
            $html = file_get_contents($htmlPath);
            
            // Extract data
            $stats = $data['stats'] ?? [];
            $destinasi = $data['destinasi'] ?? collect();
            $destinasiFeatured = $data['destinasiFeatured'] ?? collect();
            
            // 1. Update statistics with real data
            $html = str_replace('{{total_destinasi}}', number_format($stats['total_destinasi'] ?? 0), $html);
            $html = str_replace('{{total_kategori}}', number_format($stats['total_kategori'] ?? 0), $html);
            $html = str_replace('{{total_pengunjung}}', 
                $stats['total_pengunjung'] >= 1000 ? 
                number_format($stats['total_pengunjung']/1000, 1) . 'K' : 
                number_format($stats['total_pengunjung'] ?? 0), $html);
            
            // 2. Generate featured destinations content (horizontal scroll format)
            $featuredContent = '';
            if ($destinasiFeatured->count() > 0) {
                foreach ($destinasiFeatured->take(4) as $item) {
                    $featuredContent .= '
                    <div class="destination-card" style="width: 280px;">
                        <div class="relative">
                            <img src="' . $item->foto_utama_url . '" alt="' . htmlspecialchars($item->nama_wisata) . '" class="w-full h-40 object-cover">
                            <div class="absolute top-3 left-3">
                                <span class="bg-green-500 text-white px-2 py-1 rounded-full text-xs font-medium">
                                    <i class="fas fa-star mr-1"></i>Unggulan
                                </span>
                            </div>
                            <div class="absolute top-3 right-3">
                                <button class="p-2 rounded-full bg-white/80 hover:bg-white transition-colors">
                                    <i class="far fa-heart text-red-500"></i>
                                </button>
                            </div>
                        </div>
                        <div class="p-4">
                            <h3 class="font-bold text-gray-800 mb-1">' . htmlspecialchars($item->nama_wisata) . '</h3>
                            <p class="text-sm text-gray-600 mb-2">' . htmlspecialchars(Str::limit($item->deskripsi, 60)) . '</p>
                            <div class="flex items-center justify-between mb-3">
                                <div class="flex items-center">
                                    <div class="rating-stars mr-2">';
                    
                    // Add star rating
                    for ($i = 1; $i <= 5; $i++) {
                        if ($i <= floor($item->rating_rata_rata)) {
                            $featuredContent .= '<i class="fas fa-star"></i>';
                        } elseif ($i == ceil($item->rating_rata_rata) && $item->rating_rata_rata - floor($item->rating_rata_rata) >= 0.5) {
                            $featuredContent .= '<i class="fas fa-star-half-alt"></i>';
                        } else {
                            $featuredContent .= '<i class="far fa-star"></i>';
                        }
                    }
                    
                    $featuredContent .= '
                                    </div>
                                    <span class="text-sm text-gray-600">' . number_format($item->rating_rata_rata, 1) . ' (' . $item->jumlah_review . ' review)</span>
                                </div>
                            </div>
                            <div class="flex items-center text-sm text-gray-500 mb-3">
                                <i class="fas fa-map-marker-alt mr-1"></i>
                                <span>' . htmlspecialchars(Str::limit($item->alamat, 30)) . '</span>
                            </div>
                            <div class="flex items-center justify-between mb-3">
                                <div class="text-sm font-bold text-green-600">' . $item->harga_tiket_formatted . '</div>
                                <div class="text-xs text-gray-500">per orang</div>
                            </div>
                            <div class="flex space-x-2">
                                <button onclick="quickBooking(\'' . $item->slug . '\', \'' . addslashes($item->nama_wisata) . '\', ' . $item->harga_tiket_dewasa . ')" class="flex-1 bg-orange-500 text-white py-2 rounded-lg font-medium">
                                    <i class="fas fa-ticket-alt mr-1"></i>Booking
                                </button>
                                <button onclick="window.location.href=\'/wisata/' . $item->slug . '\'" class="flex-1 bg-blue-600 text-white py-2 rounded-lg font-medium">
                                    Detail
                                </button>
                            </div>
                        </div>
                    </div>';
                }
            }
            $html = str_replace('{{featured_destinations_content}}', $featuredContent, $html);

            // 3. Generate all destinations grid (vertical list format)
            $destinationsContent = '';
            if ($destinasi->count() > 0) {
                foreach ($destinasi as $item) {
                    $destinationsContent .= '
                    <div class="destination-card destination-item" data-category="' . strtolower($item->kategori_wisata) . '">
                        <div class="flex">
                            <img src="' . $item->foto_utama_url . '" alt="' . htmlspecialchars($item->nama_wisata) . '" class="w-24 h-24 object-cover rounded-l-lg flex-shrink-0">
                            <div class="flex-1 p-4">
                                <div class="flex items-start justify-between mb-2">
                                    <h3 class="font-bold text-gray-800 text-sm">' . htmlspecialchars($item->nama_wisata) . '</h3>
                                    <button class="p-1 rounded-full">
                                        <i class="far fa-heart text-red-400 text-sm"></i>
                                    </button>
                                </div>
                                <p class="text-xs text-gray-600 mb-2">' . htmlspecialchars(Str::limit($item->deskripsi, 80)) . '</p>
                                <div class="flex items-center mb-2">
                                    <div class="rating-stars text-xs mr-1">';
                    
                    // Add star rating for grid items
                    for ($i = 1; $i <= 5; $i++) {
                        if ($i <= floor($item->rating_rata_rata)) {
                            $destinationsContent .= '<i class="fas fa-star"></i>';
                        } elseif ($i == ceil($item->rating_rata_rata) && $item->rating_rata_rata - floor($item->rating_rata_rata) >= 0.5) {
                            $destinationsContent .= '<i class="fas fa-star-half-alt"></i>';
                        } else {
                            $destinationsContent .= '<i class="far fa-star"></i>';
                        }
                    }
                    
                    $destinationsContent .= '
                                    </div>
                                    <span class="text-xs text-gray-500">' . number_format($item->rating_rata_rata, 1) . ' (' . $item->jumlah_review . ' review)</span>
                                </div>
                                <div class="flex items-center justify-between mb-2">
                                    <div class="flex items-center text-xs text-gray-500">
                                        <i class="fas fa-map-marker-alt mr-1"></i>
                                        <span>' . $item->jarak . '</span>
                                    </div>
                                    <div class="text-xs font-bold text-green-600">' . $item->harga_tiket_formatted . '</div>
                                </div>
                                <div class="flex space-x-1">
                                    <button onclick="quickBooking(\'' . $item->slug . '\', \'' . addslashes($item->nama_wisata) . '\', ' . $item->harga_tiket_dewasa . ')" class="flex-1 bg-orange-500 text-white py-1 rounded text-xs font-medium">Booking</button>
                                    <button onclick="window.location.href=\'/wisata/' . $item->slug . '\'" class="flex-1 bg-blue-600 text-white py-1 rounded text-xs font-medium">Detail</button>
                                </div>
                            </div>
                        </div>
                    </div>';
                }
            }
            $html = str_replace('{{destinations_grid_content}}', $destinationsContent, $html);
            
            // 4. Generate simple pagination (only Next button)
            $paginationContent = '';
            if ($destinasi->hasMorePages()) {
                $paginationContent = '
                <div class="text-center mt-6 mb-8">
                    <button onclick="loadPage(' . ($destinasi->currentPage() + 1) . ')" class="bg-blue-600 text-white px-6 py-3 rounded-xl font-medium hover:bg-blue-700 transition-colors">
                        <i class="fas fa-chevron-down mr-2"></i>Lihat Destinasi Lainnya
                    </button>
                    <div class="text-sm text-gray-500 mt-2">
                        Halaman ' . $destinasi->currentPage() . ' dari ' . $destinasi->lastPage() . ' • Menampilkan ' . $destinasi->firstItem() . '-' . $destinasi->lastItem() . ' dari ' . $destinasi->total() . ' destinasi
                    </div>
                </div>';
            } else if ($destinasi->currentPage() > 1) {
                $paginationContent = '
                <div class="text-center mt-6 mb-8">
                    <div class="text-sm text-gray-500">
                        Semua destinasi telah ditampilkan (' . $destinasi->total() . ' destinasi)
                    </div>
                    <button onclick="loadPage(1)" class="mt-2 bg-gray-600 text-white px-6 py-2 rounded-lg text-sm hover:bg-gray-700 transition-colors">
                        <i class="fas fa-undo mr-2"></i>Kembali ke Halaman Pertama
                    </button>
                </div>';
            }
            $html = str_replace('{{pagination_content}}', $paginationContent, $html);
            
            return response($html)->header('Content-Type', 'text/html');
        }
        
        // Fallback ke JSON
        return response()->json([
            'status' => 'success',
            'message' => 'Objek Wisata Desa',
            'data' => $data
        ]);
    }

    private function renderTourismDetailPage($data = [])
    {
        // Path ke file HTML frontend untuk detail wisata
        $htmlPath = resource_path('views/frontend/detail-wisata.php');
        
        if (file_exists($htmlPath)) {
            $html = file_get_contents($htmlPath);
            
            // Extract destination data
            $destinasi = $data['destinasi'] ?? [];
            
            // Replace placeholders with actual data
            foreach ($destinasi as $key => $value) {
                $html = str_replace('{{' . $key . '}}', $value, $html);
            }
            
            // Generate payment methods HTML
            $paymentMethodsHtml = $this->paymentMethodSyncService->generatePaymentOptionsHtml('tourism');
            
            // Replace payment methods section
            $html = preg_replace(
                '/<!-- Payment Method -->.*?<\/div>\s*<\/div>/s',
                '<!-- Payment Method -->
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-3">Metode Pembayaran</label>
                            <div class="space-y-2">
                                ' . $paymentMethodsHtml . '
                            </div>
                        </div>',
                $html
            );
            
            return response($html)->header('Content-Type', 'text/html');
        }
        
        // Fallback ke JSON
        return response()->json([
            'status' => 'success',
            'message' => 'Detail Objek Wisata',
            'data' => $data
        ]);
    }

    public function beliTiketWisata(Request $request)
    {
        try {
            $validated = $request->validate([
                'destinasi_wisata' => 'required|string|max:255',
                'nama_pengunjung' => 'required|string|max:255',
                'email' => 'required|email',
                'no_hp' => 'required|string|max:15',
                'tanggal_kunjungan' => 'required|date|after_or_equal:today',
                'jumlah_tiket' => 'required|integer|min:1|max:10',
                'catatan' => 'nullable|string',
                'metode_pembayaran' => 'required|string',
                'harga_per_tiket' => 'required|numeric|min:0',
                'total_harga' => 'required|numeric|min:0'
            ]);

            // Generate nomor tiket dan kode booking
            $validated['nomor_tiket'] = TiketWisata::generateNomorTiket();
            $validated['kode_booking'] = TiketWisata::generateKodeBooking();
            
            // Set status berdasarkan metode pembayaran
            if ($validated['metode_pembayaran'] === 'cash') {
                $validated['status_pembayaran'] = 'pending_cash';
            } else {
                $validated['status_pembayaran'] = 'pending_payment';
            }

            // Simpan tiket
            $tiket = TiketWisata::create($validated);

            return response()->json([
                'success' => true,
                'message' => 'Tiket berhasil dipesan!',
                'data' => [
                    'kode_booking' => $tiket->kode_booking,
                    'nomor_tiket' => $tiket->nomor_tiket,
                    'status_pembayaran' => $tiket->status_pembayaran,
                    'redirect_url' => $validated['metode_pembayaran'] === 'cash' 
                        ? '/tiket-saya/' . $tiket->kode_booking
                        : '/pembayaran-tiket/' . $tiket->kode_booking
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal memesan tiket: ' . $e->getMessage()
            ], 500);
        }
    }

    public function pembayaranTiket($kode_booking)
    {
        $tiket = TiketWisata::where('kode_booking', $kode_booking)->firstOrFail();
        
        if ($tiket->status_pembayaran !== 'pending_payment') {
            return redirect('/tiket-saya/' . $kode_booking);
        }

        return $this->renderPaymentPage(compact('tiket'));
    }

    public function tiketSaya($kode_booking)
    {
        $tiket = TiketWisata::where('kode_booking', $kode_booking)->firstOrFail();
        
        return $this->renderTicketPage(compact('tiket'));
    }

    private function renderPaymentPage($data = [])
    {
        // Path ke file HTML frontend untuk pembayaran
        $htmlPath = resource_path('views/frontend/pembayaran-tiket.php');
        
        if (file_exists($htmlPath)) {
            $html = file_get_contents($htmlPath);
            
            // Extract tiket data
            $tiket = $data['tiket'] ?? null;
            
            if ($tiket) {
                // Replace placeholders with actual data
                $html = str_replace('{{kode_booking}}', $tiket->kode_booking, $html);
                $html = str_replace('{{nomor_tiket}}', $tiket->nomor_tiket, $html);
                $html = str_replace('{{destinasi_wisata}}', $tiket->destinasi_wisata, $html);
                $html = str_replace('{{nama_pengunjung}}', $tiket->nama_pengunjung, $html);
                $html = str_replace('{{tanggal_kunjungan}}', $tiket->tanggal_kunjungan->format('d M Y'), $html);
                $html = str_replace('{{jumlah_tiket}}', $tiket->jumlah_tiket, $html);
                $html = str_replace('{{total_harga}}', 'Rp ' . number_format($tiket->total_harga, 0, ',', '.'), $html);
                $html = str_replace('{{metode_pembayaran}}', ucwords($tiket->metode_pembayaran), $html);
            }
            
            return response($html)->header('Content-Type', 'text/html');
        }
        
        // Fallback ke JSON
        return response()->json([
            'status' => 'success',
            'message' => 'Pembayaran Tiket Wisata',
            'data' => $data
        ]);
    }

    private function renderTicketPage($data = [])
    {
        // Path ke file HTML frontend untuk tiket
        $htmlPath = resource_path('views/frontend/tiket-saya.php');
        
        if (file_exists($htmlPath)) {
            $html = file_get_contents($htmlPath);
            
            // Extract tiket data
            $tiket = $data['tiket'] ?? null;
            
            if ($tiket) {
                // Replace placeholders with actual data
                $html = str_replace('{{kode_booking}}', $tiket->kode_booking, $html);
                $html = str_replace('{{nomor_tiket}}', $tiket->nomor_tiket, $html);
                $html = str_replace('{{destinasi_wisata}}', $tiket->destinasi_wisata, $html);
                $html = str_replace('{{nama_pengunjung}}', $tiket->nama_pengunjung, $html);
                $html = str_replace('{{email}}', $tiket->email, $html);
                $html = str_replace('{{no_hp}}', $tiket->no_hp, $html);
                $html = str_replace('{{tanggal_kunjungan}}', $tiket->tanggal_kunjungan->format('d M Y'), $html);
                $html = str_replace('{{jumlah_tiket}}', $tiket->jumlah_tiket, $html);
                $html = str_replace('{{total_harga}}', 'Rp ' . number_format($tiket->total_harga, 0, ',', '.'), $html);
                $html = str_replace('{{status_pembayaran}}', ucwords(str_replace('_', ' ', $tiket->status_pembayaran)), $html);
                $html = str_replace('{{metode_pembayaran}}', ucwords($tiket->metode_pembayaran), $html);
            }
            
            return response($html)->header('Content-Type', 'text/html');
        }
        
        // Fallback ke JSON
        return response()->json([
            'status' => 'success',
            'message' => 'Tiket Wisata Saya',
            'data' => $data
        ]);
    }

    public function berita()
    {
        // Ambil berita terpublish dengan paginasi
        $berita_list = Berita::where('status', 'Terbit')
                            ->orderBy('created_at', 'desc')
                            ->paginate(10);
        
        $statistik_berita = [
            'total_berita' => Berita::where('status', 'Terbit')->count(),
            'berita_bulan_ini' => Berita::where('status', 'Terbit')
                                      ->whereMonth('created_at', now()->month)
                                      ->whereYear('created_at', now()->year)
                                      ->count(),
            'total_views' => Berita::where('status', 'Terbit')->sum('views'),
            'kategori_count' => Berita::where('status', 'Terbit')
                                    ->distinct('kategori')
                                    ->count()
        ];

        // Ambil berita unggulan (berdasarkan views tertinggi)
        $berita_unggulan = Berita::where('status', 'Terbit')
                                ->orderBy('views', 'desc')
                                ->first();

        // Ambil berita terbaru untuk list
        $berita_terbaru = Berita::where('status', 'Terbit')
                               ->orderBy('created_at', 'desc')
                               ->limit(10)
                               ->get();
        
        return $this->renderServicePage('berita', compact('berita_list', 'statistik_berita', 'berita_unggulan', 'berita_terbaru'));
    }

    public function peta()
    {
        // Ambil data lokasi untuk peta (tanpa lokasi rumah penduduk)
        $lokasi_data = [
            'fasilitas_count' => 18, // Static data - bisa diubah sesuai database
            'umkm_locations' => Umkm::whereNotNull('latitude')
                                   ->whereNotNull('longitude')
                                   ->select('nama_umkm', 'alamat', 'latitude', 'longitude', 'kategori')
                                   ->get(),
            'stats' => [
                'total_fasilitas' => 18,
                'total_umkm' => Umkm::count(),
                'total_wisata' => 6
            ]
        ];
        
        return $this->renderServicePage('peta', $lokasi_data);
    }

    public function apiPetaData()
    {
        try {
            // Data UMKM dari database
            $umkm_locations = Umkm::whereNotNull('latitude')
                                 ->whereNotNull('longitude')
                                 ->select('nama_umkm as title', 'alamat', 'latitude', 'longitude', 'kategori', 'deskripsi', 'foto')
                                 ->get()
                                 ->map(function($item) {
                                     return [
                                         'lat' => (float) $item->latitude,
                                         'lng' => (float) $item->longitude,
                                         'type' => 'umkm',
                                         'title' => $item->title,
                                         'description' => $item->deskripsi ?? 'UMKM ' . $item->kategori,
                                         'foto' => $item->foto ? '/storage/' . $item->foto : 'https://via.placeholder.com/300x200/7C3AED/FFFFFF?text=' . urlencode($item->title)
                                     ];
                                 });

            // Data Wisata dari database
            $wisata_locations = ObjekWisata::whereNotNull('latitude')
                                          ->whereNotNull('longitude')
                                          ->select('nama_objek as title', 'alamat', 'latitude', 'longitude', 'deskripsi', 'foto_utama')
                                          ->get()
                                          ->map(function($item) {
                                              return [
                                                  'lat' => (float) $item->latitude,
                                                  'lng' => (float) $item->longitude,
                                                  'type' => 'wisata',
                                                  'title' => $item->title,
                                                  'description' => $item->deskripsi ?? 'Objek Wisata',
                                                  'foto' => $item->foto_utama ? '/storage/' . $item->foto_utama : 'https://via.placeholder.com/300x200/EA580C/FFFFFF?text=' . urlencode($item->title)
                                              ];
                                          });

            // Data Fasilitas Umum (static untuk sekarang, bisa dipindah ke database)
            $fasilitas_locations = collect([
                [
                    'lat' => -6.7185,
                    'lng' => 107.1436,
                    'type' => 'fasilitas',
                    'title' => 'Puskesmas Sukamakmur',
                    'description' => 'Layanan Kesehatan Masyarakat',
                    'foto' => 'https://via.placeholder.com/300x200/2563EB/FFFFFF?text=Puskesmas'
                ],
                [
                    'lat' => -6.7165,
                    'lng' => 107.1416,
                    'type' => 'fasilitas',
                    'title' => 'SDN Sirnajaya 01',
                    'description' => 'Sekolah Dasar Negeri',
                    'foto' => 'https://via.placeholder.com/300x200/2563EB/FFFFFF?text=SDN+01'
                ],
                [
                    'lat' => -6.7195,
                    'lng' => 107.1446,
                    'type' => 'fasilitas',
                    'title' => 'Masjid Al-Ikhlas',
                    'description' => 'Tempat Ibadah Umat Islam',
                    'foto' => 'https://via.placeholder.com/300x200/2563EB/FFFFFF?text=Masjid'
                ]
            ]);

            // Data Kantor Desa
            $kantor_locations = collect([
                [
                    'lat' => -6.7175,
                    'lng' => 107.1426,
                    'type' => 'kantor',
                    'title' => 'Kantor Desa Sirnajaya',
                    'description' => 'Balai Desa - Pusat Administrasi',
                    'foto' => 'https://via.placeholder.com/300x200/DC2626/FFFFFF?text=Kantor+Desa'
                ]
            ]);

            // Gabungkan semua data
            $all_locations = $kantor_locations
                           ->concat($fasilitas_locations)
                           ->concat($umkm_locations)
                           ->concat($wisata_locations);

            // Statistik
            $stats = [
                'total_kantor' => $kantor_locations->count(),
                'total_fasilitas' => $fasilitas_locations->count(),
                'total_umkm' => $umkm_locations->count(),
                'total_wisata' => $wisata_locations->count(),
                'total_lokasi' => $all_locations->count()
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'locations' => $all_locations,
                    'stats' => $stats
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data peta: ' . $e->getMessage()
            ], 500);
        }
    }

    public function cekStatusLayanan(Request $request)
    {
        try {
            $validated = $request->validate([
                'tipe' => 'required|in:surat,pengaduan',
                'nomor' => 'required|string'
            ]);
            
            $data = null;
            
            if ($validated['tipe'] === 'surat') {
                $data = PelayananSurat::where('nomor_antrian', $validated['nomor'])->first();
            } else {
                $data = Pengaduan::where('nomor_tiket', $validated['nomor'])->first();
            }
            
            if ($data) {
                return response()->json([
                    'success' => true,
                    'data' => $data
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Data tidak ditemukan'
                ]);
            }
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan sistem'
            ], 500);
        }
    }

    /**
     * Get available payment methods
     */
    public function getPaymentMethods($type)
    {
        try {
            if ($type === 'tourism') {
                $methods = $this->paymentMethodSyncService->getTourismPaymentMethods();
            } elseif ($type === 'umkm') {
                $methods = $this->paymentMethodSyncService->getUmkmPaymentMethods();
            } else {
                $methods = $this->paymentMethodSyncService->getAvailablePaymentMethods();
            }

            return response()->json([
                'success' => true,
                'data' => $methods
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil metode pembayaran: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show registration form for population data
     */
    public function registrasiPenduduk()
    {
        // Refresh session to prevent timeout
        $request = request();
        $request->session()->regenerate();
        
        return view('frontend.registrasi-penduduk');
    }

    /**
     * Store registration data
     */
    public function storeRegistrasiPenduduk(Request $request)
    {
        try {
            // Debug: Log the request immediately
            Log::info('=== REGISTRASI PENDUKUD REQUEST START ===');
            Log::info('Registrasi penduduk request received', [
                'method' => $request->method(),
                'url' => $request->url(),
                'headers' => $request->headers->all(),
                'data' => $request->all(),
                'csrf_token_header' => $request->header('X-CSRF-TOKEN'),
                'csrf_token_form' => $request->input('_token'),
                'csrf_token_session' => csrf_token(),
                'session_id' => $request->session()->getId(),
                'is_ajax' => $request->ajax(),
                'wants_json' => $request->wantsJson(),
                'content_type' => $request->header('Content-Type'),
                'user_agent' => $request->header('User-Agent')
            ]);
            
            // Check CSRF token manually for better error handling
            $headerToken = $request->header('X-CSRF-TOKEN');
            $formToken = $request->input('_token');
            $sessionToken = csrf_token();
            
            if (!$headerToken && !$formToken) {
                Log::error('No CSRF token provided');
                if ($request->ajax() || $request->wantsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'CSRF token tidak ditemukan. Silakan refresh halaman dan coba lagi.',
                        'error_type' => 'csrf_missing'
                    ], 419);
                } else {
                    return redirect()->back()
                        ->withErrors(['csrf' => 'CSRF token tidak ditemukan. Silakan refresh halaman dan coba lagi.'])
                        ->withInput();
                }
            }
            
            if ($headerToken !== $sessionToken && $formToken !== $sessionToken) {
                Log::error('CSRF token mismatch', [
                    'header_token' => $headerToken,
                    'form_token' => $formToken,
                    'session_token' => $sessionToken
                ]);
                if ($request->ajax() || $request->wantsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'CSRF token tidak valid. Silakan refresh halaman dan coba lagi.',
                        'error_type' => 'csrf_mismatch'
                    ], 419);
                } else {
                    return redirect()->back()
                        ->withErrors(['csrf' => 'CSRF token tidak valid. Silakan refresh halaman dan coba lagi.'])
                        ->withInput();
                }
            }
            
            // Check if this is an AJAX request
            if ($request->ajax() || $request->wantsJson() || $request->header('X-Requested-With') === 'XMLHttpRequest') {
                Log::info('Handling as AJAX request');
                // Handle AJAX request
                return $this->handleAjaxRegistrasi($request);
            } else {
                Log::info('Handling as traditional form submission');
                // Handle traditional form submission
                return $this->handleTraditionalRegistrasi($request);
            }
            
        } catch (\Exception $e) {
            Log::error('Error in storeRegistrasiPenduduk', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Terjadi kesalahan sistem. Silakan coba lagi nanti.'
                ], 500);
            } else {
                return redirect()->back()
                    ->withErrors(['general' => 'Terjadi kesalahan sistem. Silakan coba lagi nanti.'])
                    ->withInput();
            }
        }
    }
    
    /**
     * Handle AJAX registration request
     */
    private function handleAjaxRegistrasi(Request $request)
    {
        // Validate with Dukcapil if enabled
        if (config('services.dukcapil.api_key')) {
            try {
                $dukcapilService = app(\App\Services\DukcapilService::class);
                $nikVerification = $dukcapilService->verifyNik($request->nik);
                
                if (!$nikVerification['success']) {
                    return response()->json([
                        'success' => false,
                        'message' => 'NIK tidak valid atau tidak terdaftar di Dukcapil: ' . $nikVerification['message']
                    ], 422);
                }
                
                // Auto-fill data from Dukcapil
                if ($nikVerification['data']) {
                    $dukcapilData = $nikVerification['data'];
                    $request->merge([
                        'nama' => $dukcapilData['nama'] ?? $request->nama,
                        'tempat_lahir' => $dukcapilData['tempat_lahir'] ?? $request->tempat_lahir,
                        'tanggal_lahir' => $dukcapilData['tanggal_lahir'] ?? $request->tanggal_lahir,
                        'jenis_kelamin' => $dukcapilData['jenis_kelamin'] ?? $request->jenis_kelamin,
                        'agama' => $dukcapilData['agama'] ?? $request->agama,
                    ]);
                }
            } catch (\Exception $e) {
                Log::warning('Dukcapil service error: ' . $e->getMessage());
                // Continue without Dukcapil validation if service fails
            }
        }

        try {
            $validated = $request->validate([
                'nik' => 'required|string|size:16|unique:penduduk',
                'nama' => 'required|string|max:255',
                'no_kk' => 'nullable|string|size:16',
                'alamat' => 'required|string',
                'dusun' => 'nullable|string',
                'rt' => 'nullable|string|max:5',
                'rw' => 'nullable|string|max:5',
                'tempat_lahir' => 'required|string',
                'tanggal_lahir' => 'required|date',
                'jenis_kelamin' => 'required|in:L,P',
                'status_perkawinan' => 'required|in:Belum Kawin,Kawin,Cerai Hidup,Cerai Mati',
                'agama' => 'required|string',
                'pekerjaan' => 'nullable|string',
                'pendidikan_terakhir' => 'nullable|in:Tidak Sekolah,SD,SMP,SMA,D1,D2,D3,S1,S2,S3',
                'golongan_darah' => 'nullable|in:A,B,AB,O',
                'kewarganegaraan' => 'required|in:WNI,WNA',
                'no_wa' => 'nullable|string|max:15',
                'status_hidup' => 'required|in:Hidup,Meninggal',
                'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'foto_rumah' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'latitude' => 'nullable|numeric|between:-90,90',
                'longitude' => 'nullable|numeric|between:-180,180',
                'ayah_nik' => 'nullable|string|size:16',
                'nama_ayah' => 'nullable|string|max:255',
                'ibu_nik' => 'nullable|string|size:16',
                'nama_ibu' => 'nullable|string|max:255',
                'hubungan_keluarga' => 'nullable|string|max:255',
                'no_akta_lahir' => 'nullable|string|max:50',
                'tanggal_akta_lahir' => 'nullable|date',
                'tempat_akta_lahir' => 'nullable|string|max:255',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation failed in AJAX registrasi', [
                'errors' => $e->errors(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $e->errors()
            ], 422);
        }

        // Handle file uploads
        try {
            $uploadService = new \App\Services\FileUploadService();
            
            // Handle foto upload
            if ($request->hasFile('foto')) {
                $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto'), 'penduduk');
                
                if ($uploadResult['success']) {
                    $validated['foto'] = $uploadResult['path'];
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'Gagal mengupload foto: ' . ($uploadResult['error'] ?? 'Unknown error')
                    ], 422);
                }
            }
            
            // Handle foto rumah upload
            if ($request->hasFile('foto_rumah')) {
                $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto_rumah'), 'penduduk');
                
                if ($uploadResult['success']) {
                    $validated['foto_rumah'] = $uploadResult['path'];
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'Gagal mengupload foto rumah: ' . ($uploadResult['error'] ?? 'Unknown error')
                    ], 422);
                }
            }
        } catch (\Exception $e) {
            Log::error('File upload service error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengupload file. Silakan coba lagi.'
            ], 422);
        }

        // Map status_perkawinan to status_kawin for database compatibility
        if (isset($validated['status_perkawinan'])) {
            $validated['status_kawin'] = $validated['status_perkawinan'];
            unset($validated['status_perkawinan']);
        }

        // Map pendidikan_terakhir to pendidikan for database compatibility
        if (isset($validated['pendidikan_terakhir'])) {
            $validated['pendidikan'] = $validated['pendidikan_terakhir'];
            unset($validated['pendidikan_terakhir']); // Remove the original field
        } else {
            $validated['pendidikan'] = 'Tidak Sekolah'; // Default value
        }

        // Provide default value for no_kk if empty (to prevent database error)
        if (empty($validated['no_kk'])) {
            $validated['no_kk'] = '0000000000000000'; // Default 16-digit value
        }

        // Provide default values for other required fields
        if (empty($validated['pekerjaan'])) {
            $validated['pekerjaan'] = 'Belum Bekerja'; // Default value
        }

        // Validate ayah_nik and ibu_nik exist in database before saving
        if (!empty($validated['ayah_nik'])) {
            $ayahExists = \App\Models\Penduduk::where('nik', $validated['ayah_nik'])->exists();
            if (!$ayahExists) {
                Log::warning('Ayah NIK not found in database', ['ayah_nik' => $validated['ayah_nik']]);
                $validated['ayah_nik'] = null; // Set to null if not found
            }
        }
        
        if (!empty($validated['ibu_nik'])) {
            $ibuExists = \App\Models\Penduduk::where('nik', $validated['ibu_nik'])->exists();
            if (!$ibuExists) {
                Log::warning('Ibu NIK not found in database', ['ibu_nik' => $validated['ibu_nik']]);
                $validated['ibu_nik'] = null; // Set to null if not found
            }
        }

        // Filter validated data to only include fillable fields
        $fillableFields = (new \App\Models\Penduduk())->getFillable();
        $filteredData = array_intersect_key($validated, array_flip($fillableFields));
        
        // Debug: Log the final validated data before database insertion
        Log::info('Final validated data for database insertion', [
            'validated_data' => $filteredData,
            'data_keys' => array_keys($filteredData),
            'fillable_fields' => $fillableFields
        ]);

        // Create penduduk record with error handling
        try {
            $penduduk = Penduduk::create($filteredData);
        } catch (\Exception $e) {
            Log::error('Error creating penduduk record', [
                'error' => $e->getMessage(),
                'validated_data' => $validated,
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Gagal menyimpan data penduduk: ' . $e->getMessage()
            ], 500);
        }

        // Auto-mapping nama orang tua ke NIK jika belum ada NIK
        try {
            if (!empty($validated['nama_ayah']) && empty($validated['ayah_nik'])) {
                $penduduk->mapNamaAyahToNik();
            }
            if (!empty($validated['nama_ibu']) && empty($validated['ibu_nik'])) {
                $penduduk->mapNamaIbuToNik();
            }
        } catch (\Exception $e) {
            Log::warning('Error mapping orang tua to NIK: ' . $e->getMessage());
            // Continue without mapping if it fails
        }

        // Handle keluarga data if no_kk is provided
        if (!empty($validated['no_kk'])) {
            $this->handleKeluargaData($penduduk);
        }

        // Debug: Log success
        Log::info('Registrasi penduduk successful', [
            'nik' => $penduduk->nik,
            'nama' => $penduduk->nama
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Registrasi data penduduk berhasil! Data Anda telah diterima dan akan diverifikasi oleh petugas. Foto dan lokasi rumah telah disimpan untuk keperluan verifikasi.',
            'redirect' => '/'
        ]);
    }
    
    /**
     * Handle traditional form submission
     */
    private function handleTraditionalRegistrasi(Request $request)
    {
        try {
            // Same validation and processing logic as AJAX
            $validated = $request->validate([
                'nik' => 'required|string|size:16|unique:penduduk',
                'nama' => 'required|string|max:255',
                'no_kk' => 'nullable|string|size:16',
                'alamat' => 'required|string',
                'dusun' => 'nullable|string',
                'rt' => 'nullable|string|max:5',
                'rw' => 'nullable|string|max:5',
                'tempat_lahir' => 'required|string',
                'tanggal_lahir' => 'required|date',
                'jenis_kelamin' => 'required|in:L,P',
                'status_perkawinan' => 'required|in:Belum Kawin,Kawin,Cerai Hidup,Cerai Mati',
                'agama' => 'required|string',
                'pekerjaan' => 'nullable|string',
                'pendidikan_terakhir' => 'nullable|in:Tidak Sekolah,SD,SMP,SMA,D1,D2,D3,S1,S2,S3',
                'golongan_darah' => 'nullable|in:A,B,AB,O',
                'kewarganegaraan' => 'required|in:WNI,WNA',
                'no_wa' => 'nullable|string|max:15',
                'status_hidup' => 'required|in:Hidup,Meninggal',
                'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'foto_rumah' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'latitude' => 'nullable|numeric|between:-90,90',
                'longitude' => 'nullable|numeric|between:-180,180',
                'ayah_nik' => 'nullable|string|size:16',
                'nama_ayah' => 'nullable|string|max:255',
                'ibu_nik' => 'nullable|string|size:16',
                'nama_ibu' => 'nullable|string|max:255',
                'hubungan_keluarga' => 'nullable|string|max:255',
                'no_akta_lahir' => 'nullable|string|max:50',
                'tanggal_akta_lahir' => 'nullable|date',
                'tempat_akta_lahir' => 'nullable|string|max:255',
            ]);

            // Handle file uploads
            try {
                $uploadService = new \App\Services\FileUploadService();
                
                if ($request->hasFile('foto')) {
                    $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto'), 'penduduk');
                    if ($uploadResult['success']) {
                        $validated['foto'] = $uploadResult['path'];
                    }
                }
                
                if ($request->hasFile('foto_rumah')) {
                    $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto_rumah'), 'penduduk');
                    if ($uploadResult['success']) {
                        $validated['foto_rumah'] = $uploadResult['path'];
                    }
                }
            } catch (\Exception $e) {
                Log::error('File upload service error in traditional form: ' . $e->getMessage());
                // Continue without file uploads if service fails
            }

            // Map status_perkawinan to status_kawin
            if (isset($validated['status_perkawinan'])) {
                $validated['status_kawin'] = $validated['status_perkawinan'];
                unset($validated['status_perkawinan']);
            }

            // Map pendidikan_terakhir to pendidikan for database compatibility
            if (isset($validated['pendidikan_terakhir'])) {
                $validated['pendidikan'] = $validated['pendidikan_terakhir'];
                unset($validated['pendidikan_terakhir']); // Remove the original field
            } else {
                $validated['pendidikan'] = 'Tidak Sekolah'; // Default value
            }

            // Provide default value for no_kk if empty (to prevent database error)
            if (empty($validated['no_kk'])) {
                $validated['no_kk'] = '0000000000000000'; // Default 16-digit value
            }

            // Provide default values for other required fields
            if (empty($validated['pekerjaan'])) {
                $validated['pekerjaan'] = 'Belum Bekerja'; // Default value
            }

            // Validate ayah_nik and ibu_nik exist in database before saving
            if (!empty($validated['ayah_nik'])) {
                $ayahExists = \App\Models\Penduduk::where('nik', $validated['ayah_nik'])->exists();
                if (!$ayahExists) {
                    Log::warning('Ayah NIK not found in database (traditional)', ['ayah_nik' => $validated['ayah_nik']]);
                    $validated['ayah_nik'] = null; // Set to null if not found
                }
            }
            
            if (!empty($validated['ibu_nik'])) {
                $ibuExists = \App\Models\Penduduk::where('nik', $validated['ibu_nik'])->exists();
                if (!$ibuExists) {
                    Log::warning('Ibu NIK not found in database (traditional)', ['ibu_nik' => $validated['ibu_nik']]);
                    $validated['ibu_nik'] = null; // Set to null if not found
                }
            }

            // Create penduduk record
            $penduduk = Penduduk::create($validated);

            // Auto-mapping nama orang tua ke NIK jika belum ada NIK
            if (!empty($validated['nama_ayah']) && empty($validated['ayah_nik'])) {
                $penduduk->mapNamaAyahToNik();
            }
            if (!empty($validated['nama_ibu']) && empty($validated['ibu_nik'])) {
                $penduduk->mapNamaIbuToNik();
            }

            // Handle keluarga data
            if (!empty($validated['no_kk'])) {
                $this->handleKeluargaData($penduduk);
            }

            Log::info('Traditional registrasi successful', [
                'nik' => $penduduk->nik,
                'nama' => $penduduk->nama
            ]);

            return redirect('/')
                ->with('success', 'Registrasi data penduduk berhasil! Data Anda telah diterima dan akan diverifikasi oleh petugas.');
                
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Validation error in traditional registrasi', [
                'errors' => $e->errors(),
                'request_data' => $request->all()
            ]);
            
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
                
        } catch (\Exception $e) {
            Log::error('Error in traditional registrasi', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return redirect()->back()
                ->withErrors(['general' => 'Terjadi kesalahan sistem: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Handle keluarga data creation
     */
    private function handleKeluargaData($penduduk)
    {
        try {
            // Check if keluarga with no_kk already exists
            $keluarga = Keluarga::where('no_kk', $penduduk->no_kk)->first();
            
            if (!$keluarga) {
                // Create new keluarga if doesn't exist
                $keluarga = Keluarga::create([
                    'no_kk' => $penduduk->no_kk,
                    'kepala_keluarga_nik' => $penduduk->nik,
                    'alamat' => $penduduk->alamat,
                    'dusun' => $penduduk->dusun,
                    'rt' => $penduduk->rt,
                    'rw' => $penduduk->rw
                ]);
            }

            // Check if penduduk is already registered as family member
            $anggota = AnggotaKeluarga::where('no_kk', $penduduk->no_kk)
                                                 ->where('nik', $penduduk->nik)
                                                 ->first();

            if (!$anggota) {
                // Determine status in family
                $existingMembers = AnggotaKeluarga::where('no_kk', $penduduk->no_kk)->count();
                $statusKeluarga = $existingMembers == 0 ? 'Kepala Keluarga' : 'Anggota Keluarga';

                AnggotaKeluarga::create([
                    'no_kk' => $penduduk->no_kk,
                    'nik' => $penduduk->nik,
                    'status_keluarga' => $statusKeluarga
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Error handling keluarga data: ' . $e->getMessage());
        }
    }
} 
