<?php

namespace App\Http\Controllers;

use App\Models\WilayahAdministratif;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class WilayahAdministratifController extends Controller
{
    public function index(Request $request)
    {
        // Optimize query with specific column selection and pagination
        $query = WilayahAdministratif::with(['user:id,name', 'parent:id,nama_wilayah'])
            ->select(['id', 'jenis_wilayah', 'kode_wilayah', 'nama_wilayah', 
                     'parent_id', 'kepala_wilayah', 'jumlah_kk', 'jumlah_penduduk', 'user_id', 'status']);

        if ($request->has('tipe')) {
            $query->where('jenis_wilayah', $request->tipe);
        }

        if ($request->has('parent')) {
            $query->where('parent_id', $request->parent);
        }

        // Add pagination (20 items per page)
        $wilayah = $query->orderBy('jenis_wilayah')->orderBy('kode_wilayah')->paginate(20);

        // Use cached statistics for better performance
        $stats = WilayahAdministratif::getStatistikWilayah();

        // Optimize dropdown lists
        $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')
                                       ->select(['id', 'nama_wilayah'])
                                       ->orderBy('nama_wilayah')
                                       ->get();
        $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')
                                    ->select(['id', 'nama_wilayah', 'parent_id'])
                                    ->with('parent:id,nama_wilayah')
                                    ->orderBy('nama_wilayah')
                                    ->get();

        // If AJAX request, return JSON
        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'data' => $wilayah,
                'stats' => $stats
            ]);
        }

        // Return view for normal requests
        return view('admin.wilayah-administratif.index', compact('wilayah', 'stats', 'dusunList', 'rwList'));
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'jenis_wilayah' => 'required|in:dusun,rw,rt',
            'kode_wilayah' => 'required|string|max:20|unique:wilayah_administratif,kode_wilayah',
            'nama_wilayah' => 'required|string|max:255',
            'parent_id' => 'nullable|exists:wilayah_administratif,id',
            'kepala_wilayah' => 'nullable|string|max:255',
            'nomor_hp_kepala' => 'nullable|string|max:15',
            'jumlah_kk' => 'nullable|integer|min:0',
            'jumlah_penduduk' => 'nullable|integer|min:0',
            'luas_wilayah' => 'nullable|numeric|min:0',
            'batas_wilayah' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $validator->validated();
        $data['user_id'] = Auth::id();

        $wilayah = WilayahAdministratif::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Data wilayah administratif berhasil ditambahkan',
                'data' => $wilayah
            ], 201);
        }

        return redirect()->route('admin.wilayah-administratif.index')
                        ->with('success', 'Data wilayah administratif berhasil ditambahkan');
    }

    public function create()
    {
        $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')->get();
        $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')->with('parent')->get();
        
        return view('admin.wilayah-administratif.create', compact('dusunList', 'rwList'));
    }

    public function edit($id)
    {
        $wilayah = WilayahAdministratif::with(['parent'])->findOrFail($id);
        $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')->where('id', '!=', $id)->get();
        $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')->where('id', '!=', $id)->with('parent')->get();
        
        return view('admin.wilayah-administratif.edit', compact('wilayah', 'dusunList', 'rwList'));
    }

    public function show($id)
    {
        try {
            Log::info('Accessing show method for wilayah administratif', ['id' => $id]);
            
            $wilayah = WilayahAdministratif::with(['user', 'parent', 'children'])->findOrFail($id);
            
            Log::info('Wilayah found', ['wilayah' => $wilayah->toArray()]);

            // If AJAX request, return JSON
            if (request()->wantsJson() || request()->ajax()) {
                Log::info('Returning JSON response');
                return response()->json([
                    'success' => true,
                    'data' => $wilayah
                ]);
            }

            // Return view for normal requests
            Log::info('Returning view response');
            return view('admin.wilayah-administratif.show', compact('wilayah'));

        } catch (\Exception $e) {
            Log::error('Error showing wilayah administratif: ' . $e->getMessage(), [
                'wilayah_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);

            if (request()->wantsJson() || request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal memuat data wilayah administratif: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->route('admin.wilayah-administratif.index')
                            ->with('error', 'Gagal memuat data wilayah administratif: ' . $e->getMessage());
        }
    }

    public function update(Request $request, $id)
    {
        $wilayah = WilayahAdministratif::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'jenis_wilayah' => 'required|in:dusun,rw,rt',
            'kode_wilayah' => 'required|string|max:20|unique:wilayah_administratif,kode_wilayah,' . $id,
            'nama_wilayah' => 'required|string|max:255',
            'parent_id' => 'nullable|exists:wilayah_administratif,id',
            'kepala_wilayah' => 'nullable|string|max:255',
            'nomor_hp_kepala' => 'nullable|string|max:15',
            'jumlah_kk' => 'nullable|integer|min:0',
            'jumlah_penduduk' => 'nullable|integer|min:0',
            'luas_wilayah' => 'nullable|numeric|min:0',
            'batas_wilayah' => 'nullable|string',
            'status' => 'required|in:aktif,nonaktif',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $validator->validated();
        $wilayah->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Data wilayah administratif berhasil diperbarui',
                'data' => $wilayah
            ]);
        }

        return redirect()->route('admin.wilayah-administratif.index')
                        ->with('success', 'Data wilayah administratif berhasil diperbarui');
    }

    public function destroy($id)
    {
        try {
            Log::info('Accessing destroy method for wilayah administratif', ['id' => $id]);
            
            $wilayah = WilayahAdministratif::findOrFail($id);
            Log::info('Wilayah found for deletion', ['wilayah' => $wilayah->toArray()]);

            // Check if has children
            $childrenCount = $wilayah->children()->count();
            Log::info('Children count check', ['children_count' => $childrenCount]);
            
            if ($childrenCount > 0) {
                Log::warning('Cannot delete wilayah with children', ['wilayah_id' => $id, 'children_count' => $childrenCount]);
                
                if (request()->wantsJson() || request()->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Tidak dapat menghapus wilayah yang masih memiliki wilayah di bawahnya'
                    ], 400);
                }
                return redirect()->route('admin.wilayah-administratif.index')
                                ->with('error', 'Tidak dapat menghapus wilayah yang masih memiliki wilayah di bawahnya');
            }

            // Check if wilayah is referenced by penduduk
            $pendudukCount = \App\Models\Penduduk::where('dusun', $wilayah->nama_wilayah)
                                                  ->orWhere('rt', $wilayah->nama_wilayah)
                                                  ->orWhere('rw', $wilayah->nama_wilayah)
                                                  ->count();
            
            Log::info('Penduduk count check', ['penduduk_count' => $pendudukCount]);
            
            if ($pendudukCount > 0) {
                Log::warning('Cannot delete wilayah with penduduk data', ['wilayah_id' => $id, 'penduduk_count' => $pendudukCount]);
                
                if (request()->wantsJson() || request()->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Tidak dapat menghapus wilayah yang masih memiliki data penduduk'
                    ], 400);
                }
                return redirect()->route('admin.wilayah-administratif.index')
                                ->with('error', 'Tidak dapat menghapus wilayah yang masih memiliki data penduduk');
            }

            Log::info('Proceeding with deletion', ['wilayah_id' => $id]);
            $wilayah->delete();
            Log::info('Wilayah deleted successfully', ['wilayah_id' => $id]);

            if (request()->wantsJson() || request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Data wilayah administratif berhasil dihapus'
                ]);
            }

            return redirect()->route('admin.wilayah-administratif.index')
                            ->with('success', 'Data wilayah administratif berhasil dihapus');

        } catch (\Exception $e) {
            Log::error('Error deleting wilayah administratif: ' . $e->getMessage(), [
                'wilayah_id' => $id,
                'trace' => $e->getTraceAsString()
            ]);
            
            if (request()->wantsJson() || request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal menghapus data wilayah administratif: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->route('admin.wilayah-administratif.index')
                            ->with('error', 'Gagal menghapus data wilayah administratif: ' . $e->getMessage());
        }
    }

    public function byJenis($jenis)
    {
        try {
            $query = WilayahAdministratif::where('jenis_wilayah', $jenis)
                                        ->where('status', 'aktif')
                                        ->orderBy('nama_wilayah');

            // Filter berdasarkan parent jika ada
            if (request()->has('parent_dusun') && $jenis === 'rw') {
                $parentDusun = request('parent_dusun');
                $query->whereHas('parent', function($q) use ($parentDusun) {
                    $q->where('nama_wilayah', $parentDusun);
                });
            }

            if (request()->has('parent_rw') && $jenis === 'rt') {
                $parentRw = request('parent_rw');
                $query->whereHas('parent', function($q) use ($parentRw) {
                    $q->where('nama_wilayah', $parentRw);
                });
            }

            $wilayah = $query->get();

            Log::info('Wilayah by jenis loaded', [
                'jenis' => $jenis,
                'count' => $wilayah->count(),
                'filters' => request()->all()
            ]);

            return response()->json([
                'success' => true,
                'data' => $wilayah
            ]);

        } catch (\Exception $e) {
            Log::error('Error loading wilayah by jenis: ' . $e->getMessage(), [
                'jenis' => $jenis,
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Gagal memuat data wilayah: ' . $e->getMessage()
            ], 500);
        }
    }

    public function treeStructure()
    {
        $tree = WilayahAdministratif::getTreeStructure();

        return response()->json([
            'success' => true,
            'data' => $tree
        ]);
    }

    public function statistik()
    {
        try {
            $stats = WilayahAdministratif::getStatistikRealTime();

            return response()->json([
                'success' => true,
                'data' => $stats
            ]);
        } catch (\Exception $e) {
            Log::error('Error getting real-time statistics: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil statistik real-time: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateStatistik($id)
    {
        $wilayah = WilayahAdministratif::findOrFail($id);
        $wilayah->updateStatistikPenduduk();

        return response()->json([
            'success' => true,
            'message' => 'Statistik penduduk berhasil diperbarui'
        ]);
    }

    public function dusunWithRwRt()
    {
        $dusun = WilayahAdministratif::getDusunWithRwRt();

        return response()->json([
            'success' => true,
            'data' => $dusun
        ]);
    }

    public function sync(Request $request)
    {
        try {
            Log::info('Starting wilayah administratif data synchronization');
            
            // Get all active wilayah ordered by hierarchy (RT first, then RW, then Dusun)
            $rtList = WilayahAdministratif::where('jenis_wilayah', 'rt')->where('status', 'aktif')->get();
            $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')->where('status', 'aktif')->get();
            $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')->where('status', 'aktif')->get();
            
            $updated = [];
            
            // Update RT first (most granular level)
            foreach ($rtList as $rt) {
                $stats = $rt->updateStatistikPenduduk();
                $updated[] = [
                    'nama' => $rt->nama_wilayah,
                    'jenis' => 'RT',
                    'kk' => $stats['jumlah_kk'],
                    'penduduk' => $stats['jumlah_penduduk']
                ];
            }
            
            // Update RW (aggregate from RT or direct from data)
            foreach ($rwList as $rw) {
                $stats = $rw->updateStatistikPenduduk();
                $updated[] = [
                    'nama' => $rw->nama_wilayah,
                    'jenis' => 'RW',
                    'kk' => $stats['jumlah_kk'],
                    'penduduk' => $stats['jumlah_penduduk']
                ];
            }
            
            // Update Dusun (aggregate from RW or direct from data)
            foreach ($dusunList as $dusun) {
                $stats = $dusun->updateStatistikPenduduk();
                $updated[] = [
                    'nama' => $dusun->nama_wilayah,
                    'jenis' => 'Dusun',
                    'kk' => $stats['jumlah_kk'],
                    'penduduk' => $stats['jumlah_penduduk']
                ];
            }
            
            // Clear cache
            \Illuminate\Support\Facades\Cache::forget('wilayah_statistik');
            
            Log::info('Wilayah administratif synchronization completed', [
                'updated_count' => count($updated),
                'rt_count' => $rtList->count(),
                'rw_count' => $rwList->count(),
                'dusun_count' => $dusunList->count()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Data penduduk berhasil disinkronkan dengan wilayah administratif',
                'summary' => [
                    'total_updated' => count($updated),
                    'rt_updated' => $rtList->count(),
                    'rw_updated' => $rwList->count(),
                    'dusun_updated' => $dusunList->count()
                ],
                'details' => $updated
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error syncing wilayah administratif: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Gagal melakukan sinkronisasi: ' . $e->getMessage()
            ], 500);
        }
    }
} 