<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PerangkatDesa extends Model
{
    use HasFactory;

    protected $table = 'perangkat_desa';

    protected $fillable = [
        'nik',
        'nama',
        'nip',
        'jabatan',
        'bidang_tugas',
        'tugas_tanggung_jawab',
        'mulai_tugas',
        'selesai_tugas',
        'status',
        'no_hp',
        'email',
        'alamat',
        'foto',
        'file_sk',
        'tanda_tangan',
        'wilayah_kerja',
        'urutan',
        'level_hierarki',
        'telepon',
        'pendidikan',
        'tanggal_mulai_tugas',
        'status_kepegawaian',
        'is_active'
    ];

    protected $casts = [
        'mulai_tugas' => 'date',
        'selesai_tugas' => 'date',
        'tanggal_mulai_tugas' => 'date',
        'is_active' => 'boolean'
    ];

    // Relationships
    public function penduduk()
    {
        return $this->belongsTo(Penduduk::class, 'nik', 'nik');
    }

    public function user()
    {
        return $this->hasOne(User::class, 'email', 'email');
    }

    // Scope methods
    public function scopeAktif($query)
    {
        // Pastikan $query adalah Builder instance
        if (!$query instanceof \Illuminate\Database\Eloquent\Builder) {
            $query = $this->newQuery();
        }
        return $query->where('status', 'Aktif');
    }

    public function scopeByJabatan($query, $jabatan)
    {
        // Pastikan $query adalah Builder instance
        if (!$query instanceof \Illuminate\Database\Eloquent\Builder) {
            $query = $this->newQuery();
        }
        return $query->where('jabatan', $jabatan);
    }

    public function scopeByLevel($query, $level)
    {
        try {
            // Pastikan query adalah Builder object, bukan string
            if (!$query instanceof \Illuminate\Database\Eloquent\Builder) {
                return $this->newQuery()->where('level_hierarki', $level);
            }
            return $query->where('level_hierarki', $level);
        } catch (\Exception $e) {
            // Fallback to filter by jabatan if level_hierarki doesn't exist
            $jabatanByLevel = [
                0 => ['Kepala Desa'],
                1 => ['Sekretaris Desa'],
                2 => ['Kaur Tata Usaha dan Umum', 'Kaur Keuangan', 'Kaur Perencanaan', 
                      'Kasi Pemerintahan', 'Kasi Kesejahteraan', 'Kasi Pelayanan', 'Kepala Dusun'],
                3 => ['RT', 'RW']
            ];
            
            $jabatan = $jabatanByLevel[$level] ?? [];
            
            if (!$query instanceof \Illuminate\Database\Eloquent\Builder) {
                return $this->newQuery()->whereIn('jabatan', $jabatan);
            }
            return $query->whereIn('jabatan', $jabatan);
        }
    }

    public function scopeOrderByHierarki($query)
    {
        // Pastikan kita bekerja dengan query builder yang valid
        if (!is_object($query) || !method_exists($query, 'orderBy')) {
            // Jika bukan query builder, buat query baru
            $query = $this->newQuery();
        }
        
        try {
            // Coba order berdasarkan level_hierarki dan urutan
            return $query->orderBy('level_hierarki')->orderBy('urutan');
        } catch (\Exception $e) {
            // Fallback ke ordering berdasarkan jabatan
            return $query->orderByRaw("
                CASE jabatan
                    WHEN 'Kepala Desa' THEN 1
                    WHEN 'Sekretaris Desa' THEN 2
                    WHEN 'Kasi Pemerintahan' THEN 3
                    WHEN 'Kasi Kesejahteraan' THEN 4
                    WHEN 'Kasi Kesra' THEN 5
                    WHEN 'Kasi Pelayanan' THEN 6
                    WHEN 'Kasi Pembangunan' THEN 7
                    WHEN 'Kaur Umum' THEN 8
                    WHEN 'Kaur Keuangan' THEN 9
                    WHEN 'Kaur Perencanaan' THEN 10
                    WHEN 'Kepala Dusun' THEN 11
                    WHEN 'Ketua RT' THEN 12
                    WHEN 'Ketua RW' THEN 13
                    ELSE 99
                END, nama
            ");
        }
    }

    // Method untuk get hierarki struktur
    public static function getStrukturHierarki()
    {
        return [
            0 => 'Kepala Desa',
            1 => 'Sekretaris & Staff',
            2 => 'Kepala Dusun',
            3 => 'RT/RW'
        ];
    }

    // Method untuk cek apakah bisa tanda tangan surat
    public function canSignDocument()
    {
        return $this->status === 'Aktif' && 
               in_array($this->jabatan, [
                   'Kepala Desa',
                   'Sekretaris Desa',
                   'Kasi Pemerintahan',
                   'Kasi Kesejahteraan',
                   'Kasi Kesra',
                   'Kasi Pelayanan',
                   'Kasi Pembangunan'
               ]) &&
               $this->tanda_tangan;
    }

    // Method untuk masa jabatan
    public function getMasaJabatan()
    {
        if (!$this->mulai_tugas) {
            return null;
        }

        $selesai = $this->selesai_tugas ?: now();
        return $this->mulai_tugas->diffForHumans($selesai, true);
    }

    // Attribute untuk badge color
    public function getStatusBadgeColorAttribute()
    {
        return match($this->status) {
            'Aktif' => 'success',
            'Tidak Aktif' => 'warning',
            'Pensiun' => 'info',
            'Mutasi' => 'danger',
            default => 'secondary'
        };
    }

    // Method untuk get perangkat by level
    public static function getByLevel($level)
    {
        try {
            return self::aktif()
                      ->where('level_hierarki', $level)
                      ->orderBy('urutan')
                      ->get();
        } catch (\Exception $e) {
            // Fallback manual jika kolom tidak ada
            $jabatanByLevel = [
                0 => ['Kepala Desa'],
                1 => ['Sekretaris Desa'],
                2 => ['Kaur Tata Usaha dan Umum', 'Kaur Keuangan', 'Kaur Perencanaan', 
                      'Kasi Pemerintahan', 'Kasi Kesejahteraan', 'Kasi Pelayanan', 'Kepala Dusun'],
                3 => ['RT', 'RW']
            ];
            
            $jabatan = $jabatanByLevel[$level] ?? [];
            return self::aktif()->whereIn('jabatan', $jabatan)->orderBy('nama')->get();
        }
    }

    // Method untuk get kepala desa aktif
    public static function getKepalaDesa()
    {
        return self::aktif()
                  ->where('jabatan', 'Kepala Desa')
                  ->first();
    }

    // Method untuk get sekretaris desa
    public static function getSekretarisDesa()
    {
        return self::aktif()
                  ->where('jabatan', 'Sekretaris Desa')
                  ->first();
    }

    // Method untuk get perangkat yang bisa sign surat
    public static function getPenandatanganSurat()
    {
        return self::aktif()
                  ->whereIn('jabatan', [
                      'Kepala Desa',
                      'Sekretaris Desa', 
                      'Kasi Pemerintahan',
                      'Kasi Pelayanan'
                  ])
                  ->whereNotNull('tanda_tangan')
                  ->get();
    }
} 