<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class StrukturOrganisasi extends Model
{
    use HasFactory;

    protected $table = 'struktur_organisasi';

    protected $fillable = [
        'kategori_lembaga',
        'nama_lembaga',
        'nama_pengurus',
        'jabatan',
        'nik',
        'tempat_lahir',
        'tanggal_lahir',
        'jenis_kelamin',
        'alamat',
        'nomor_hp',
        'email',
        'pendidikan',
        'pekerjaan',
        'tanggal_mulai_jabatan',
        'tanggal_selesai_jabatan',
        'status_aktif',
        'foto',
        'keterangan',
        'urutan_tampil',
        'user_id'
    ];

    protected $casts = [
        'tanggal_lahir' => 'date',
        'tanggal_mulai_jabatan' => 'date',
        'tanggal_selesai_jabatan' => 'date'
    ];

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    // Scopes
    public function scopeAktif($query)
    {
        return $query->where('status_aktif', 'aktif');
    }

    public function scopeByKategori($query, $kategori)
    {
        return $query->where('kategori_lembaga', $kategori);
    }

    public function scopeByLembaga($query, $lembaga)
    {
        return $query->where('nama_lembaga', $lembaga);
    }

    public function scopeOrderByUrutan($query)
    {
        return $query->orderBy('urutan_tampil', 'asc');
    }

    // Accessors
    public function getUmurAttribute()
    {
        if ($this->tanggal_lahir) {
            return $this->tanggal_lahir->age;
        }
        return null;
    }

    public function getMasaJabatanAttribute()
    {
        $start = $this->tanggal_mulai_jabatan;
        $end = $this->tanggal_selesai_jabatan ?? now();
        
        return $start->diffInYears($end) . ' tahun ' . 
               ($start->diffInMonths($end) % 12) . ' bulan';
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'aktif' => '<span class="badge bg-success">Aktif</span>',
            'tidak_aktif' => '<span class="badge bg-secondary">Tidak Aktif</span>'
        ];

        return $badges[$this->status_aktif] ?? '';
    }

    public function getFotoUrlAttribute()
    {
        if ($this->foto) {
            return asset('storage/organisasi/' . $this->foto);
        }
        return asset('images/default-avatar.png');
    }

    public function getNamaLengkapAttribute()
    {
        return $this->nama_pengurus . ' (' . $this->jabatan . ')';
    }

    // Methods
    public function nonaktifkan($tanggalSelesai = null)
    {
        $this->update([
            'status_aktif' => 'tidak_aktif',
            'tanggal_selesai_jabatan' => $tanggalSelesai ?? now()->toDateString()
        ]);
    }

    public function aktifkan()
    {
        $this->update([
            'status_aktif' => 'aktif',
            'tanggal_selesai_jabatan' => null
        ]);
    }

    public static function getKategoriLembagaOptions()
    {
        return [
            'perangkat_desa' => 'Perangkat Desa',
            'bpd' => 'Badan Permusyawaratan Desa (BPD)',
            'lpm' => 'Lembaga Pemberdayaan Masyarakat (LPM)',
            'karang_taruna' => 'Karang Taruna',
            'pkk' => 'Pemberdayaan Kesejahteraan Keluarga (PKK)',
            'rt_rw' => 'RT/RW',
            'tokoh_agama' => 'Tokoh Agama',
            'tokoh_masyarakat' => 'Tokoh Masyarakat'
        ];
    }

    public static function getPendidikanOptions()
    {
        return [
            'SD' => 'SD/Sederajat',
            'SMP' => 'SMP/Sederajat',
            'SMA' => 'SMA/Sederajat',
            'D1' => 'Diploma 1',
            'D2' => 'Diploma 2',
            'D3' => 'Diploma 3',
            'S1' => 'Sarjana (S1)',
            'S2' => 'Magister (S2)',
            'S3' => 'Doktor (S3)'
        ];
    }

    public static function getJenisKelaminOptions()
    {
        return [
            'L' => 'Laki-laki',
            'P' => 'Perempuan'
        ];
    }

    public static function getStatusAktifOptions()
    {
        return [
            'aktif' => 'Aktif',
            'tidak_aktif' => 'Tidak Aktif'
        ];
    }

    public static function getPerangkatDesaByJabatan()
    {
        return self::byKategori('perangkat_desa')
            ->aktif()
            ->orderByUrutan()
            ->get()
            ->groupBy('jabatan');
    }

    public static function getStatistikOrganisasi()
    {
        return [
            'total_pengurus' => self::aktif()->count(),
            'perangkat_desa' => self::byKategori('perangkat_desa')->aktif()->count(),
            'bpd' => self::byKategori('bpd')->aktif()->count(),
            'lpm' => self::byKategori('lpm')->aktif()->count(),
            'karang_taruna' => self::byKategori('karang_taruna')->aktif()->count(),
            'pkk' => self::byKategori('pkk')->aktif()->count(),
            'rt_rw' => self::byKategori('rt_rw')->aktif()->count(),
            'rata_rata_umur' => self::aktif()->whereNotNull('tanggal_lahir')->get()->avg('umur'),
            'laki_laki' => self::aktif()->where('jenis_kelamin', 'L')->count(),
            'perempuan' => self::aktif()->where('jenis_kelamin', 'P')->count()
        ];
    }
} 