<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class TiketWisata extends Model
{
    use HasFactory;

    protected $table = 'tiket_wisata';

    protected $fillable = [
        'nomor_tiket',
        'destinasi_wisata',
        'nama_pengunjung',
        'email',
        'no_hp',
        'tanggal_kunjungan',
        'jumlah_tiket',
        'harga_per_tiket',
        'total_harga',
        'status_pembayaran',
        'metode_pembayaran',
        'kode_booking',
        'catatan',
        'qr_code',
        'objek_wisata_id',
        'kode_booking',
        'qr_code',
        'nama_pembeli',
        'email_pembeli',
        'nomor_hp_pembeli',
        'alamat_pembeli',
        'jenis_tiket',
        'harga_tiket',
        'jumlah_tiket',
        'total_harga',
        'biaya_admin',
        'total_bayar',
        'tanggal_kunjungan',
        'jam_kunjungan',
        'durasi_kunjungan_jam',
        'catatan_khusus',
        'permintaan_khusus',
        'status_tiket',
        'metode_pembayaran',
        'bukti_pembayaran',
        'tanggal_pembayaran',
        'tanggal_expired',
        'check_in_at',
        'check_out_at',
        'petugas_check_in',
        'petugas_check_out',
        'lokasi_check_in',
        'device_check_in',
        'ip_address_check_in',
        'foto_check_in',
        'catatan_check_in',
        'status_kunjungan',
        'rating_kunjungan',
        'review_kunjungan',
        'foto_review',
        'tanggal_review',
        'is_cancelled',
        'cancelled_at',
        'cancelled_by',
        'cancel_reason',
        'refund_amount',
        'refund_status',
        'refund_at',
        'refund_reference',
        'notification_sent',
        'reminder_sent',
        'whatsapp_sent',
        'email_sent',
        'user_id',
        'created_by'
    ];

    protected $casts = [
        'tanggal_kunjungan' => 'date',
        'harga_per_tiket' => 'decimal:2',
        'total_harga' => 'decimal:2',
        'harga_tiket' => 'decimal:2',
        'biaya_admin' => 'decimal:2',
        'total_bayar' => 'decimal:2',
        'refund_amount' => 'decimal:2',
        'tanggal_kunjungan' => 'date',
        'tanggal_pembayaran' => 'datetime',
        'tanggal_expired' => 'datetime',
        'check_in_at' => 'datetime',
        'check_out_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'tanggal_review' => 'datetime',
        'refund_at' => 'datetime',
        'foto_review' => 'array',
        'is_cancelled' => 'boolean',
        'notification_sent' => 'boolean',
        'reminder_sent' => 'boolean',
        'whatsapp_sent' => 'boolean',
        'email_sent' => 'boolean',
    ];

    // Boot method untuk auto-generate kode booking dan QR code
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($tiket) {
            if (empty($tiket->kode_booking)) {
                $tiket->kode_booking = 'TKT-' . date('Ymd') . '-' . strtoupper(Str::random(6));
                
                // Ensure unique booking code
                while (static::where('kode_booking', $tiket->kode_booking)->exists()) {
                    $tiket->kode_booking = 'TKT-' . date('Ymd') . '-' . strtoupper(Str::random(6));
                }
            }
            
            // Generate QR Code
            if (empty($tiket->qr_code)) {
                $tiket->qr_code = $tiket->generateQrCode();
            }
            
            // Set expired time (24 hours for unpaid tickets)
            if (empty($tiket->tanggal_expired) && $tiket->status_tiket === 'pending_payment') {
                $tiket->tanggal_expired = now()->addHours(24);
            }
        });
    }

    // Relationships
    public function objekWisata(): BelongsTo
    {
        return $this->belongsTo(ObjekWisata::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function petugasCheckIn(): BelongsTo
    {
        return $this->belongsTo(User::class, 'petugas_check_in');
    }

    public function petugasCheckOut(): BelongsTo
    {
        return $this->belongsTo(User::class, 'petugas_check_out');
    }

    public function cancelledBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'cancelled_by');
    }

    public function transaksi()
    {
        return $this->morphMany(Transaksi::class, 'transactionable');
    }

    // Scopes
    public function scopeLunas($query)
    {
        return $query->where('status_tiket', 'paid');
    }

    public function scopePending($query)
    {
        return $query->where('status_tiket', 'pending_payment');
    }

    public function scopeExpired($query)
    {
        return $query->where('tanggal_expired', '<', now())
                    ->where('status_tiket', 'pending_payment');
    }

    public function scopeActive($query)
    {
        return $query->where('status_tiket', 'paid')
                    ->where('is_cancelled', false)
                    ->where('tanggal_kunjungan', '>=', now()->toDateString());
    }

    public function scopeUsed($query)
    {
        return $query->whereNotNull('check_in_at');
    }

    public function scopeUnused($query)
    {
        return $query->whereNull('check_in_at')
                    ->where('status_tiket', 'paid')
                    ->where('is_cancelled', false);
    }

    public function scopeByJenisTiket($query, $jenis)
    {
        return $query->where('jenis_tiket', $jenis);
    }

    public function scopeByTanggalKunjungan($query, $tanggal)
    {
        return $query->where('tanggal_kunjungan', $tanggal);
    }

    public function scopeToday($query)
    {
        return $query->where('tanggal_kunjungan', now()->toDateString());
    }

    // Accessors & Mutators
    public function getStatusPembayaranBadgeAttribute()
    {
        $badges = [
            'pending_payment' => '<span class="badge bg-warning">Pending</span>',
            'paid' => '<span class="badge bg-success">Lunas</span>',
            'expired' => '<span class="badge bg-danger">Expired</span>',
            'cancelled' => '<span class="badge bg-secondary">Cancelled</span>',
            'refunded' => '<span class="badge bg-info">Refunded</span>',
        ];
        
        return $badges[$this->status_tiket] ?? '<span class="badge bg-secondary">Unknown</span>';
    }

    public function getStatusKunjunganBadgeAttribute()
    {
        $badges = [
            'belum_datang' => '<span class="badge bg-secondary">Belum Datang</span>',
            'sudah_checkin' => '<span class="badge bg-primary">Sudah Check-in</span>',
            'sedang_berkunjung' => '<span class="badge bg-info">Sedang Berkunjung</span>',
            'sudah_checkout' => '<span class="badge bg-success">Sudah Check-out</span>',
            'tidak_datang' => '<span class="badge bg-danger">Tidak Datang</span>',
        ];
        
        return $badges[$this->status_kunjungan] ?? '<span class="badge bg-secondary">Unknown</span>';
    }

    public function getJenisTiketLabelAttribute()
    {
        $labels = [
            'dewasa' => 'Tiket Dewasa',
            'anak' => 'Tiket Anak-anak',
            'wisatawan_asing' => 'Tiket Wisatawan Asing',
            'rombongan' => 'Tiket Rombongan',
            'pelajar' => 'Tiket Pelajar',
            'lansia' => 'Tiket Lansia',
            'disabilitas' => 'Tiket Disabilitas',
            'vip' => 'Tiket VIP',
            'paket_keluarga' => 'Paket Keluarga',
            'weekend' => 'Tiket Weekend',
            'weekday' => 'Tiket Weekday',
            'annual_pass' => 'Annual Pass',
        ];
        
        return $labels[$this->jenis_tiket] ?? 'Unknown';
    }

    public function getQrCodeUrlAttribute()
    {
        if ($this->qr_code) {
            return asset('storage/qrcodes/' . $this->qr_code);
        }
        return null;
    }

    public function getBuktiPembayaranUrlAttribute()
    {
        if ($this->bukti_pembayaran) {
            return asset('storage/bukti_bayar/' . $this->bukti_pembayaran);
        }
        return null;
    }

    public function getFotoCheckInUrlAttribute()
    {
        if ($this->foto_check_in) {
            return asset('storage/checkin/' . $this->foto_check_in);
        }
        return null;
    }

    public function getFotoReviewUrlsAttribute()
    {
        if ($this->foto_review) {
            return collect($this->foto_review)->map(function ($foto) {
                return asset('storage/review/' . $foto);
            })->toArray();
        }
        return [];
    }

    public function getDurasiKunjunganAttribute()
    {
        if ($this->check_in_at && $this->check_out_at) {
            return $this->check_in_at->diffInMinutes($this->check_out_at);
        }
        return null;
    }

    public function getIsExpiredAttribute()
    {
        return $this->tanggal_expired && now()->gt($this->tanggal_expired) && $this->status_tiket === 'pending_payment';
    }

    public function getIsUsableAttribute()
    {
        return $this->status_tiket === 'paid' 
               && !$this->is_cancelled 
               && !$this->check_in_at
               && $this->tanggal_kunjungan >= now()->toDateString();
    }

    public function getCanCheckInAttribute()
    {
        if (!$this->is_usable) {
            return false;
        }
        
        // Can check in on the visit date
        return $this->tanggal_kunjungan === now()->toDateString();
    }

    public function getCanCheckOutAttribute()
    {
        return $this->check_in_at && !$this->check_out_at;
    }

    // Methods
    public function generateQrCode()
    {
        $qrData = [
            'kode_booking' => $this->kode_booking,
            'objek_wisata_id' => $this->objek_wisata_id,
            'tanggal_kunjungan' => $this->tanggal_kunjungan,
            'jumlah_tiket' => $this->jumlah_tiket,
            'timestamp' => now()->timestamp
        ];
        
        $qrString = json_encode($qrData);
        $filename = 'qr_' . $this->kode_booking . '.png';
        
        // Generate QR Code image
        QrCode::format('png')
              ->size(300)
              ->margin(2)
              ->generate($qrString, storage_path('app/public/qrcodes/' . $filename));
        
        return $filename;
    }

    public function checkIn($petugasId = null, $lokasi = null, $foto = null, $catatan = null)
    {
        if (!$this->can_check_in) {
            return false;
        }
        
        $this->update([
            'check_in_at' => now(),
            'petugas_check_in' => $petugasId,
            'lokasi_check_in' => $lokasi,
            'device_check_in' => request()->header('User-Agent'),
            'ip_address_check_in' => request()->ip(),
            'foto_check_in' => $foto,
            'catatan_check_in' => $catatan,
            'status_kunjungan' => 'sudah_checkin'
        ]);
        
        // Update visitor count for tourism object
        $this->objekWisata->incrementPengunjung($this->jumlah_tiket);
        
        return true;
    }

    public function checkOut($petugasId = null, $catatan = null)
    {
        if (!$this->can_check_out) {
            return false;
        }
        
        $this->update([
            'check_out_at' => now(),
            'petugas_check_out' => $petugasId,
            'status_kunjungan' => 'sudah_checkout'
        ]);
        
        return true;
    }

    public function cancel($reason = null, $userId = null)
    {
        if ($this->is_cancelled || $this->check_in_at) {
            return false;
        }
        
        $this->update([
            'is_cancelled' => true,
            'cancelled_at' => now(),
            'cancelled_by' => $userId,
            'cancel_reason' => $reason,
            'status_tiket' => 'cancelled'
        ]);
        
        return true;
    }

    public function refund($amount = null, $reference = null)
    {
        if (!$this->is_cancelled && !$this->isExpired) {
            return false;
        }
        
        $refundAmount = $amount ?? $this->total_bayar;
        
        $this->update([
            'refund_amount' => $refundAmount,
            'refund_status' => 'processed',
            'refund_at' => now(),
            'refund_reference' => $reference,
            'status_tiket' => 'refunded'
        ]);
        
        return true;
    }

    public function addReview($rating, $review, $photos = [])
    {
        if (!$this->check_out_at) {
            return false;
        }
        
        $this->update([
            'rating_kunjungan' => $rating,
            'review_kunjungan' => $review,
            'foto_review' => $photos,
            'tanggal_review' => now()
        ]);
        
        // Update tourism object rating
        $this->objekWisata->updateRating($rating);
        
        return true;
    }

    public function sendNotification($type = 'booking_confirmation')
    {
        // Implementation for sending notifications
        // This would integrate with your notification system
        
        switch ($type) {
            case 'booking_confirmation':
                // Send booking confirmation
                break;
            case 'payment_reminder':
                // Send payment reminder
                break;
            case 'visit_reminder':
                // Send visit reminder
                break;
            case 'check_in_success':
                // Send check-in confirmation
                break;
        }
        
        $this->update(['notification_sent' => true]);
    }

    public function sendWhatsAppNotification($message)
    {
        // Implementation for WhatsApp notification
        // This would integrate with your WhatsApp API
        
        $this->update(['whatsapp_sent' => true]);
    }

    public function sendEmailNotification($subject, $message)
    {
        // Implementation for email notification
        // This would integrate with your email system
        
        $this->update(['email_sent' => true]);
    }

    public function markAsExpired()
    {
        if ($this->status_tiket === 'pending_payment' && now()->gt($this->tanggal_expired)) {
            $this->update(['status_tiket' => 'expired']);
            return true;
        }
        return false;
    }

    public function extendExpiry($hours = 24)
    {
        if ($this->status_tiket === 'pending_payment') {
            $this->update(['tanggal_expired' => now()->addHours($hours)]);
            return true;
        }
        return false;
    }

    public function getTotalRevenue()
    {
        return static::where('status_tiket', 'paid')->sum('total_bayar');
    }

    public static function getStatistikHarian($tanggal = null)
    {
        $tanggal = $tanggal ?? now()->toDateString();
        
        return [
            'total_tiket' => static::where('tanggal_kunjungan', $tanggal)->sum('jumlah_tiket'),
            'total_pendapatan' => static::where('tanggal_kunjungan', $tanggal)
                                       ->where('status_tiket', 'paid')
                                       ->sum('total_bayar'),
            'total_pengunjung' => static::where('tanggal_kunjungan', $tanggal)
                                       ->whereNotNull('check_in_at')
                                       ->sum('jumlah_tiket'),
            'tingkat_kunjungan' => static::where('tanggal_kunjungan', $tanggal)
                                        ->whereNotNull('check_in_at')
                                        ->count() / max(static::where('tanggal_kunjungan', $tanggal)
                                        ->where('status_tiket', 'paid')
                                        ->count(), 1) * 100
        ];
    }

    // Generate nomor tiket otomatis
    public static function generateNomorTiket()
    {
        $lastTicket = self::whereDate('created_at', today())->count();
        return 'TWS-' . date('Ymd') . '-' . str_pad($lastTicket + 1, 4, '0', STR_PAD_LEFT);
    }

    // Generate kode booking
    public static function generateKodeBooking()
    {
        return 'BKG-' . strtoupper(uniqid());
    }
}
