<?php

namespace App\Services;

use App\Models\Bug;
use App\Http\Controllers\BugController;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;

class BugDetectionService
{
    /**
     * Auto-capture errors from any controller
     */
    public static function captureFromController(\Throwable $exception, Request $request = null, $controllerName = null)
    {
        try {
            // Extract module name from controller
            $module = $controllerName ? class_basename($controllerName) : 'Unknown';
            
            // Use BugController's captureError method
            return BugController::captureError($exception, $request, $module);
            
        } catch (\Exception $e) {
            Log::error('Error in BugDetectionService::captureFromController', [
                'error' => $e->getMessage(),
                'original_exception' => $exception->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Monitor system health and detect potential issues
     */
    public static function healthCheck()
    {
        $issues = [];
        
        try {
            // Check database connection
            try {
                \DB::connection()->getPdo();
            } catch (\Exception $e) {
                $issues[] = [
                    'type' => 'critical',
                    'module' => 'Database',
                    'message' => 'Database connection failed: ' . $e->getMessage()
                ];
            }

            // Check storage permissions
            if (!is_writable(storage_path())) {
                $issues[] = [
                    'type' => 'high',
                    'module' => 'Storage',
                    'message' => 'Storage directory is not writable'
                ];
            }

            // Check memory usage
            $memoryUsage = memory_get_usage(true);
            $memoryLimit = ini_get('memory_limit');
            $memoryLimitBytes = self::convertToBytes($memoryLimit);
            
            if ($memoryUsage > ($memoryLimitBytes * 0.8)) {
                $issues[] = [
                    'type' => 'warning',
                    'module' => 'System',
                    'message' => 'High memory usage: ' . round($memoryUsage / 1024 / 1024, 2) . 'MB'
                ];
            }

            // Check for recent critical bugs
            $criticalBugs = Bug::where('severity', 'critical')
                              ->where('status', 'open')
                              ->where('created_at', '>=', now()->subHour())
                              ->count();

            if ($criticalBugs > 5) {
                $issues[] = [
                    'type' => 'critical',
                    'module' => 'BugSystem',
                    'message' => "High number of critical bugs in last hour: {$criticalBugs}"
                ];
            }

            // Create bug reports for detected issues
            foreach ($issues as $issue) {
                self::createHealthCheckBug($issue);
            }

            return [
                'status' => empty($issues) ? 'healthy' : 'issues_detected',
                'issues' => $issues,
                'timestamp' => now()
            ];

        } catch (\Exception $e) {
            Log::error('Error in health check', ['error' => $e->getMessage()]);
            return [
                'status' => 'error',
                'message' => 'Health check failed: ' . $e->getMessage(),
                'timestamp' => now()
            ];
        }
    }

    /**
     * Monitor specific modules for errors
     */
    public static function monitorModule($moduleName, $callback)
    {
        try {
            return $callback();
        } catch (\Throwable $exception) {
            // Auto-capture the error
            Bug::createFromException($exception, request(), $moduleName);
            
            // Re-throw the exception so normal error handling continues
            throw $exception;
        }
    }

    /**
     * Analyze error patterns and detect recurring issues
     */
    public static function analyzeErrorPatterns()
    {
        try {
            $patterns = [];

            // Find frequently occurring errors
            $frequentErrors = Bug::selectRaw('title, module, COUNT(*) as occurrence_count')
                               ->where('created_at', '>=', now()->subWeek())
                               ->groupBy('title', 'module')
                               ->having('occurrence_count', '>', 10)
                               ->orderByDesc('occurrence_count')
                               ->get();

            foreach ($frequentErrors as $error) {
                $patterns[] = [
                    'type' => 'frequent_error',
                    'title' => $error->title,
                    'module' => $error->module,
                    'count' => $error->occurrence_count,
                    'severity' => 'medium'
                ];
            }

            // Find error spikes (sudden increase in errors)
            $today = Bug::whereDate('created_at', today())->count();
            $yesterday = Bug::whereDate('created_at', today()->subDay())->count();
            
            if ($today > ($yesterday * 2) && $today > 50) {
                $patterns[] = [
                    'type' => 'error_spike',
                    'message' => "Error spike detected: {$today} errors today vs {$yesterday} yesterday",
                    'severity' => 'high'
                ];
            }

            // Find modules with high error rates
            $problemModules = Bug::selectRaw('module, COUNT(*) as error_count')
                               ->where('created_at', '>=', now()->subDay())
                               ->whereNotNull('module')
                               ->groupBy('module')
                               ->having('error_count', '>', 20)
                               ->orderByDesc('error_count')
                               ->get();

            foreach ($problemModules as $module) {
                $patterns[] = [
                    'type' => 'problem_module',
                    'module' => $module->module,
                    'error_count' => $module->error_count,
                    'severity' => 'medium'
                ];
            }

            return $patterns;

        } catch (\Exception $e) {
            Log::error('Error in pattern analysis', ['error' => $e->getMessage()]);
            return [];
        }
    }

    /**
     * Generate system health report
     */
    public static function generateHealthReport()
    {
        try {
            $report = [
                'timestamp' => now(),
                'system_status' => self::healthCheck(),
                'error_patterns' => self::analyzeErrorPatterns(),
                'statistics' => Bug::getStatistics('week'),
                'critical_bugs' => Bug::critical()->open()->count(),
                'total_bugs_today' => Bug::whereDate('created_at', today())->count(),
                'resolution_rate' => self::calculateResolutionRate(),
                'top_error_modules' => self::getTopErrorModules()
            ];

            // Store report in logs
            Log::info('System Health Report Generated', $report);

            return $report;

        } catch (\Exception $e) {
            Log::error('Error generating health report', ['error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Setup error handlers for all controllers
     */
    public static function setupGlobalErrorCapture()
    {
        // This would be called in a service provider or middleware
        set_exception_handler(function(\Throwable $exception) {
            // Capture the error
            self::captureFromController($exception, request());
            
            // Let Laravel handle the exception normally
            if (app()->bound('exception.handler')) {
                app('exception.handler')->report($exception);
            }
        });
    }

    /**
     * Create bug report from health check issue
     */
    private static function createHealthCheckBug($issue)
    {
        try {
            // Check if similar health issue already exists
            $existingBug = Bug::where('module', $issue['module'])
                             ->where('type', 'warning')
                             ->where('status', 'open')
                             ->where('title', 'like', '%' . $issue['module'] . ' Health Issue%')
                             ->first();

            if ($existingBug) {
                $existingBug->incrementOccurrence();
                return $existingBug;
            }

            return Bug::create([
                'title' => $issue['module'] . ' Health Issue',
                'description' => $issue['message'],
                'severity' => $issue['type'] === 'critical' ? 'critical' : 
                            ($issue['type'] === 'high' ? 'high' : 'medium'),
                'type' => 'warning',
                'module' => $issue['module'],
                'environment_data' => [
                    'health_check' => true,
                    'detected_at' => now(),
                    'system_info' => self::getSystemInfo()
                ],
                'last_occurred_at' => now()
            ]);

        } catch (\Exception $e) {
            Log::error('Error creating health check bug', [
                'error' => $e->getMessage(),
                'issue' => $issue
            ]);
        }
    }

    /**
     * Convert memory limit string to bytes
     */
    private static function convertToBytes($value)
    {
        $value = trim($value);
        $last = strtolower($value[strlen($value)-1]);
        $value = (int) $value;

        switch($last) {
            case 'g':
                $value *= 1024;
            case 'm':
                $value *= 1024;
            case 'k':
                $value *= 1024;
        }

        return $value;
    }

    /**
     * Calculate bug resolution rate
     */
    private static function calculateResolutionRate()
    {
        $totalBugs = Bug::where('created_at', '>=', now()->subWeek())->count();
        $resolvedBugs = Bug::where('created_at', '>=', now()->subWeek())
                          ->whereNotNull('resolved_at')
                          ->count();

        return $totalBugs > 0 ? round(($resolvedBugs / $totalBugs) * 100, 2) : 0;
    }

    /**
     * Get top error modules
     */
    private static function getTopErrorModules()
    {
        return Bug::selectRaw('module, COUNT(*) as error_count')
                 ->where('created_at', '>=', now()->subWeek())
                 ->whereNotNull('module')
                 ->groupBy('module')
                 ->orderByDesc('error_count')
                 ->limit(5)
                 ->get();
    }

    /**
     * Get system information
     */
    private static function getSystemInfo()
    {
        return [
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
            'memory_usage' => memory_get_usage(true),
            'memory_peak' => memory_get_peak_usage(true),
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
            'timestamp' => now()
        ];
    }
} 