<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Config;

class NotificationService
{
    private $whatsappApiUrl;
    private $whatsappToken;
    private $whatsappFrom;

    public function __construct()
    {
        $this->whatsappApiUrl = config('whatsapp.api_url');
        $this->whatsappToken = config('whatsapp.token');
        $this->whatsappFrom = config('whatsapp.from_number');
    }

    /**
     * Send WhatsApp message
     */
    public function sendWhatsApp($to, $message, $type = 'text', $media = null)
    {
        try {
            // Remove leading zero and add country code if needed
            $phoneNumber = $this->formatPhoneNumber($to);
            
            $payload = [
                'to' => $phoneNumber,
                'type' => $type,
                'message' => $message
            ];

            if ($media && in_array($type, ['image', 'document', 'video'])) {
                $payload['media'] = $media;
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->whatsappToken,
                'Content-Type' => 'application/json'
            ])->post($this->whatsappApiUrl . '/send-message', $payload);

            if ($response->successful()) {
                Log::info('WhatsApp message sent successfully', [
                    'to' => $phoneNumber,
                    'message' => substr($message, 0, 100) . '...'
                ]);
                
                return [
                    'success' => true,
                    'message_id' => $response->json('message_id'),
                    'status' => 'sent'
                ];
            }

            Log::error('WhatsApp message failed', [
                'to' => $phoneNumber,
                'error' => $response->body()
            ]);

            return [
                'success' => false,
                'error' => $response->json('error') ?? 'Failed to send WhatsApp message'
            ];

        } catch (\Exception $e) {
            Log::error('WhatsApp service error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Send Email notification
     */
    public function sendEmail($to, $subject, $message, $template = null, $data = [])
    {
        try {
            if ($template) {
                Mail::send($template, $data, function ($mail) use ($to, $subject) {
                    $mail->to($to)
                         ->subject($subject)
                         ->from(config('mail.from.address'), config('mail.from.name'));
                });
            } else {
                Mail::raw($message, function ($mail) use ($to, $subject) {
                    $mail->to($to)
                         ->subject($subject)
                         ->from(config('mail.from.address'), config('mail.from.name'));
                });
            }

            Log::info('Email sent successfully', [
                'to' => $to,
                'subject' => $subject
            ]);

            return [
                'success' => true,
                'status' => 'sent'
            ];

        } catch (\Exception $e) {
            Log::error('Email service error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Send notification to multiple channels
     */
    public function sendMultiChannel($recipients, $subject, $message, $channels = ['email', 'whatsapp'])
    {
        $results = [];

        foreach ($recipients as $recipient) {
            $recipientResults = [];

            if (in_array('email', $channels) && isset($recipient['email'])) {
                $emailResult = $this->sendEmail($recipient['email'], $subject, $message);
                $recipientResults['email'] = $emailResult;
            }

            if (in_array('whatsapp', $channels) && isset($recipient['phone'])) {
                $whatsappResult = $this->sendWhatsApp($recipient['phone'], $message);
                $recipientResults['whatsapp'] = $whatsappResult;
            }

            $results[] = [
                'recipient' => $recipient,
                'results' => $recipientResults
            ];
        }

        return $results;
    }

    /**
     * Send booking confirmation notification
     */
    public function sendBookingConfirmation($booking)
    {
        $message = "Halo {$booking['customer_name']},\n\n";
        $message .= "Booking tiket wisata Anda telah dikonfirmasi!\n\n";
        $message .= "Detail Booking:\n";
        $message .= "Kode Booking: {$booking['booking_code']}\n";
        $message .= "Objek Wisata: {$booking['destination']}\n";
        $message .= "Tanggal Kunjungan: {$booking['visit_date']}\n";
        $message .= "Jumlah Tiket: {$booking['ticket_count']}\n";
        $message .= "Total Bayar: Rp " . number_format($booking['total_amount']) . "\n\n";
        $message .= "Silakan tunjukkan kode booking ini saat check-in.\n\n";
        $message .= "Terima kasih!";

        $subject = "Konfirmasi Booking Tiket Wisata - {$booking['booking_code']}";

        $recipients = [
            [
                'email' => $booking['customer_email'],
                'phone' => $booking['customer_phone']
            ]
        ];

        return $this->sendMultiChannel($recipients, $subject, $message);
    }

    /**
     * Send payment reminder
     */
    public function sendPaymentReminder($booking)
    {
        $message = "Halo {$booking['customer_name']},\n\n";
        $message .= "Pembayaran untuk booking {$booking['booking_code']} belum diterima.\n\n";
        $message .= "Detail Booking:\n";
        $message .= "Objek Wisata: {$booking['destination']}\n";
        $message .= "Tanggal Kunjungan: {$booking['visit_date']}\n";
        $message .= "Total Bayar: Rp " . number_format($booking['total_amount']) . "\n";
        $message .= "Batas Waktu: {$booking['payment_deadline']}\n\n";
        $message .= "Silakan lakukan pembayaran sebelum batas waktu.\n\n";
        $message .= "Link Pembayaran: {$booking['payment_url']}";

        $subject = "Reminder Pembayaran - {$booking['booking_code']}";

        $recipients = [
            [
                'email' => $booking['customer_email'],
                'phone' => $booking['customer_phone']
            ]
        ];

        return $this->sendMultiChannel($recipients, $subject, $message);
    }

    /**
     * Send UMKM product approval notification
     */
    public function sendProductApproval($product, $approved = true)
    {
        $status = $approved ? 'disetujui' : 'ditolak';
        $message = "Halo {$product['owner_name']},\n\n";
        $message .= "Produk '{$product['product_name']}' telah {$status}.\n\n";
        
        if ($approved) {
            $message .= "Produk Anda kini dapat dilihat di marketplace desa.\n";
            $message .= "Link Produk: {$product['product_url']}\n\n";
        } else {
            $message .= "Alasan: {$product['rejection_reason']}\n\n";
            $message .= "Silakan perbaiki dan ajukan kembali.\n\n";
        }
        
        $message .= "Terima kasih atas partisipasi Anda dalam mengembangkan UMKM desa.";

        $subject = "Status Produk UMKM - {$product['product_name']}";

        $recipients = [
            [
                'email' => $product['owner_email'],
                'phone' => $product['owner_phone']
            ]
        ];

        return $this->sendMultiChannel($recipients, $subject, $message);
    }

    /**
     * Send order notification
     */
    public function sendOrderNotification($order)
    {
        // To customer
        $customerMessage = "Halo {$order['customer_name']},\n\n";
        $customerMessage .= "Pesanan Anda telah diterima!\n\n";
        $customerMessage .= "No. Pesanan: {$order['order_number']}\n";
        $customerMessage .= "Total: Rp " . number_format($order['total_amount']) . "\n";
        $customerMessage .= "Status: {$order['status']}\n\n";
        $customerMessage .= "Kami akan segera memproses pesanan Anda.\n\n";
        $customerMessage .= "Terima kasih!";

        // To seller
        $sellerMessage = "Halo {$order['seller_name']},\n\n";
        $sellerMessage .= "Anda mendapat pesanan baru!\n\n";
        $sellerMessage .= "No. Pesanan: {$order['order_number']}\n";
        $sellerMessage .= "Pembeli: {$order['customer_name']}\n";
        $sellerMessage .= "Produk: {$order['product_name']}\n";
        $sellerMessage .= "Jumlah: {$order['quantity']}\n";
        $sellerMessage .= "Total: Rp " . number_format($order['total_amount']) . "\n\n";
        $sellerMessage .= "Silakan login ke admin panel untuk memproses pesanan.";

        $results = [];

        // Send to customer
        $customerRecipients = [
            [
                'email' => $order['customer_email'],
                'phone' => $order['customer_phone']
            ]
        ];
        $results['customer'] = $this->sendMultiChannel(
            $customerRecipients, 
            "Konfirmasi Pesanan - {$order['order_number']}", 
            $customerMessage
        );

        // Send to seller
        $sellerRecipients = [
            [
                'email' => $order['seller_email'],
                'phone' => $order['seller_phone']
            ]
        ];
        $results['seller'] = $this->sendMultiChannel(
            $sellerRecipients, 
            "Pesanan Baru - {$order['order_number']}", 
            $sellerMessage
        );

        return $results;
    }

    /**
     * Send complaint notification
     */
    public function sendComplaintNotification($complaint)
    {
        $message = "Pengaduan baru diterima:\n\n";
        $message .= "Dari: {$complaint['sender_name']}\n";
        $message .= "Kategori: {$complaint['category']}\n";
        $message .= "Subjek: {$complaint['subject']}\n";
        $message .= "Isi: {$complaint['message']}\n\n";
        $message .= "Harap segera ditindaklanjuti melalui admin panel.";

        $subject = "Pengaduan Baru - {$complaint['category']}";

        // Send to admin
        $adminRecipients = [
            [
                'email' => config('app.admin_email'),
                'phone' => config('app.admin_phone')
            ]
        ];

        return $this->sendMultiChannel($adminRecipients, $subject, $message);
    }

    /**
     * Send bulk notification
     */
    public function sendBulkNotification($recipients, $subject, $message, $channels = ['whatsapp'])
    {
        $results = [];
        $batchSize = 50; // Process in batches to avoid overwhelming the service

        $batches = array_chunk($recipients, $batchSize);

        foreach ($batches as $batch) {
            $batchResults = $this->sendMultiChannel($batch, $subject, $message, $channels);
            $results = array_merge($results, $batchResults);

            // Add small delay between batches
            sleep(1);
        }

        return $results;
    }

    /**
     * Format phone number for WhatsApp
     */
    private function formatPhoneNumber($phone)
    {
        // Remove all non-numeric characters
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Remove leading zero and add country code (62 for Indonesia)
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1);
        } elseif (substr($phone, 0, 2) !== '62') {
            $phone = '62' . $phone;
        }

        return $phone;
    }

    /**
     * Get notification status
     */
    public function getNotificationStatus($messageId, $type = 'whatsapp')
    {
        try {
            if ($type === 'whatsapp') {
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $this->whatsappToken
                ])->get($this->whatsappApiUrl . '/message-status/' . $messageId);

                if ($response->successful()) {
                    return $response->json();
                }
            }

            return ['success' => false, 'error' => 'Failed to get status'];

        } catch (\Exception $e) {
            Log::error('Get notification status error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Create notification template
     */
    public function createTemplate($name, $content, $type = 'text')
    {
        // Store template in config or database
        $templates = config('notification.templates', []);
        $templates[$name] = [
            'content' => $content,
            'type' => $type,
            'created_at' => now()
        ];

        Config::set('notification.templates', $templates);

        return true;
    }

    /**
     * Use notification template
     */
    public function useTemplate($templateName, $variables = [])
    {
        $templates = config('notification.templates', []);
        
        if (!isset($templates[$templateName])) {
            throw new \Exception("Template '{$templateName}' not found");
        }

        $content = $templates[$templateName]['content'];

        // Replace variables in template
        foreach ($variables as $key => $value) {
            $content = str_replace('{' . $key . '}', $value, $content);
        }

        return $content;
    }
} 