<?php

namespace App\Services;

use App\Models\Setting;
use App\Services\TripayService;
use App\Services\OttopayService;

class PaymentGatewayService
{
    protected $setting;

    public function __construct()
    {
        $this->setting = Setting::first();
    }

    /**
     * Get available payment gateways
     */
    public function getAvailableGateways()
    {
        $gateways = [];

        // Check Tripay configuration
        if ($this->isTripayConfigured()) {
            $gateways['tripay'] = [
                'name' => 'Tripay',
                'logo' => '/images/tripay-logo.png',
                'channels' => $this->setting->tripay_channels ?? [],
                'fee_customer' => $this->setting->tripay_fee_customer ?? false
            ];
        }

        // Check OttoPay configuration
        if ($this->isOttopayConfigured()) {
            $gateways['ottopay'] = [
                'name' => 'OttoPay',
                'logo' => '/images/ottopay-logo.png',
                'channels' => $this->setting->ottopay_channels ?? [],
                'fee_customer' => $this->setting->ottopay_fee_customer ?? false
            ];
        }

        return $gateways;
    }

    /**
     * Get primary payment gateway
     */
    public function getPrimaryGateway()
    {
        $gateways = $this->getAvailableGateways();
        
        // Priority: Tripay first, then OttoPay
        if (isset($gateways['tripay'])) {
            return 'tripay';
        }
        
        if (isset($gateways['ottopay'])) {
            return 'ottopay';
        }

        return null;
    }

    /**
     * Get payment service instance
     */
    public function getPaymentService($gateway = null)
    {
        if (!$gateway) {
            $gateway = $this->getPrimaryGateway();
        }

        switch ($gateway) {
            case 'tripay':
                return new TripayService();
            case 'ottopay':
                return new OttopayService();
            default:
                throw new \Exception('Payment gateway not configured or not supported');
        }
    }

    /**
     * Create payment transaction
     */
    public function createTransaction($data, $gateway = null)
    {
        $service = $this->getPaymentService($gateway);
        return $service->createTransaction($data);
    }

    /**
     * Get transaction status
     */
    public function getTransactionStatus($transactionId, $gateway = null)
    {
        $service = $this->getPaymentService($gateway);
        return $service->getTransactionStatus($transactionId);
    }

    /**
     * Get payment methods for specific gateway
     */
    public function getPaymentMethods($gateway = null)
    {
        $service = $this->getPaymentService($gateway);
        return $service->getPaymentMethods();
    }

    /**
     * Check if Tripay is configured
     */
    private function isTripayConfigured()
    {
        return !empty($this->setting->tripay_api_key) && 
               !empty($this->setting->tripay_private_key) && 
               !empty($this->setting->tripay_merchant_code);
    }

    /**
     * Check if OttoPay is configured
     */
    private function isOttopayConfigured()
    {
        return !empty($this->setting->ottopay_api_key) && 
               !empty($this->setting->ottopay_secret_key) && 
               !empty($this->setting->ottopay_merchant_code);
    }

    /**
     * Get gateway configuration
     */
    public function getGatewayConfig($gateway)
    {
        switch ($gateway) {
            case 'tripay':
                return [
                    'api_key' => $this->setting->tripay_api_key,
                    'private_key' => $this->setting->tripay_private_key,
                    'merchant_code' => $this->setting->tripay_merchant_code,
                    'mode' => $this->setting->tripay_mode ?? 'sandbox',
                    'expiry_time' => $this->setting->tripay_expiry_time ?? 24,
                    'fee_customer' => $this->setting->tripay_fee_customer ?? false,
                    'channels' => $this->setting->tripay_channels ?? []
                ];
            case 'ottopay':
                return [
                    'api_key' => $this->setting->ottopay_api_key,
                    'secret_key' => $this->setting->ottopay_secret_key,
                    'merchant_code' => $this->setting->ottopay_merchant_code,
                    'mode' => $this->setting->ottopay_mode ?? 'sandbox',
                    'expiry_time' => $this->setting->ottopay_expiry_time ?? 24,
                    'fee_customer' => $this->setting->ottopay_fee_customer ?? false,
                    'channels' => $this->setting->ottopay_channels ?? []
                ];
            default:
                throw new \Exception('Unsupported payment gateway: ' . $gateway);
        }
    }
} 