<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

class PpobService
{
    private $ottopayService;
    private $baseUrl;
    private $apiKey;

    public function __construct(OttopayService $ottopayService)
    {
        $this->ottopayService = $ottopayService;
        $this->baseUrl = 'https://api.ottopay.id/ppob'; // PPOB endpoint
        $this->apiKey = config('ottopay.api_key');
    }

    /**
     * Get available PPOB products
     */
    public function getProducts()
    {
        return [
            'pulsa' => [
                'name' => 'Pulsa',
                'icon' => 'fas fa-mobile-alt',
                'color' => 'bg-blue-500',
                'providers' => [
                    'TELKOMSEL' => 'Telkomsel',
                    'INDOSAT' => 'Indosat',
                    'XL' => 'XL Axiata',
                    'TRI' => '3 (Tri)',
                    'SMARTFREN' => 'Smartfren',
                    'AXIS' => 'Axis'
                ]
            ],
            'paket_data' => [
                'name' => 'Paket Data',
                'icon' => 'fas fa-wifi',
                'color' => 'bg-green-500',
                'providers' => [
                    'TELKOMSEL' => 'Telkomsel',
                    'INDOSAT' => 'Indosat',
                    'XL' => 'XL Axiata',
                    'TRI' => '3 (Tri)',
                    'SMARTFREN' => 'Smartfren',
                    'AXIS' => 'Axis'
                ]
            ],
            'pln' => [
                'name' => 'PLN',
                'icon' => 'fas fa-bolt',
                'color' => 'bg-yellow-500',
                'types' => [
                    'prabayar' => 'Token Listrik',
                    'pascabayar' => 'Tagihan Listrik'
                ]
            ],
            'pbb' => [
                'name' => 'PBB',
                'icon' => 'fas fa-home',
                'color' => 'bg-cyan-500',
                'description' => 'Pajak Bumi dan Bangunan'
            ],
            'bpjs' => [
                'name' => 'BPJS',
                'icon' => 'fas fa-heartbeat',
                'color' => 'bg-red-500',
                'types' => [
                    'kesehatan' => 'BPJS Kesehatan',
                    'ketenagakerjaan' => 'BPJS Ketenagakerjaan'
                ]
            ],
            'tv_kabel' => [
                'name' => 'TV Kabel',
                'icon' => 'fas fa-tv',
                'color' => 'bg-purple-500',
                'providers' => [
                    'INDIHOME' => 'IndiHome',
                    'BIZNET' => 'Biznet',
                    'FIRST_MEDIA' => 'First Media',
                    'MNC_PLAY' => 'MNC Play'
                ]
            ],
            'internet' => [
                'name' => 'Internet',
                'icon' => 'fas fa-globe',
                'color' => 'bg-indigo-500',
                'providers' => [
                    'INDIHOME' => 'IndiHome',
                    'BIZNET' => 'Biznet',
                    'FIRST_MEDIA' => 'First Media',
                    'MNC_PLAY' => 'MNC Play'
                ]
            ],
            'finance' => [
                'name' => 'Multifinance',
                'icon' => 'fas fa-credit-card',
                'color' => 'bg-orange-500',
                'providers' => [
                    'ADIRA' => 'Adira Finance',
                    'BAF' => 'BAF',
                    'MEGA_FINANCE' => 'Mega Finance',
                    'WOM_FINANCE' => 'WOM Finance'
                ]
            ]
        ];
    }

    /**
     * Get pulsa denominations
     */
    public function getPulsaDenominations($provider)
    {
        $denominations = [
            'TELKOMSEL' => [
                ['amount' => 5000, 'price' => 5500, 'description' => 'Pulsa 5K'],
                ['amount' => 10000, 'price' => 10500, 'description' => 'Pulsa 10K'],
                ['amount' => 20000, 'price' => 20500, 'description' => 'Pulsa 20K'],
                ['amount' => 25000, 'price' => 25500, 'description' => 'Pulsa 25K'],
                ['amount' => 50000, 'price' => 50500, 'description' => 'Pulsa 50K'],
                ['amount' => 100000, 'price' => 100500, 'description' => 'Pulsa 100K'],
            ],
            'INDOSAT' => [
                ['amount' => 5000, 'price' => 5500, 'description' => 'Pulsa 5K'],
                ['amount' => 10000, 'price' => 10500, 'description' => 'Pulsa 10K'],
                ['amount' => 20000, 'price' => 20500, 'description' => 'Pulsa 20K'],
                ['amount' => 25000, 'price' => 25500, 'description' => 'Pulsa 25K'],
                ['amount' => 50000, 'price' => 50500, 'description' => 'Pulsa 50K'],
                ['amount' => 100000, 'price' => 100500, 'description' => 'Pulsa 100K'],
            ],
            'XL' => [
                ['amount' => 5000, 'price' => 5500, 'description' => 'Pulsa 5K'],
                ['amount' => 10000, 'price' => 10500, 'description' => 'Pulsa 10K'],
                ['amount' => 15000, 'price' => 15500, 'description' => 'Pulsa 15K'],
                ['amount' => 25000, 'price' => 25500, 'description' => 'Pulsa 25K'],
                ['amount' => 50000, 'price' => 50500, 'description' => 'Pulsa 50K'],
                ['amount' => 100000, 'price' => 100500, 'description' => 'Pulsa 100K'],
            ]
        ];

        return $denominations[$provider] ?? [];
    }

    /**
     * Get data package options
     */
    public function getDataPackages($provider)
    {
        $packages = [
            'TELKOMSEL' => [
                ['amount' => 1000, 'price' => 1500, 'description' => '1GB 1 Hari'],
                ['amount' => 5000, 'price' => 5500, 'description' => '2GB 3 Hari'],
                ['amount' => 10000, 'price' => 10500, 'description' => '3GB 7 Hari'],
                ['amount' => 25000, 'price' => 25500, 'description' => '8GB 30 Hari'],
                ['amount' => 50000, 'price' => 50500, 'description' => '15GB 30 Hari'],
                ['amount' => 100000, 'price' => 100500, 'description' => '35GB 30 Hari'],
            ],
            'INDOSAT' => [
                ['amount' => 5000, 'price' => 5500, 'description' => '1GB 3 Hari'],
                ['amount' => 10000, 'price' => 10500, 'description' => '2GB 7 Hari'],
                ['amount' => 25000, 'price' => 25500, 'description' => '6GB 30 Hari'],
                ['amount' => 50000, 'price' => 50500, 'description' => '12GB 30 Hari'],
                ['amount' => 100000, 'price' => 100500, 'description' => '25GB 30 Hari'],
            ]
        ];

        return $packages[$provider] ?? [];
    }

    /**
     * Process PPOB transaction
     */
    public function processTransaction($data)
    {
        try {
            // Generate order ID
            $orderId = 'PPOB-' . time() . '-' . rand(1000, 9999);
            
            // Calculate admin fee (2% or minimum 1000)
            $adminFee = max(1000, $data['amount'] * 0.02);
            $totalAmount = $data['amount'] + $adminFee;

            // Create payment via Ottopay
            $paymentData = [
                'order_id' => $orderId,
                'amount' => $totalAmount,
                'description' => $data['description'] ?? 'Pembayaran PPOB',
                'customer_name' => $data['customer_name'],
                'customer_email' => $data['customer_email'],
                'customer_phone' => $data['customer_phone'] ?? null,
                'return_url' => route('ppob.payment.return'),
                'callback_url' => route('ppob.payment.callback'),
                'payment_methods' => ['VA', 'QRIS', 'EWALLET']
            ];

            $paymentResult = $this->ottopayService->createPayment($paymentData);

            if ($paymentResult['success']) {
                // Save transaction to database
                $transaction = \App\Models\PpobTransaction::create([
                    'order_id' => $orderId,
                    'user_id' => Auth::check() ? Auth::user()->id : null,
                    'product_type' => $data['product_type'],
                    'provider' => $data['provider'] ?? null,
                    'target_number' => $data['target_number'],
                    'amount' => $data['amount'],
                    'admin_fee' => $adminFee,
                    'total_amount' => $totalAmount,
                    'status' => 'pending',
                    'payment_url' => $paymentResult['data']['payment_url'] ?? null,
                    'payment_data' => json_encode($paymentResult['data'])
                ]);

                return [
                    'success' => true,
                    'data' => [
                        'order_id' => $orderId,
                        'payment_url' => $paymentResult['data']['payment_url'] ?? null,
                        'total_amount' => $totalAmount,
                        'admin_fee' => $adminFee,
                        'transaction' => $transaction
                    ]
                ];
            }

            return $paymentResult;
        } catch (\Exception $e) {
            Log::error('PPOB transaction failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Gagal memproses transaksi: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Check transaction status
     */
    public function checkTransactionStatus($orderId)
    {
        try {
            $transaction = \App\Models\PpobTransaction::where('order_id', $orderId)->first();
            
            if (!$transaction) {
                return [
                    'success' => false,
                    'message' => 'Transaksi tidak ditemukan'
                ];
            }

            // Check payment status via Ottopay
            $paymentStatus = $this->ottopayService->checkPaymentStatus($orderId);

            if ($paymentStatus['success']) {
                $status = $paymentStatus['data']['status'] ?? 'pending';
                
                // Update transaction status
                $transaction->update([
                    'status' => $status,
                    'payment_response' => json_encode($paymentStatus['data'])
                ]);

                return [
                    'success' => true,
                    'data' => [
                        'transaction' => $transaction,
                        'payment_status' => $paymentStatus['data']
                    ]
                ];
            }

            return $paymentStatus;
        } catch (\Exception $e) {
            Log::error('Check PPOB transaction status failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Gagal mengecek status transaksi: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get user transactions
     */
    public function getUserTransactions($userId, $limit = 10)
    {
        return \App\Models\PpobTransaction::where('user_id', $userId)
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }
} 