<?php

namespace App\Services;

use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Endroid\QrCode\Writer\PngWriter;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Str;
use Carbon\Carbon;

class QRCodeService
{
    private $storagePath = 'qrcodes';
    private $publicPath = 'public/qrcodes';

    /**
     * Generate QR Code for ticket booking
     */
    public function generateTicketQR($booking)
    {
        try {
            $qrData = [
                'type' => 'ticket',
                'booking_id' => $booking->id,
                'booking_code' => $booking->kode_booking,
                'visitor_name' => $booking->nama_pemesan,
                'destination' => $booking->objekWisata->nama,
                'visit_date' => $booking->tanggal_kunjungan,
                'adult_count' => $booking->jumlah_dewasa,
                'child_count' => $booking->jumlah_anak,
                'total_amount' => $booking->total_harga,
                'generated_at' => now()->toISOString(),
                'expires_at' => Carbon::parse($booking->tanggal_kunjungan)->addDays(1)->toISOString()
            ];

            // Encrypt the data for security
            $encryptedData = Crypt::encrypt(json_encode($qrData));
            
            // Generate QR code
            $fileName = 'ticket_' . $booking->kode_booking . '_' . time() . '.png';
            $qrCode = QrCode::format('png')
                ->size(300)
                ->margin(2)
                ->errorCorrection('M')
                ->generate($encryptedData);

            // Save to storage
            Storage::put($this->publicPath . '/' . $fileName, $qrCode);

            return [
                'success' => true,
                'qr_code_path' => $this->storagePath . '/' . $fileName,
                'qr_code_url' => Storage::url($this->storagePath . '/' . $fileName),
                'file_name' => $fileName,
                'data' => $qrData
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Generate QR Code for product
     */
    public function generateProductQR($product)
    {
        try {
            $qrData = [
                'type' => 'product',
                'product_id' => $product->id,
                'sku' => $product->sku,
                'name' => $product->nama,
                'price' => $product->harga,
                'umkm_name' => $product->umkm->nama_umkm,
                'category' => $product->kategori,
                'product_url' => url("/marketplace/product/{$product->id}"),
                'generated_at' => now()->toISOString()
            ];

            $encryptedData = Crypt::encrypt(json_encode($qrData));
            
            $fileName = 'product_' . $product->sku . '_' . time() . '.png';
            $qrCode = QrCode::format('png')
                ->size(200)
                ->margin(1)
                ->errorCorrection('M')
                ->generate($encryptedData);

            Storage::put($this->publicPath . '/' . $fileName, $qrCode);

            return [
                'success' => true,
                'qr_code_path' => $this->storagePath . '/' . $fileName,
                'qr_code_url' => Storage::url($this->storagePath . '/' . $fileName),
                'file_name' => $fileName,
                'data' => $qrData
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Generate QR Code for UMKM business
     */
    public function generateUMKMQR($umkm)
    {
        try {
            $qrData = [
                'type' => 'umkm',
                'umkm_id' => $umkm->id,
                'business_name' => $umkm->nama_umkm,
                'owner' => $umkm->pemilik,
                'category' => $umkm->kategori_usaha,
                'address' => $umkm->alamat,
                'phone' => $umkm->nomor_telepon,
                'profile_url' => url("/umkm/profile/{$umkm->id}"),
                'generated_at' => now()->toISOString()
            ];

            $encryptedData = Crypt::encrypt(json_encode($qrData));
            
            $fileName = 'umkm_' . Str::slug($umkm->nama_umkm) . '_' . time() . '.png';
            $qrCode = QrCode::format('png')
                ->size(250)
                ->margin(2)
                ->errorCorrection('M')
                ->generate($encryptedData);

            Storage::put($this->publicPath . '/' . $fileName, $qrCode);

            return [
                'success' => true,
                'qr_code_path' => $this->storagePath . '/' . $fileName,
                'qr_code_url' => Storage::url($this->storagePath . '/' . $fileName),
                'file_name' => $fileName,
                'data' => $qrData
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Generate QR Code for tourism destination
     */
    public function generateDestinationQR($destination)
    {
        try {
            $qrData = [
                'type' => 'destination',
                'destination_id' => $destination->id,
                'name' => $destination->nama,
                'category' => $destination->kategori,
                'address' => $destination->alamat,
                'adult_price' => $destination->harga_tiket_dewasa,
                'child_price' => $destination->harga_tiket_anak,
                'destination_url' => url("/tourism/destination/{$destination->id}"),
                'booking_url' => url("/tourism/booking/{$destination->id}"),
                'generated_at' => now()->toISOString()
            ];

            $encryptedData = Crypt::encrypt(json_encode($qrData));
            
            $fileName = 'destination_' . Str::slug($destination->nama) . '_' . time() . '.png';
            $qrCode = QrCode::format('png')
                ->size(250)
                ->margin(2)
                ->errorCorrection('M')
                ->generate($encryptedData);

            Storage::put($this->publicPath . '/' . $fileName, $qrCode);

            return [
                'success' => true,
                'qr_code_path' => $this->storagePath . '/' . $fileName,
                'qr_code_url' => Storage::url($this->storagePath . '/' . $fileName),
                'file_name' => $fileName,
                'data' => $qrData
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Validate and decode QR code data
     */
    public function validateQRCode($qrData)
    {
        try {
            // Decrypt the data
            $decryptedData = Crypt::decrypt($qrData);
            $data = json_decode($decryptedData, true);

            if (!$data) {
                return [
                    'success' => false,
                    'error' => 'Invalid QR code format'
                ];
            }

            // Check if QR code has expired (for tickets)
            if (isset($data['expires_at'])) {
                $expiresAt = Carbon::parse($data['expires_at']);
                if (now()->gt($expiresAt)) {
                    return [
                        'success' => false,
                        'error' => 'QR code has expired',
                        'expired' => true
                    ];
                }
            }

            return [
                'success' => true,
                'data' => $data,
                'type' => $data['type'] ?? 'unknown'
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => 'Invalid or corrupted QR code: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Process ticket check-in via QR code
     */
    public function processTicketCheckIn($qrData, $scannedBy = null)
    {
        try {
            $validation = $this->validateQRCode($qrData);

            if (!$validation['success']) {
                return $validation;
            }

            $data = $validation['data'];

            if ($data['type'] !== 'ticket') {
                return [
                    'success' => false,
                    'error' => 'Invalid QR code type for check-in'
                ];
            }

            // Find the booking
            $booking = \App\Models\TiketWisata::where('id', $data['booking_id'])
                ->where('kode_booking', $data['booking_code'])
                ->first();

            if (!$booking) {
                return [
                    'success' => false,
                    'error' => 'Booking not found'
                ];
            }

            // Check if already checked in
            if ($booking->status_checkin === 'checked_in') {
                return [
                    'success' => false,
                    'error' => 'Ticket already checked in',
                    'checked_in_at' => $booking->waktu_checkin
                ];
            }

            // Check visit date
            $visitDate = Carbon::parse($data['visit_date']);
            $today = now()->startOfDay();

            if ($today->gt($visitDate->addDays(1))) {
                return [
                    'success' => false,
                    'error' => 'Ticket visit date has passed'
                ];
            }

            // Update booking status
            $booking->update([
                'status_checkin' => 'checked_in',
                'waktu_checkin' => now(),
                'petugas_checkin' => $scannedBy
            ]);

            return [
                'success' => true,
                'message' => 'Check-in successful',
                'booking' => $booking,
                'visitor_info' => [
                    'name' => $data['visitor_name'],
                    'destination' => $data['destination'],
                    'adult_count' => $data['adult_count'],
                    'child_count' => $data['child_count']
                ]
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => 'Check-in failed: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Generate simple text QR code
     */
    public function generateTextQR($text, $fileName = null, $size = 200)
    {
        try {
            if (!$fileName) {
                $fileName = 'text_' . Str::random(10) . '_' . time() . '.png';
            }

            $qrCode = QrCode::format('png')
                ->size($size)
                ->margin(2)
                ->errorCorrection('M')
                ->generate($text);

            Storage::put($this->publicPath . '/' . $fileName, $qrCode);

            return [
                'success' => true,
                'qr_code_path' => $this->storagePath . '/' . $fileName,
                'qr_code_url' => Storage::url($this->storagePath . '/' . $fileName),
                'file_name' => $fileName
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Generate URL QR code
     */
    public function generateUrlQR($url, $fileName = null, $size = 200)
    {
        try {
            if (!$fileName) {
                $fileName = 'url_' . Str::random(10) . '_' . time() . '.png';
            }

            $qrCode = QrCode::format('png')
                ->size($size)
                ->margin(2)
                ->errorCorrection('M')
                ->generate($url);

            Storage::put($this->publicPath . '/' . $fileName, $qrCode);

            return [
                'success' => true,
                'qr_code_path' => $this->storagePath . '/' . $fileName,
                'qr_code_url' => Storage::url($this->storagePath . '/' . $fileName),
                'file_name' => $fileName
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Delete QR code file
     */
    public function deleteQRCode($fileName)
    {
        try {
            $filePath = $this->publicPath . '/' . $fileName;
            
            if (Storage::exists($filePath)) {
                Storage::delete($filePath);
                return true;
            }

            return false;

        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Get QR code statistics
     */
    public function getQRCodeStats()
    {
        try {
            $files = Storage::files($this->publicPath);
            
            $stats = [
                'total_qr_codes' => count($files),
                'ticket_qr_codes' => 0,
                'product_qr_codes' => 0,
                'umkm_qr_codes' => 0,
                'destination_qr_codes' => 0,
                'other_qr_codes' => 0
            ];

            foreach ($files as $file) {
                $fileName = basename($file);
                if (strpos($fileName, 'ticket_') === 0) {
                    $stats['ticket_qr_codes']++;
                } elseif (strpos($fileName, 'product_') === 0) {
                    $stats['product_qr_codes']++;
                } elseif (strpos($fileName, 'umkm_') === 0) {
                    $stats['umkm_qr_codes']++;
                } elseif (strpos($fileName, 'destination_') === 0) {
                    $stats['destination_qr_codes']++;
                } else {
                    $stats['other_qr_codes']++;
                }
            }

            return $stats;

        } catch (\Exception $e) {
            return [];
        }
    }

    /**
     * Cleanup expired QR codes
     */
    public function cleanupExpiredQRCodes()
    {
        try {
            $files = Storage::files($this->publicPath);
            $deletedCount = 0;

            foreach ($files as $file) {
                $fileName = basename($file);
                
                // Only cleanup ticket QR codes (they have expiry)
                if (strpos($fileName, 'ticket_') === 0) {
                    $fileAge = Carbon::createFromTimestamp(Storage::lastModified($file));
                    
                    // Delete QR codes older than 7 days
                    if ($fileAge->lt(now()->subDays(7))) {
                        Storage::delete($file);
                        $deletedCount++;
                    }
                }
            }

            return [
                'success' => true,
                'deleted_count' => $deletedCount
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Generate QR Code as base64 image
     */
    public function generateQRCodeImage($data, $size = 200)
    {
        try {
            // Create QR code with the data
            $qrCode = QrCode::create($data)
                ->setSize($size)
                ->setMargin(10);

            // Create writer
            $writer = new PngWriter();
            
            // Generate QR code image
            $result = $writer->write($qrCode);
            
            // Convert to base64
            $base64 = base64_encode($result->getString());
            
            return 'data:image/png;base64,' . $base64;
            
        } catch (\Exception $e) {
            Log::error('QR Code generation failed: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Generate QR Code for surat validation
     */
    public function generateSuratQRCode($pelayananSurat)
    {
        $qrData = json_encode([
            'type' => 'surat_validation',
            'nomor_surat' => $pelayananSurat->nomor_surat,
            'nik' => $pelayananSurat->nik,
            'nama' => $pelayananSurat->penduduk->nama ?? '',
            'jenis_surat' => $pelayananSurat->jenis_surat,
            'tanggal' => $pelayananSurat->created_at->format('Y-m-d'),
            'hash' => hash('sha256', $pelayananSurat->nomor_surat . $pelayananSurat->nik . $pelayananSurat->created_at),
            'url' => url('/validasi-surat/' . base64_encode($pelayananSurat->nomor_surat))
        ]);

        return $this->generateQRCodeImage($qrData, 150);
    }

    /**
     * Validate surat QR Code
     */
    public function validateSuratQRCode($qrData)
    {
        try {
            $data = json_decode($qrData, true);
            
            if (!$data || !isset($data['nomor_surat'], $data['hash'])) {
                return [
                    'valid' => false,
                    'message' => 'QR Code tidak valid'
                ];
            }

            // Find surat by nomor
            $surat = \App\Models\PelayananSurat::where('nomor_surat', $data['nomor_surat'])->first();
            
            if (!$surat) {
                return [
                    'valid' => false,
                    'message' => 'Surat tidak ditemukan'
                ];
            }

            // Verify hash
            $expectedHash = hash('sha256', $surat->nomor_surat . $surat->nik . $surat->created_at);
            
            if ($data['hash'] !== $expectedHash) {
                return [
                    'valid' => false,
                    'message' => 'QR Code tidak valid atau telah diubah'
                ];
            }

            return [
                'valid' => true,
                'message' => 'Surat valid',
                'data' => [
                    'nomor_surat' => $surat->nomor_surat,
                    'jenis_surat' => $surat->jenis_surat,
                    'nama_pemohon' => $surat->penduduk->nama ?? $surat->nama_pemohon,
                    'nik' => $surat->nik,
                    'tanggal_terbit' => $surat->tanggal_selesai?->format('d F Y'),
                    'status' => $surat->status
                ]
            ];
            
        } catch (\Exception $e) {
            return [
                'valid' => false,
                'message' => 'Error validasi: ' . $e->getMessage()
            ];
        }
    }
} 
