<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('penduduk', function (Blueprint $table) {
            // Check if columns don't exist before adding them
            if (!Schema::hasColumn('penduduk', 'status_kawin')) {
                $table->enum('status_kawin', ['Belum Kawin', 'Kawin', 'Cerai Hidup', 'Cerai Mati'])->default('Belum Kawin')->after('jenis_kelamin');
            }
            
            if (!Schema::hasColumn('penduduk', 'kewarganegaraan')) {
                $table->enum('kewarganegaraan', ['WNI', 'WNA'])->default('WNI')->after('no_wa');
            }
            
            if (!Schema::hasColumn('penduduk', 'desa')) {
                $table->string('desa')->nullable()->after('dusun');
            }
            
            if (!Schema::hasColumn('penduduk', 'keluarga_id')) {
                $table->foreignId('keluarga_id')->nullable()->after('no_kk')->constrained('keluarga')->onDelete('set null');
            }
            
            if (!Schema::hasColumn('penduduk', 'is_kepala_keluarga')) {
                $table->boolean('is_kepala_keluarga')->default(false)->after('keluarga_id');
            }
            
            // Add missing demographic fields
            if (!Schema::hasColumn('penduduk', 'status_hidup')) {
                $table->enum('status_hidup', ['Hidup', 'Meninggal'])->default('Hidup')->after('kewarganegaraan');
            }
            
            if (!Schema::hasColumn('penduduk', 'tanggal_meninggal')) {
                $table->date('tanggal_meninggal')->nullable()->after('status_hidup');
            }
            
            if (!Schema::hasColumn('penduduk', 'sebab_kematian')) {
                $table->string('sebab_kematian')->nullable()->after('tanggal_meninggal');
            }
            
            // Add mutation tracking fields
            if (!Schema::hasColumn('penduduk', 'status_mutasi')) {
                $table->enum('status_mutasi', ['Tetap', 'Pindah Masuk', 'Pindah Keluar', 'Lahir', 'Meninggal'])->default('Tetap')->after('sebab_kematian');
            }
            
            if (!Schema::hasColumn('penduduk', 'tanggal_mutasi')) {
                $table->date('tanggal_mutasi')->nullable()->after('status_mutasi');
            }
            
            if (!Schema::hasColumn('penduduk', 'keterangan_mutasi')) {
                $table->text('keterangan_mutasi')->nullable()->after('tanggal_mutasi');
            }
            
            if (!Schema::hasColumn('penduduk', 'asal_mutasi')) {
                $table->string('asal_mutasi')->nullable()->after('keterangan_mutasi'); // Untuk pindah masuk
            }
            
            if (!Schema::hasColumn('penduduk', 'tujuan_mutasi')) {
                $table->string('tujuan_mutasi')->nullable()->after('asal_mutasi'); // Untuk pindah keluar
            }
            
            // Add additional demographic info
            if (!Schema::hasColumn('penduduk', 'golongan_darah')) {
                $table->string('golongan_darah', 2)->nullable()->after('tujuan_mutasi');
            }
            
            if (!Schema::hasColumn('penduduk', 'cacat_fisik')) {
                $table->enum('cacat_fisik', ['Tidak Ada', 'Tuna Netra', 'Tuna Rungu', 'Tuna Wicara', 'Tuna Daksa', 'Cacat Mental', 'Cacat Fisik dan Mental'])->default('Tidak Ada')->after('golongan_darah');
            }
        });
        
        // Add indexes after all columns are created
        Schema::table('penduduk', function (Blueprint $table) {
            if (!Schema::hasIndex('penduduk', 'penduduk_status_hidup_index')) {
                $table->index('status_hidup');
            }
            if (!Schema::hasIndex('penduduk', 'penduduk_status_mutasi_index')) {
                $table->index('status_mutasi');
            }
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('penduduk', function (Blueprint $table) {
            // Drop indexes first
            $indexes = ['penduduk_status_hidup_index', 'penduduk_status_mutasi_index'];
            foreach ($indexes as $index) {
                if (Schema::hasIndex('penduduk', $index)) {
                    $table->dropIndex($index);
                }
            }
            
            // Drop foreign key if exists
            if (Schema::hasColumn('penduduk', 'keluarga_id')) {
                $table->dropForeign(['keluarga_id']);
            }
            
            // Drop columns if they exist
            $columns = [
                'status_kawin', 'kewarganegaraan', 'desa', 'keluarga_id', 'is_kepala_keluarga',
                'status_hidup', 'tanggal_meninggal', 'sebab_kematian', 'status_mutasi',
                'tanggal_mutasi', 'keterangan_mutasi', 'asal_mutasi', 'tujuan_mutasi',
                'golongan_darah', 'cacat_fisik'
            ];
            
            foreach ($columns as $column) {
                if (Schema::hasColumn('penduduk', $column)) {
                    $table->dropColumn($column);
                }
            }
        });
    }
}; 