#!/bin/bash

# Shared Hosting Deployment Script for Desa Susukan
# This script prepares the application for shared hosting deployment

echo "🚀 Starting shared hosting deployment..."

# 1. Create storage symlink (if supported)
echo "📁 Creating storage symlink..."
php artisan storage:link

# 2. Sync storage files to public directory
echo "📂 Syncing storage files for shared hosting compatibility..."
php artisan storage:sync-shared-hosting

# 3. Set proper permissions
echo "🔐 Setting file permissions..."
chmod -R 755 public/storage
chmod -R 755 storage/app/public

# 4. Clear and cache config
echo "🧹 Clearing caches..."
php artisan config:clear
php artisan cache:clear
php artisan view:clear
php artisan route:clear

# 5. Optimize for production
echo "⚡ Optimizing for production..."
php artisan config:cache
php artisan route:cache
php artisan view:cache

# 6. Create .htaccess for storage directory if it doesn't exist
if [ ! -f "public/storage/.htaccess" ]; then
    echo "📝 Creating .htaccess for storage directory..."
    cat > public/storage/.htaccess << 'EOF'
RewriteEngine On

# Handle requests for files in storage directory
RewriteCond %{REQUEST_FILENAME} !-f
RewriteCond %{REQUEST_FILENAME} !-d
RewriteRule ^(.*)$ index.php [QSA,L]

# Security headers
<Files "index.php">
    Header set X-Content-Type-Options nosniff
    Header set X-Frame-Options DENY
    Header set X-XSS-Protection "1; mode=block"
</Files>

# Prevent access to sensitive files
<FilesMatch "\.(php|phtml|php3|php4|php5|pl|py|jsp|asp|sh|cgi)$">
    Order Deny,Allow
    Deny from all
</FilesMatch>

# Allow specific file types
<FilesMatch "\.(jpg|jpeg|png|gif|pdf|doc|docx|xls|xlsx|txt|csv)$">
    Order Allow,Deny
    Allow from all
</FilesMatch>
EOF
fi

# 7. Create index.php for storage directory if it doesn't exist
if [ ! -f "public/storage/index.php" ]; then
    echo "📄 Creating index.php for storage directory..."
    cat > public/storage/index.php << 'EOF'
<?php
/**
 * Shared Hosting Compatible Storage Access
 * This file provides access to storage files when symlinks don't work on shared hosting
 */

// Security check - only allow access to specific file types
$allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt', 'csv'];
$allowedMimeTypes = [
    'image/jpeg', 'image/png', 'image/gif', 'application/pdf',
    'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'text/plain', 'text/csv'
];

// Get the requested file path
$requestUri = $_SERVER['REQUEST_URI'];
$path = parse_url($requestUri, PHP_URL_PATH);

// Remove /storage/ prefix
$filePath = str_replace('/storage/', '', $path);
$filePath = ltrim($filePath, '/');

// Security: prevent directory traversal
if (strpos($filePath, '..') !== false || strpos($filePath, '\\') !== false) {
    http_response_code(403);
    exit('Access denied');
}

// Build full file path
$fullPath = __DIR__ . '/../storage/app/public/' . $filePath;

// Check if file exists
if (!file_exists($fullPath) || !is_file($fullPath)) {
    http_response_code(404);
    exit('File not found');
}

// Get file extension
$extension = strtolower(pathinfo($fullPath, PATHINFO_EXTENSION));

// Check if extension is allowed
if (!in_array($extension, $allowedExtensions)) {
    http_response_code(403);
    exit('File type not allowed');
}

// Get MIME type
$mimeType = mime_content_type($fullPath);

// Check if MIME type is allowed
if (!in_array($mimeType, $allowedMimeTypes)) {
    http_response_code(403);
    exit('File type not allowed');
}

// Set appropriate headers
header('Content-Type: ' . $mimeType);
header('Content-Length: ' . filesize($fullPath));
header('Cache-Control: public, max-age=31536000'); // Cache for 1 year
header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 31536000) . ' GMT');

// Output the file
readfile($fullPath);
exit;
EOF
fi

echo "✅ Shared hosting deployment completed!"
echo ""
echo "📋 Deployment Summary:"
echo "   - Storage symlink created"
echo "   - Storage files synced to public/storage/"
echo "   - File permissions set"
echo "   - Caches cleared and optimized"
echo "   - Security files created"
echo ""
echo "🌐 Your application is now ready for shared hosting!"
echo "   Files will be accessible via: https://yourdomain.com/storage/"
