import 'package:flutter/foundation.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'dart:convert';

import '../models/staff.dart';
import '../services/api_service.dart';
import '../core/constants/app_constants.dart';

class StaffProvider extends ChangeNotifier {
  List<Staff> _staffList = [];
  bool _isLoading = false;
  String? _errorMessage;
  int _currentSlide = 0;
  int _totalSlides = 0;

  // Getters
  List<Staff> get staffList => _staffList;
  bool get isLoading => _isLoading;
  String? get errorMessage => _errorMessage;
  bool get hasError => _errorMessage != null;
  bool get hasData => _staffList.isNotEmpty;
  int get currentSlide => _currentSlide;
  int get totalSlides => _totalSlides;

  // Staff slides (2 per slide for grid layout)
  List<List<Staff>> get staffSlides {
    if (_staffList.isEmpty) return [];
    
    final slides = <List<Staff>>[];
    for (int i = 0; i < _staffList.length; i += 2) {
      final slideStaff = _staffList.skip(i).take(2).toList();
      // Pad with empty staff if needed to maintain grid consistency
      while (slideStaff.length < 2) {
        slideStaff.add(const Staff()); // Empty staff placeholder
      }
      slides.add(slideStaff);
    }
    
    _totalSlides = slides.length;
    return slides;
  }

  StaffProvider() {
    _loadCachedData();
    loadStaffList();
  }

  Future<void> _loadCachedData() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      final cachedData = prefs.getString(AppConstants.staffDataKey);
      
      if (cachedData != null) {
        final jsonData = json.decode(cachedData);
        final staffJsonList = jsonData['data'] as List;
        _staffList = staffJsonList.map((json) => Staff.fromJson(json)).toList();
        
        // Check if cache is expired
        final cacheTime = DateTime.parse(jsonData['cached_at']);
        final now = DateTime.now();
        final difference = now.difference(cacheTime).inHours;
        
        if (difference < AppConstants.cacheStaffDuration) {
          notifyListeners();
          return;
        }
      }
    } catch (e) {
      debugPrint('Error loading cached staff data: $e');
    }
  }

  Future<void> loadStaffList({bool forceRefresh = false}) async {
    if (_isLoading && !forceRefresh) return;

    _setLoading(true);
    _clearError();

    try {
      final response = await ApiService.instance.handleRequest(
        () => ApiService.instance.client.getStaffList(),
      );

      if (response.success && response.data != null) {
        _staffList = response.data!;
        _updateSlideCount();
        await _cacheData();
        _setLoading(false);
      } else {
        _setError(response.message ?? 'Gagal memuat data perangkat desa');
      }
    } catch (e) {
      _setError(e.toString());
    }
  }

  Future<void> _cacheData() async {
    if (_staffList.isEmpty) return;
    
    try {
      final prefs = await SharedPreferences.getInstance();
      final cacheData = {
        'data': _staffList.map((staff) => staff.toJson()).toList(),
        'cached_at': DateTime.now().toIso8601String(),
      };
      
      await prefs.setString(
        AppConstants.staffDataKey,
        json.encode(cacheData),
      );
    } catch (e) {
      debugPrint('Error caching staff data: $e');
    }
  }

  void _updateSlideCount() {
    _totalSlides = (_staffList.length / 2).ceil();
    if (_currentSlide >= _totalSlides) {
      _currentSlide = 0;
    }
  }

  // Slider control methods
  void nextSlide() {
    if (_totalSlides <= 1) return;
    
    _currentSlide = (_currentSlide + 1) % _totalSlides;
    notifyListeners();
  }

  void previousSlide() {
    if (_totalSlides <= 1) return;
    
    _currentSlide = _currentSlide > 0 ? _currentSlide - 1 : _totalSlides - 1;
    notifyListeners();
  }

  void goToSlide(int index) {
    if (index >= 0 && index < _totalSlides) {
      _currentSlide = index;
      notifyListeners();
    }
  }

  // Get staff for current slide
  List<Staff> getCurrentSlideStaff() {
    final slides = staffSlides;
    if (slides.isEmpty || _currentSlide >= slides.length) {
      return [];
    }
    return slides[_currentSlide];
  }

  void _setLoading(bool loading) {
    _isLoading = loading;
    notifyListeners();
  }

  void _setError(String error) {
    _errorMessage = error;
    _isLoading = false;
    notifyListeners();
  }

  void _clearError() {
    _errorMessage = null;
    notifyListeners();
  }

  // Refresh data
  Future<void> refresh() async {
    await loadStaffList(forceRefresh: true);
  }

  // Clear cache
  Future<void> clearCache() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      await prefs.remove(AppConstants.staffDataKey);
      _staffList.clear();
      _currentSlide = 0;
      _totalSlides = 0;
      notifyListeners();
    } catch (e) {
      debugPrint('Error clearing staff cache: $e');
    }
  }

}
