<?php
/**
 * Create Tourism Images for susukannet.id
 * This script creates the missing tourism images directly on the server
 */

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Create Tourism Images - susukannet.id</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .step { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .test-image { max-width: 200px; height: 150px; object-fit: cover; border: 2px solid #ddd; margin: 10px; border-radius: 4px; }
        .test-image.loaded { border-color: #28a745; }
        .test-image.error { border-color: #dc3545; }
        .url-box { font-family: monospace; background: #f1f1f1; padding: 8px; border-radius: 4px; margin: 5px 0; word-break: break-all; }
    </style>
    <script>
        function imageLoaded(img) {
            img.classList.add('loaded');
            img.classList.remove('error');
            console.log('Image loaded:', img.src);
        }
        
        function imageError(img) {
            img.classList.add('error');
            img.classList.remove('loaded');
            console.log('Image error:', img.src);
        }
    </script>
</head>
<body>
    <div class="container">
        <h1>🖼️ Create Tourism Images</h1>
        
        <?php
        $baseUrl = 'https://susukannet.id';
        echo "<p><strong>Target Domain:</strong> {$baseUrl}</p>";
        echo "<p><strong>Server Path:</strong> " . __DIR__ . "</p>";
        echo "<p><strong>Time:</strong> " . date('Y-m-d H:i:s') . "</p>";
        
        // Step 1: Create directories
        echo "<div class='step'>";
        echo "<h2>📁 Step 1: Create Directories</h2>";
        
        $directories = [
            __DIR__ . '/storage',
            __DIR__ . '/storage/wisata',
            __DIR__ . '/storage/wisata/galeri',
            __DIR__ . '/images'
        ];
        
        foreach ($directories as $dir) {
            if (!is_dir($dir)) {
                if (mkdir($dir, 0755, true)) {
                    echo "<div class='success'>✅ Created: " . str_replace(__DIR__ . '/', '', $dir) . "</div>";
                } else {
                    echo "<div class='error'>❌ Failed to create: " . str_replace(__DIR__ . '/', '', $dir) . "</div>";
                }
            } else {
                $fileCount = count(glob($dir . '/*'));
                echo "<div class='info'>📁 Exists: " . str_replace(__DIR__ . '/', '', $dir) . " ({$fileCount} files)</div>";
            }
        }
        echo "</div>";
        
        // Step 2: Create default tourism image
        echo "<div class='step'>";
        echo "<h2>🖼️ Step 2: Create Default Tourism Image</h2>";
        
        $defaultImagePath = __DIR__ . '/images/default-tourism.png';
        
        if (!file_exists($defaultImagePath)) {
            // Try to create with GD
            if (extension_loaded('gd')) {
                $img = imagecreate(400, 300);
                $bg = imagecolorallocate($img, 240, 248, 255); // Light blue background
                $text_color = imagecolorallocate($img, 70, 130, 180); // Steel blue text
                $border_color = imagecolorallocate($img, 176, 196, 222); // Light steel blue border
                
                // Fill background
                imagefill($img, 0, 0, $bg);
                
                // Draw border
                imagerectangle($img, 5, 5, 394, 294, $border_color);
                imagerectangle($img, 6, 6, 393, 293, $border_color);
                
                // Add text
                $text1 = 'Default Tourism';
                $text2 = 'Image';
                
                // Center the text
                $text1_x = (400 - strlen($text1) * 10) / 2;
                $text2_x = (400 - strlen($text2) * 10) / 2;
                
                imagestring($img, 5, $text1_x, 130, $text1, $text_color);
                imagestring($img, 5, $text2_x, 160, $text2, $text_color);
                
                if (imagepng($img, $defaultImagePath)) {
                    echo "<div class='success'>✅ Created default-tourism.png with GD</div>";
                } else {
                    echo "<div class='error'>❌ Failed to save default-tourism.png</div>";
                }
                imagedestroy($img);
            } else {
                // Create a simple text file as fallback
                $content = "Default Tourism Image - PNG format not available";
                if (file_put_contents($defaultImagePath . '.txt', $content)) {
                    echo "<div class='warning'>⚠️ GD not available, created text file instead</div>";
                } else {
                    echo "<div class='error'>❌ Cannot create default image</div>";
                }
            }
        } else {
            $fileSize = filesize($defaultImagePath);
            echo "<div class='info'>📷 default-tourism.png already exists (" . number_format($fileSize) . " bytes)</div>";
        }
        
        // Test default image
        if (file_exists($defaultImagePath)) {
            $defaultUrl = $baseUrl . '/images/default-tourism.png';
            echo "<div class='url-box'>{$defaultUrl}</div>";
            echo "<img src='{$defaultUrl}?v=" . time() . "' class='test-image' alt='Default Tourism' onload='imageLoaded(this)' onerror='imageError(this)'>";
        }
        echo "</div>";
        
        // Step 3: Create sample tourism images
        echo "<div class='step'>";
        echo "<h2>🏞️ Step 3: Create Sample Tourism Images</h2>";
        
        $sampleImages = [
            'sample-tourism-1.png' => ['color' => [52, 152, 219], 'text' => 'Tourism 1'], // Blue
            'sample-tourism-2.png' => ['color' => [46, 204, 113], 'text' => 'Tourism 2']  // Green
        ];
        
        foreach ($sampleImages as $filename => $config) {
            $filepath = __DIR__ . '/storage/wisata/' . $filename;
            
            if (!file_exists($filepath)) {
                if (extension_loaded('gd')) {
                    $img = imagecreate(600, 400);
                    $bg = imagecolorallocate($img, $config['color'][0], $config['color'][1], $config['color'][2]);
                    $text_color = imagecolorallocate($img, 255, 255, 255);
                    $border_color = imagecolorallocate($img, 255, 255, 255);
                    
                    // Fill background
                    imagefill($img, 0, 0, $bg);
                    
                    // Draw border
                    imagerectangle($img, 10, 10, 589, 389, $border_color);
                    imagerectangle($img, 11, 11, 588, 388, $border_color);
                    
                    // Add text
                    $text1 = 'Sample';
                    $text2 = $config['text'];
                    
                    // Center the text
                    $text1_x = (600 - strlen($text1) * 15) / 2;
                    $text2_x = (600 - strlen($text2) * 15) / 2;
                    
                    imagestring($img, 5, $text1_x, 180, $text1, $text_color);
                    imagestring($img, 5, $text2_x, 210, $text2, $text_color);
                    
                    if (imagepng($img, $filepath)) {
                        echo "<div class='success'>✅ Created {$filename}</div>";
                    } else {
                        echo "<div class='error'>❌ Failed to create {$filename}</div>";
                    }
                    imagedestroy($img);
                } else {
                    echo "<div class='warning'>⚠️ GD not available for {$filename}</div>";
                }
            } else {
                $fileSize = filesize($filepath);
                echo "<div class='info'>📷 {$filename} already exists (" . number_format($fileSize) . " bytes)</div>";
            }
            
            // Test image
            if (file_exists($filepath)) {
                $imageUrl = $baseUrl . '/storage/wisata/' . $filename;
                echo "<div class='url-box'>{$imageUrl}</div>";
                echo "<img src='{$imageUrl}?v=" . time() . "' class='test-image' alt='{$filename}' onload='imageLoaded(this)' onerror='imageError(this)'>";
                echo "<p><a href='{$imageUrl}' target='_blank'>Open {$filename} in new tab</a></p>";
            }
        }
        echo "</div>";
        
        // Step 4: Test file permissions
        echo "<div class='step'>";
        echo "<h2>🔐 Step 4: Check File Permissions</h2>";
        
        $testFiles = [
            'images/default-tourism.png' => __DIR__ . '/images/default-tourism.png',
            'storage/wisata/sample-tourism-1.png' => __DIR__ . '/storage/wisata/sample-tourism-1.png',
            'storage/wisata/sample-tourism-2.png' => __DIR__ . '/storage/wisata/sample-tourism-2.png'
        ];
        
        foreach ($testFiles as $relativePath => $fullPath) {
            if (file_exists($fullPath)) {
                $perms = fileperms($fullPath);
                $permsOctal = sprintf('%o', $perms);
                $readable = is_readable($fullPath);
                $fileSize = filesize($fullPath);
                
                echo "<div class='info'>";
                echo "<strong>{$relativePath}:</strong><br>";
                echo "Permissions: {$permsOctal} | ";
                echo "Readable: " . ($readable ? '✅ YES' : '❌ NO') . " | ";
                echo "Size: " . number_format($fileSize) . " bytes";
                echo "</div>";
            } else {
                echo "<div class='error'><strong>{$relativePath}:</strong> File not found</div>";
            }
        }
        echo "</div>";
        
        // Step 5: Laravel database cleanup (if available)
        echo "<div class='step'>";
        echo "<h2>🧹 Step 5: Database Cleanup</h2>";
        
        try {
            if (file_exists(__DIR__ . '/../vendor/autoload.php')) {
                require_once __DIR__ . '/../vendor/autoload.php';
                $app = require_once __DIR__ . '/../bootstrap/app.php';
                $app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();
                
                $tourismCount = \App\Models\ObjekWisata::count();
                echo "<div class='info'>📊 Total tourism objects: {$tourismCount}</div>";
                
                // Clean temporary paths
                $tempObjects = \App\Models\ObjekWisata::where('foto_utama', 'like', '%tmp%')
                                                    ->orWhere('foto_utama', 'like', '%php%')
                                                    ->get();
                
                $cleanedCount = 0;
                foreach ($tempObjects as $obj) {
                    $oldPath = $obj->foto_utama;
                    $obj->update(['foto_utama' => 'sample-tourism-1.png']);
                    echo "<div class='warning'>🧹 Fixed {$obj->nama_wisata}: {$oldPath} → sample-tourism-1.png</div>";
                    $cleanedCount++;
                }
                
                if ($cleanedCount > 0) {
                    echo "<div class='success'>✅ Fixed {$cleanedCount} tourism objects</div>";
                } else {
                    echo "<div class='info'>ℹ️ No temporary paths found</div>";
                }
                
                // Test URL generation
                $testTourism = \App\Models\ObjekWisata::first();
                if ($testTourism) {
                    $generatedUrl = $testTourism->foto_utama_url;
                    echo "<div class='info'><strong>Sample Generated URL:</strong> {$generatedUrl}</div>";
                    
                    if (strpos($generatedUrl, 'susukannet.id') !== false) {
                        echo "<div class='success'>✅ URL uses correct domain</div>";
                    } else {
                        echo "<div class='error'>❌ URL uses wrong domain: {$generatedUrl}</div>";
                    }
                }
                
            } else {
                echo "<div class='warning'>⚠️ Laravel not available for database operations</div>";
            }
        } catch (Exception $e) {
            echo "<div class='error'>❌ Laravel Error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Step 6: Final verification
        echo "<div class='step'>";
        echo "<h2>✅ Step 6: Final Verification</h2>";
        
        $verificationUrls = [
            'Default Image' => $baseUrl . '/images/default-tourism.png',
            'Sample Tourism 1' => $baseUrl . '/storage/wisata/sample-tourism-1.png',
            'Sample Tourism 2' => $baseUrl . '/storage/wisata/sample-tourism-2.png',
            'Admin Page' => $baseUrl . '/admin/objek-wisata',
            'Public Page' => $baseUrl . '/wisata'
        ];
        
        echo "<div class='info'><strong>Test these URLs:</strong></div>";
        foreach ($verificationUrls as $name => $url) {
            echo "<div class='url-box'><a href='{$url}' target='_blank'>{$name}: {$url}</a></div>";
        }
        
        echo "<div class='success'>";
        echo "<strong>Expected Results After This Fix:</strong><br>";
        echo "• All image URLs should return actual images<br>";
        echo "• No more 404 errors for tourism images<br>";
        echo "• Admin page shows tourism with working images<br>";
        echo "• All URLs use https://susukannet.id domain";
        echo "</div>";
        echo "</div>";
        ?>
        
        <div class="step">
            <h2>🚀 Next Steps</h2>
            <p><strong>After running this script:</strong></p>
            <ol>
                <li>Run: <code>php artisan config:clear && php artisan view:clear</code></li>
                <li>Test: <a href="https://susukannet.id/test-live-images.php" target="_blank">Live Image Test</a></li>
                <li>Verify: <a href="https://susukannet.id/admin/objek-wisata" target="_blank">Admin Tourism Page</a></li>
                <li>Check: <a href="https://susukannet.id/wisata" target="_blank">Public Tourism Page</a></li>
            </ol>
            
            <p><strong>If images still don't show:</strong></p>
            <ul>
                <li>Check server error logs</li>
                <li>Verify .htaccess allows image access</li>
                <li>Ensure proper file permissions (644 for files, 755 for directories)</li>
                <li>Contact hosting provider about storage access</li>
            </ul>
        </div>
    </div>
</body>
</html>
