<?php
/**
 * Debug Upload Process for Tourism
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ObjekWisata;
use App\Services\FileUploadService;
use Illuminate\Support\Facades\Storage;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Debug Upload Process</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .step { margin: 15px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .code { background: #f1f1f1; padding: 8px; border-radius: 4px; font-family: monospace; margin: 5px 0; white-space: pre-wrap; }
        .form-control { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; margin: 5px 0; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; }
    </style>
</head>
<body>
    <h1>🔍 Debug Upload Process</h1>
    
    <?php
    // Step 1: Create test image programmatically
    echo "<div class='step'>";
    echo "<h2>📷 Step 1: Create Test Image</h2>";
    
    $testImagePath = __DIR__ . '/test-tourism-upload.png';
    
    // Create test image with GD
    if (extension_loaded('gd')) {
        $img = imagecreate(400, 300);
        $bg = imagecolorallocate($img, 100, 150, 200);
        $text = imagecolorallocate($img, 255, 255, 255);
        
        imagefill($img, 0, 0, $bg);
        imagestring($img, 5, 150, 140, 'TEST UPLOAD', $text);
        imagestring($img, 3, 160, 160, date('H:i:s'), $text);
        
        if (imagepng($img, $testImagePath)) {
            echo "<div class='success'>✅ Created test image: test-tourism-upload.png</div>";
        }
        imagedestroy($img);
    } else {
        echo "<div class='error'>❌ GD extension not available</div>";
    }
    echo "</div>";
    
    // Step 2: Test FileUploadService directly
    echo "<div class='step'>";
    echo "<h2>🧪 Step 2: Test FileUploadService</h2>";
    
    if (file_exists($testImagePath)) {
        try {
            // Create UploadedFile object
            $uploadedFile = new \Illuminate\Http\UploadedFile(
                $testImagePath,
                'test-tourism-upload.png',
                'image/png',
                null,
                true
            );
            
            echo "<div class='info'>📁 Test file created: " . filesize($testImagePath) . " bytes</div>";
            
            $uploadService = new FileUploadService();
            $result = $uploadService->uploadPhotoWithResize($uploadedFile, 'wisata');
            
            if ($result['success']) {
                echo "<div class='success'>✅ Upload successful!</div>";
                echo "<div class='info'><strong>Filename:</strong> {$result['filename']}</div>";
                echo "<div class='info'><strong>Path:</strong> {$result['path']}</div>";
                
                // Check if file exists in storage
                $storagePath = storage_path('app/public/wisata/' . $result['filename']);
                $publicPath = public_path('storage/wisata/' . $result['filename']);
                
                echo "<div class='info'><strong>Storage exists:</strong> " . (file_exists($storagePath) ? '✅ YES' : '❌ NO') . "</div>";
                echo "<div class='info'><strong>Public exists:</strong> " . (file_exists($publicPath) ? '✅ YES' : '❌ NO') . "</div>";
                
                // Test URL
                $testUrl = 'https://susukannet.id/storage/wisata/' . $result['filename'];
                echo "<div class='code'>{$testUrl}</div>";
                echo "<img src='{$testUrl}?v=" . time() . "' style='max-width: 200px; border: 1px solid #ddd; margin: 10px;' alt='Test Upload'>";
                
            } else {
                echo "<div class='error'>❌ Upload failed: {$result['error']}</div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Exception: " . $e->getMessage() . "</div>";
        }
    }
    echo "</div>";
    
    // Step 3: Test update process
    echo "<div class='step'>";
    echo "<h2>🔄 Step 3: Test Tourism Update</h2>";
    
    try {
        $tourism = ObjekWisata::first();
        
        if ($tourism && isset($result) && $result['success']) {
            $oldPhoto = $tourism->foto_utama;
            
            echo "<div class='info'><strong>Tourism:</strong> {$tourism->nama_wisata}</div>";
            echo "<div class='info'><strong>Old photo:</strong> " . ($oldPhoto ?: 'NULL') . "</div>";
            
            // Update with new photo
            $tourism->update(['foto_utama' => $result['filename']]);
            
            echo "<div class='success'>✅ Updated tourism with new photo</div>";
            echo "<div class='info'><strong>New photo:</strong> {$result['filename']}</div>";
            
            // Test accessor
            $generatedUrl = $tourism->fresh()->foto_utama_url;
            echo "<div class='info'><strong>Generated URL:</strong> {$generatedUrl}</div>";
            
            // Display updated image
            echo "<img src='{$generatedUrl}?v=" . time() . "' style='max-width: 200px; border: 1px solid #ddd; margin: 10px;' alt='Updated Tourism'>";
            
        } else {
            echo "<div class='warning'>⚠️ No tourism found or upload failed</div>";
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ Update error: " . $e->getMessage() . "</div>";
    }
    echo "</div>";
    
    // Step 4: Test controller simulation
    echo "<div class='step'>";
    echo "<h2>🎮 Step 4: Simulate Controller Update</h2>";
    
    try {
        // Simulate the update process like controller
        $tourism = ObjekWisata::first();
        
        if ($tourism && file_exists($testImagePath)) {
            echo "<div class='info'>📋 Simulating controller update process...</div>";
            
            // Create request-like data
            $uploadedFile = new \Illuminate\Http\UploadedFile(
                $testImagePath,
                'controller-test.png',
                'image/png',
                null,
                true
            );
            
            // Simulate controller logic
            $data = [];
            
            // Handle foto utama upload (like in controller)
            if ($uploadedFile->isValid()) {
                // Delete old photo if exists
                if ($tourism->foto_utama) {
                    $oldPath = storage_path('app/public/wisata/' . $tourism->foto_utama);
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                        echo "<div class='warning'>🗑️ Deleted old photo: {$tourism->foto_utama}</div>";
                    }
                }
                
                $uploadService = new FileUploadService();
                $uploadResult = $uploadService->uploadPhotoWithResize($uploadedFile, 'wisata');
                
                if ($uploadResult['success']) {
                    $data['foto_utama'] = $uploadResult['filename'];
                    echo "<div class='success'>✅ Controller simulation: Upload successful</div>";
                    echo "<div class='info'><strong>New filename:</strong> {$uploadResult['filename']}</div>";
                    
                    // Update tourism
                    $tourism->update($data);
                    echo "<div class='success'>✅ Tourism updated via controller simulation</div>";
                    
                } else {
                    echo "<div class='error'>❌ Controller simulation: Upload failed - {$uploadResult['error']}</div>";
                }
            }
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ Controller simulation error: " . $e->getMessage() . "</div>";
    }
    echo "</div>";
    
    // Step 5: Cleanup and final check
    echo "<div class='step'>";
    echo "<h2>🧹 Step 5: Final Check</h2>";
    
    try {
        $tourism = ObjekWisata::first();
        
        if ($tourism) {
            echo "<div class='info'><strong>Final tourism state:</strong></div>";
            echo "<div class='info'><strong>ID:</strong> {$tourism->id}</div>";
            echo "<div class='info'><strong>Name:</strong> {$tourism->nama_wisata}</div>";
            echo "<div class='info'><strong>Photo:</strong> " . ($tourism->foto_utama ?: 'NULL') . "</div>";
            
            if ($tourism->foto_utama) {
                $finalUrl = $tourism->foto_utama_url;
                echo "<div class='code'>{$finalUrl}</div>";
                echo "<img src='{$finalUrl}?v=" . time() . "' style='max-width: 200px; border: 1px solid #ddd; margin: 10px;' alt='Final State'>";
            }
        }
        
        // Cleanup test file
        if (file_exists($testImagePath)) {
            unlink($testImagePath);
            echo "<div class='info'>🧹 Cleaned up test file</div>";
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ Final check error: " . $e->getMessage() . "</div>";
    }
    echo "</div>";
    ?>
    
    <div class="step">
        <h2>📋 Conclusion</h2>
        <p><strong>If this test works but admin form doesn't:</strong></p>
        <ol>
            <li>Check form enctype="multipart/form-data"</li>
            <li>Verify input name="foto_utama" matches controller</li>
            <li>Check file input accept="image/*"</li>
            <li>Ensure CSRF token is present</li>
            <li>Check PHP upload limits</li>
            <li>Verify admin authentication</li>
        </ol>
        
        <p><strong>Test admin form:</strong></p>
        <ul>
            <li><a href="https://susukannet.id/admin/objek-wisata/1/edit" target="_blank">Edit Tourism Form</a></li>
            <li><a href="https://susukannet.id/admin/objek-wisata/create" target="_blank">Create Tourism Form</a></li>
        </ul>
    </div>
</body>
</html>
