<?php
/**
 * Fix Admin Authentication Issues
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Fix Admin Authentication</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1000px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .code { background: #f1f1f1; padding: 8px; border-radius: 4px; font-family: monospace; margin: 5px 0; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; margin: 5px; text-decoration: none; display: inline-block; }
        .btn-success { background: #28a745; }
        .btn-danger { background: #dc3545; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Fix Admin Authentication Issues</h1>
        
        <?php
        try {
            echo "<div class='section'>";
            echo "<h2>🔍 Current Admin Users Status</h2>";
            
            // Get all users
            $allUsers = User::all();
            echo "<div class='info'>📊 Total users in database: " . $allUsers->count() . "</div>";
            
            // Check admin users
            $adminUsers = User::whereIn('role', ['admin', 'superadmin', 'super_admin'])
                             ->orWhere('is_admin', true)
                             ->get();
            
            echo "<div class='info'>👥 Admin users found: " . $adminUsers->count() . "</div>";
            
            if ($adminUsers->count() === 0) {
                echo "<div class='warning'>⚠️ No admin users found! Creating default admin...</div>";
                
                // Create default admin
                $admin = User::create([
                    'name' => 'Administrator',
                    'email' => 'admin@aplikasidesa.com',
                    'password' => Hash::make('admin123'),
                    'role' => 'admin',
                    'is_admin' => true,
                    'is_active' => true,
                    'email_verified_at' => now(),
                ]);
                
                echo "<div class='success'>✅ Created default admin user</div>";
                echo "<div class='code'>";
                echo "Email: admin@aplikasidesa.com\n";
                echo "Password: admin123\n";
                echo "Role: admin\n";
                echo "Is Admin: YES\n";
                echo "Is Active: YES\n";
                echo "</div>";
                
            } else {
                echo "<div class='success'>✅ Admin users found</div>";
                
                foreach ($adminUsers as $user) {
                    echo "<div class='code'>";
                    echo "ID: {$user->id}\n";
                    echo "Name: {$user->name}\n";
                    echo "Email: {$user->email}\n";
                    echo "Role: {$user->role}\n";
                    echo "Is Admin: " . ($user->is_admin ? 'YES' : 'NO') . "\n";
                    echo "Is Active: " . ($user->is_active ? 'YES' : 'NO') . "\n";
                    echo "Can Access Admin: " . ($user->canAccessAdmin() ? 'YES' : 'NO') . "\n";
                    echo "</div>";
                    
                    // Fix user if needed
                    $needsUpdate = false;
                    $updates = [];
                    
                    if (!$user->is_admin) {
                        $user->is_admin = true;
                        $updates[] = 'is_admin = true';
                        $needsUpdate = true;
                    }
                    
                    if (!$user->is_active) {
                        $user->is_active = true;
                        $updates[] = 'is_active = true';
                        $needsUpdate = true;
                    }
                    
                    if (!in_array($user->role, ['admin', 'superadmin', 'super_admin'])) {
                        $user->role = 'admin';
                        $updates[] = 'role = admin';
                        $needsUpdate = true;
                    }
                    
                    if ($needsUpdate) {
                        $user->save();
                        echo "<div class='success'>✅ Fixed user: " . implode(', ', $updates) . "</div>";
                    } else {
                        echo "<div class='info'>✅ User is properly configured</div>";
                    }
                }
            }
            echo "</div>";
            
            // Test database connection
            echo "<div class='section'>";
            echo "<h2>🗄️ Database Connection Test</h2>";
            
            try {
                $userCount = DB::table('users')->count();
                echo "<div class='success'>✅ Database connection successful</div>";
                echo "<div class='info'>📊 Users table accessible, {$userCount} users found</div>";
                
                // Check if users table has required columns
                $columns = DB::getSchemaBuilder()->getColumnListing('users');
                $requiredColumns = ['id', 'name', 'email', 'password', 'role', 'is_admin', 'is_active'];
                $missingColumns = array_diff($requiredColumns, $columns);
                
                if (empty($missingColumns)) {
                    echo "<div class='success'>✅ All required columns present in users table</div>";
                } else {
                    echo "<div class='error'>❌ Missing columns: " . implode(', ', $missingColumns) . "</div>";
                }
                
            } catch (Exception $e) {
                echo "<div class='error'>❌ Database error: " . $e->getMessage() . "</div>";
            }
            echo "</div>";
            
            // Check session configuration
            echo "<div class='section'>";
            echo "<h2>⚙️ Session Configuration</h2>";
            
            $sessionConfig = [
                'driver' => config('session.driver'),
                'lifetime' => config('session.lifetime'),
                'domain' => config('session.domain'),
                'path' => config('session.path'),
                'secure' => config('session.secure'),
                'http_only' => config('session.http_only'),
                'same_site' => config('session.same_site'),
            ];
            
            echo "<div class='code'>";
            foreach ($sessionConfig as $key => $value) {
                $displayValue = is_bool($value) ? ($value ? 'YES' : 'NO') : $value;
                echo "Session {$key}: {$displayValue}\n";
            }
            echo "</div>";
            
            // Check if session directory is writable
            $sessionPath = config('session.files');
            if ($sessionPath && is_dir($sessionPath)) {
                if (is_writable($sessionPath)) {
                    echo "<div class='success'>✅ Session directory is writable: {$sessionPath}</div>";
                } else {
                    echo "<div class='error'>❌ Session directory is not writable: {$sessionPath}</div>";
                }
            }
            echo "</div>";
            
            // Test authentication flow
            echo "<div class='section'>";
            echo "<h2>🔐 Authentication Flow Test</h2>";
            
            $testUser = User::where('email', 'admin@aplikasidesa.com')->first();
            if ($testUser) {
                echo "<div class='info'>🧪 Testing authentication for: {$testUser->email}</div>";
                
                // Test password hash
                if (Hash::check('admin123', $testUser->password)) {
                    echo "<div class='success'>✅ Password hash is correct</div>";
                } else {
                    echo "<div class='error'>❌ Password hash is incorrect, updating...</div>";
                    $testUser->password = Hash::make('admin123');
                    $testUser->save();
                    echo "<div class='success'>✅ Password hash updated</div>";
                }
                
                // Test user methods
                echo "<div class='code'>";
                echo "isAdmin(): " . ($testUser->isAdmin() ? 'TRUE' : 'FALSE') . "\n";
                echo "isSuperAdmin(): " . ($testUser->isSuperAdmin() ? 'TRUE' : 'FALSE') . "\n";
                echo "canAccessAdmin(): " . ($testUser->canAccessAdmin() ? 'TRUE' : 'FALSE') . "\n";
                echo "</div>";
                
            } else {
                echo "<div class='error'>❌ Test user not found</div>";
            }
            echo "</div>";
            
            // Clear caches
            echo "<div class='section'>";
            echo "<h2>🧹 Clear Caches</h2>";
            
            try {
                // Clear config cache
                if (file_exists(storage_path('framework/cache/config.php'))) {
                    unlink(storage_path('framework/cache/config.php'));
                    echo "<div class='success'>✅ Config cache cleared</div>";
                }
                
                // Clear route cache
                if (file_exists(storage_path('framework/cache/routes-v7.php'))) {
                    unlink(storage_path('framework/cache/routes-v7.php'));
                    echo "<div class='success'>✅ Route cache cleared</div>";
                }
                
                // Clear view cache
                $viewCachePath = storage_path('framework/views');
                if (is_dir($viewCachePath)) {
                    $files = glob($viewCachePath . '/*');
                    foreach ($files as $file) {
                        if (is_file($file)) {
                            unlink($file);
                        }
                    }
                    echo "<div class='success'>✅ View cache cleared</div>";
                }
                
                echo "<div class='info'>🔄 All caches cleared successfully</div>";
                
            } catch (Exception $e) {
                echo "<div class='error'>❌ Cache clear error: " . $e->getMessage() . "</div>";
            }
            echo "</div>";
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Fatal error: " . $e->getMessage() . "</div>";
            echo "<div class='code'>Stack trace:\n" . $e->getTraceAsString() . "</div>";
        }
        ?>
        
        <div class="section">
            <h2>🔗 Next Steps</h2>
            <p><strong>After running this fix, try these steps:</strong></p>
            <ol>
                <li><strong>Test Authentication:</strong> <a href="https://susukannet.id/test-admin-auth.php" target="_blank" class="btn">Test Admin Auth</a></li>
                <li><strong>Login to Admin:</strong> <a href="https://susukannet.id/login" target="_blank" class="btn">Login Page</a></li>
                <li><strong>Access Tourism Edit:</strong> <a href="https://susukannet.id/admin/objek-wisata/1/edit" target="_blank" class="btn">Edit Tourism</a></li>
            </ol>
            
            <h3>🔑 Default Admin Credentials</h3>
            <div class="code">
                Email: admin@aplikasidesa.com
                Password: admin123
            </div>
            
            <h3>🔧 If Still Having Issues</h3>
            <ul>
                <li>Check Laravel logs in <code>storage/logs/laravel.log</code></li>
                <li>Verify session configuration in <code>.env</code></li>
                <li>Check file permissions on <code>storage/</code> directory</li>
                <li>Ensure database connection is working</li>
            </ul>
        </div>
    </div>
</body>
</html>
