<?php
/**
 * Fix All Image Issues for susukannet.id
 * Complete fix for tourism images on shared hosting
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ObjekWisata;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Fix All Image Issues - susukannet.id</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .step { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .code { background: #f1f1f1; padding: 8px; border-radius: 4px; font-family: monospace; margin: 5px 0; }
    </style>
</head>
<body>
    <h1>🔧 Fix All Image Issues</h1>
    
    <?php
    $baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
    echo "<p><strong>Current Domain:</strong> {$baseUrl}</p>";
    echo "<p><strong>Expected Domain:</strong> https://susukannet.id</p>";
    
    // Step 1: Create default tourism image
    echo "<div class='step'>";
    echo "<h2>📷 Step 1: Create Default Tourism Image</h2>";
    
    $defaultImagePath = __DIR__ . '/images/default-tourism.png';
    $bgImagePath = __DIR__ . '/images/bg-pettren.png';
    
    if (!file_exists($defaultImagePath)) {
        // Create a simple default image
        if (function_exists('imagecreate')) {
            $img = imagecreate(400, 300);
            $bg = imagecolorallocate($img, 240, 240, 240);
            $text_color = imagecolorallocate($img, 100, 100, 100);
            
            imagestring($img, 5, 120, 140, 'Default Tourism', $text_color);
            imagestring($img, 3, 150, 160, 'Image', $text_color);
            
            if (imagepng($img, $defaultImagePath)) {
                echo "<div class='success'>✅ Created default-tourism.png</div>";
            } else {
                echo "<div class='error'>❌ Failed to create default-tourism.png</div>";
            }
            imagedestroy($img);
        } elseif (file_exists($bgImagePath)) {
            // Copy from bg-pettren.png
            if (copy($bgImagePath, $defaultImagePath)) {
                echo "<div class='success'>✅ Copied bg-pettren.png to default-tourism.png</div>";
            } else {
                echo "<div class='error'>❌ Failed to copy bg-pettren.png</div>";
            }
        } else {
            echo "<div class='error'>❌ No way to create default image</div>";
        }
    } else {
        echo "<div class='info'>ℹ️ default-tourism.png already exists</div>";
    }
    
    // Verify file
    if (file_exists($defaultImagePath)) {
        $fileSize = filesize($defaultImagePath);
        echo "<div class='success'>✅ File verified: " . number_format($fileSize) . " bytes</div>";
    }
    echo "</div>";
    
    // Step 2: Clean database temporary paths
    echo "<div class='step'>";
    echo "<h2>🧹 Step 2: Clean Database Temporary Paths</h2>";
    
    try {
        $tempObjects = ObjekWisata::where('foto_utama', 'like', '%tmp%')
                                 ->orWhere('foto_utama', 'like', '%php%')
                                 ->get();
        
        $cleanedCount = 0;
        foreach ($tempObjects as $obj) {
            $oldPath = $obj->foto_utama;
            $obj->update(['foto_utama' => null]);
            echo "<div class='warning'>🧹 Cleaned {$obj->nama_wisata}: {$oldPath} → NULL</div>";
            $cleanedCount++;
        }
        
        if ($cleanedCount > 0) {
            echo "<div class='success'>✅ Cleaned {$cleanedCount} temporary paths</div>";
        } else {
            echo "<div class='info'>ℹ️ No temporary paths found</div>";
        }
        
        // Clean gallery arrays
        $galleryObjects = ObjekWisata::whereNotNull('galeri_foto')->get();
        $galleryCleaned = 0;
        
        foreach ($galleryObjects as $obj) {
            $galeri = $obj->galeri_foto;
            if (is_array($galeri)) {
                $cleaned = array_filter($galeri, function($foto) {
                    return is_string($foto) && !empty($foto) && 
                           strpos($foto, 'tmp') === false && 
                           strpos($foto, 'php') === false;
                });
                
                if (count($cleaned) !== count($galeri)) {
                    $obj->update(['galeri_foto' => array_values($cleaned)]);
                    echo "<div class='warning'>🧹 Cleaned gallery for {$obj->nama_wisata}</div>";
                    $galleryCleaned++;
                }
            }
        }
        
        if ($galleryCleaned > 0) {
            echo "<div class='success'>✅ Cleaned {$galleryCleaned} gallery arrays</div>";
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ Database Error: " . $e->getMessage() . "</div>";
    }
    echo "</div>";
    
    // Step 3: Create sample images in wisata directory
    echo "<div class='step'>";
    echo "<h2>🖼️ Step 3: Create Sample Tourism Images</h2>";
    
    $wisataDir = __DIR__ . '/storage/wisata';
    if (!is_dir($wisataDir)) {
        if (mkdir($wisataDir, 0755, true)) {
            echo "<div class='success'>✅ Created wisata directory</div>";
        } else {
            echo "<div class='error'>❌ Failed to create wisata directory</div>";
        }
    }
    
    // Create sample images
    $sampleImages = ['sample-tourism-1.png', 'sample-tourism-2.png'];
    
    foreach ($sampleImages as $filename) {
        $filepath = $wisataDir . '/' . $filename;
        
        if (!file_exists($filepath)) {
            if (function_exists('imagecreate')) {
                $img = imagecreate(600, 400);
                $bg = imagecolorallocate($img, rand(200, 255), rand(200, 255), rand(200, 255));
                $text_color = imagecolorallocate($img, 50, 50, 50);
                
                imagestring($img, 5, 200, 180, 'Sample Tourism', $text_color);
                imagestring($img, 3, 250, 200, $filename, $text_color);
                
                if (imagepng($img, $filepath)) {
                    echo "<div class='success'>✅ Created {$filename}</div>";
                } else {
                    echo "<div class='error'>❌ Failed to create {$filename}</div>";
                }
                imagedestroy($img);
            } elseif (file_exists($defaultImagePath)) {
                // Copy from default image
                if (copy($defaultImagePath, $filepath)) {
                    echo "<div class='success'>✅ Copied default to {$filename}</div>";
                } else {
                    echo "<div class='error'>❌ Failed to copy to {$filename}</div>";
                }
            }
        } else {
            echo "<div class='info'>ℹ️ {$filename} already exists</div>";
        }
    }
    echo "</div>";
    
    // Step 4: Update tourism objects with sample images
    echo "<div class='step'>";
    echo "<h2>🎯 Step 4: Update Tourism Objects</h2>";
    
    try {
        $tourismWithoutImages = ObjekWisata::whereNull('foto_utama')->limit(2)->get();
        
        foreach ($tourismWithoutImages as $index => $tourism) {
            $sampleImage = $sampleImages[$index] ?? $sampleImages[0];
            $tourism->update(['foto_utama' => $sampleImage]);
            echo "<div class='success'>✅ Updated {$tourism->nama_wisata} with {$sampleImage}</div>";
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ Update Error: " . $e->getMessage() . "</div>";
    }
    echo "</div>";
    
    // Step 5: Test URLs
    echo "<div class='step'>";
    echo "<h2>🧪 Step 5: Test Generated URLs</h2>";
    
    try {
        $testTourism = ObjekWisata::first();
        if ($testTourism) {
            echo "<div class='info'><strong>Test Tourism:</strong> {$testTourism->nama_wisata}</div>";
            echo "<div class='info'><strong>foto_utama:</strong> " . ($testTourism->foto_utama ?: 'NULL') . "</div>";
            
            $generatedUrl = $testTourism->foto_utama_url;
            echo "<div class='code'><strong>Generated URL:</strong> {$generatedUrl}</div>";
            
            // Check if URL is using correct domain
            if (strpos($generatedUrl, 'susukannet.id') !== false) {
                echo "<div class='success'>✅ URL uses correct domain (susukannet.id)</div>";
            } else {
                echo "<div class='warning'>⚠️ URL using wrong domain: {$generatedUrl}</div>";
            }
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ URL Test Error: " . $e->getMessage() . "</div>";
    }
    echo "</div>";
    
    // Step 6: Summary and next steps
    echo "<div class='step'>";
    echo "<h2>📋 Step 6: Summary</h2>";
    
    $defaultExists = file_exists($defaultImagePath);
    $wisataDirExists = is_dir($wisataDir);
    $wisataFileCount = $wisataDirExists ? count(glob($wisataDir . '/*')) : 0;
    
    echo "<div class='info'>";
    echo "<strong>File Status:</strong><br>";
    echo "• Default image: " . ($defaultExists ? '✅ EXISTS' : '❌ MISSING') . "<br>";
    echo "• Wisata directory: " . ($wisataDirExists ? '✅ EXISTS' : '❌ MISSING') . "<br>";
    echo "• Wisata files: {$wisataFileCount}<br>";
    echo "</div>";
    
    echo "<h3>🎯 Next Steps:</h3>";
    echo "<ol>";
    echo "<li>Clear Laravel caches: <code>php artisan config:clear && php artisan view:clear</code></li>";
    echo "<li>Test admin page: <a href='{$baseUrl}/admin/objek-wisata' target='_blank'>Admin Tourism</a></li>";
    echo "<li>Test public page: <a href='{$baseUrl}/wisata' target='_blank'>Public Tourism</a></li>";
    echo "<li>Test images: <a href='{$baseUrl}/test-live-images.php' target='_blank'>Image Test</a></li>";
    echo "</ol>";
    
    echo "<h3>🔧 Manual Fixes if Needed:</h3>";
    echo "<ul>";
    echo "<li>Run sync command: <code>php artisan shared-hosting:sync-files --force</code></li>";
    echo "<li>Check .env APP_URL setting</li>";
    echo "<li>Verify file permissions (755 for directories, 644 for files)</li>";
    echo "</ul>";
    echo "</div>";
    ?>
    
</body>
</html>
