<?php
/**
 * Script untuk memperbaiki data UMKM dan masalah URL
 * Akses via: https://susukannet.id/fix-umkm-data.php
 */

// Include Laravel bootstrap
require_once __DIR__ . '/../vendor/autoload.php';
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\Umkm;
use App\Models\Penduduk;
use Illuminate\Support\Facades\DB;

echo "<h1>🔧 Fix UMKM Data Script</h1>";

try {
    // 1. Check if we have penduduk data
    echo "<h2>👥 Checking Penduduk Data</h2>";
    $pendudukCount = Penduduk::count();
    echo "<div class='info'>📋 Total Penduduk: {$pendudukCount}</div>";
    
    $penduduk = null;
    if ($pendudukCount > 0) {
        $penduduk = Penduduk::first();
        echo "<div class='success'>✅ Using penduduk: {$penduduk->nama} (ID: {$penduduk->id})</div>";
    } else {
        echo "<div class='error'>❌ No penduduk found. Creating sample penduduk...</div>";
        
        // Create sample penduduk
        $penduduk = new Penduduk();
        $penduduk->nama = 'Sample Penduduk';
        $penduduk->nik = '1234567890123456';
        $penduduk->tempat_lahir = 'Sample City';
        $penduduk->tanggal_lahir = '1990-01-01';
        $penduduk->jenis_kelamin = 'L';
        $penduduk->alamat = 'Sample Address';
        $penduduk->rt = '01';
        $penduduk->rw = '01';
        $penduduk->desa = 'Sample Village';
        $penduduk->kecamatan = 'Sample District';
        $penduduk->kabupaten = 'Sample Regency';
        $penduduk->provinsi = 'Sample Province';
        $penduduk->agama = 'Islam';
        $penduduk->status_perkawinan = 'Belum Kawin';
        $penduduk->pekerjaan = 'Wiraswasta';
        $penduduk->kewarganegaraan = 'WNI';
        $penduduk->save();
        
        echo "<div class='success'>✅ Sample penduduk created with ID: {$penduduk->id}</div>";
    }
    
    echo "<hr>";
    
    // 2. Create sample UMKM data
    echo "<h2>🏢 Creating Sample UMKM Data</h2>";
    
    $sampleUmkmData = [
        [
            'nama_umkm' => 'Warung Makan Sederhana',
            'kategori' => 'makanan_minuman',
            'deskripsi' => 'Warung makan yang menyajikan makanan tradisional dengan harga terjangkau.',
            'alamat' => 'Jl. Raya Desa No. 123',
            'skala_usaha' => 'mikro',
            'status' => 'aktif',
            'logo' => 'umkm/logo/sample-logo.png',
            'galeri_foto' => ['umkm/galeri/sample-gallery1.png'],
            'nomor_telepon' => '081234567890',
            'email' => 'warung@example.com',
            'jam_operasional' => ['Senin-Minggu: 06:00-22:00'],
            'hari_libur' => [],
            'is_verified' => true,
            'aktif_marketplace' => false,
            'bisa_cod' => true,
            'bisa_transfer' => true
        ],
        [
            'nama_umkm' => 'Kerajinan Tangan Kreatif',
            'kategori' => 'kerajinan_tangan',
            'deskripsi' => 'Membuat berbagai kerajinan tangan dari bahan daur ulang yang unik dan menarik.',
            'alamat' => 'Jl. Kerajinan No. 456',
            'skala_usaha' => 'kecil',
            'status' => 'aktif',
            'logo' => 'umkm/logo/sample-logo.png',
            'galeri_foto' => ['umkm/galeri/sample-gallery1.png'],
            'nomor_telepon' => '081234567891',
            'email' => 'kerajinan@example.com',
            'jam_operasional' => ['Senin-Jumat: 08:00-17:00'],
            'hari_libur' => ['Sabtu', 'Minggu'],
            'is_verified' => true,
            'aktif_marketplace' => true,
            'bisa_cod' => true,
            'bisa_transfer' => true
        ],
        [
            'nama_umkm' => 'Jasa Service Elektronik',
            'kategori' => 'jasa_layanan',
            'deskripsi' => 'Menyediakan jasa service dan perbaikan berbagai peralatan elektronik.',
            'alamat' => 'Jl. Service No. 789',
            'skala_usaha' => 'mikro',
            'status' => 'aktif',
            'logo' => 'umkm/logo/sample-logo.png',
            'galeri_foto' => ['umkm/galeri/sample-gallery1.png'],
            'nomor_telepon' => '081234567892',
            'email' => 'service@example.com',
            'jam_operasional' => ['Senin-Sabtu: 09:00-18:00'],
            'hari_libur' => ['Minggu'],
            'is_verified' => false,
            'aktif_marketplace' => false,
            'bisa_cod' => false,
            'bisa_transfer' => true
        ]
    ];
    
    $createdCount = 0;
    foreach ($sampleUmkmData as $data) {
        try {
            $umkm = new Umkm();
            $umkm->fill($data);
            $umkm->pemilik_id = $penduduk->id;
            $umkm->user_id = 1; // Assuming admin user ID is 1
            $umkm->slug = \Illuminate\Support\Str::slug($data['nama_umkm']);
            $umkm->save();
            
            $createdCount++;
            echo "<div class='success'>✅ Created UMKM: {$data['nama_umkm']} (ID: {$umkm->id})</div>";
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Failed to create UMKM {$data['nama_umkm']}: " . $e->getMessage() . "</div>";
        }
    }
    
    echo "<div class='info'>📋 Total UMKM created: {$createdCount}</div>";
    
    echo "<hr>";
    
    // 3. Test the created UMKM
    echo "<h2>🧪 Testing Created UMKM</h2>";
    
    $umkmList = Umkm::all();
    echo "<div class='info'>📋 Total UMKM in database: " . $umkmList->count() . "</div>";
    
    foreach ($umkmList as $umkm) {
        echo "<h3>Testing: {$umkm->nama_umkm}</h3>";
        
        try {
            $logoUrl = $umkm->logo_url;
            echo "<div class='success'>✅ Logo URL: <a href='{$logoUrl}' target='_blank'>{$logoUrl}</a></div>";
        } catch (Exception $e) {
            echo "<div class='error'>❌ Logo URL Error: " . $e->getMessage() . "</div>";
        }
        
        try {
            $galleryUrls = $umkm->galeri_foto_urls;
            echo "<div class='success'>✅ Gallery URLs: " . count($galleryUrls) . " items</div>";
            
            foreach ($galleryUrls as $index => $url) {
                echo "<div class='info'>📋 Gallery {$index}: <a href='{$url}' target='_blank'>{$url}</a></div>";
            }
        } catch (Exception $e) {
            echo "<div class='error'>❌ Gallery URLs Error: " . $e->getMessage() . "</div>";
        }
        
        echo "<br>";
    }
    
    echo "<hr>";
    
    // 4. Fix ImageHelper URL issue
    echo "<h2>🔧 Fixing ImageHelper URL Issue</h2>";
    
    // The issue is that ImageHelper is using request()->getSchemeAndHttpHost() 
    // which returns the current domain, but we want it to use susukannet.id
    
    echo "<div class='info'>📋 Current request domain: " . (request() ? request()->getSchemeAndHttpHost() : 'No request') . "</div>";
    echo "<div class='info'>📋 App URL config: " . config('app.url') . "</div>";
    
    // Update ImageHelper to use consistent domain
    $imageHelperPath = app_path('Helpers/ImageHelper.php');
    $imageHelperContent = file_get_contents($imageHelperPath);
    
    // Replace the dynamic domain with susukannet.id
    $newContent = str_replace(
        "request() ? request()->getSchemeAndHttpHost() : 'http://127.0.0.1:8000'",
        "'https://susukannet.id'",
        $imageHelperContent
    );
    
    if ($newContent !== $imageHelperContent) {
        file_put_contents($imageHelperPath, $newContent);
        echo "<div class='success'>✅ ImageHelper updated to use susukannet.id domain</div>";
    } else {
        echo "<div class='info'>📋 ImageHelper already using correct domain</div>";
    }
    
    echo "<hr>";
    
    // 5. Final test
    echo "<h2>🎯 Final Test</h2>";
    
    $testUmkm = Umkm::first();
    if ($testUmkm) {
        $logoUrl = $testUmkm->logo_url;
        $galleryUrls = $testUmkm->galeri_foto_urls;
        
        echo "<div class='success'>✅ Final Logo URL: <a href='{$logoUrl}' target='_blank'>{$logoUrl}</a></div>";
        echo "<div class='success'>✅ Final Gallery URLs: " . count($galleryUrls) . " items</div>";
        
        // Check if URLs use correct domain
        if (strpos($logoUrl, 'susukannet.id') !== false) {
            echo "<div class='success'>✅ Logo URL uses correct domain (susukannet.id)</div>";
        } else {
            echo "<div class='error'>❌ Logo URL uses wrong domain</div>";
        }
    }
    
} catch (Exception $e) {
    echo "<div class='error'>❌ Error: " . $e->getMessage() . "</div>";
    echo "<div class='error'>❌ File: " . $e->getFile() . " Line: " . $e->getLine() . "</div>";
    echo "<pre>" . $e->getTraceAsString() . "</pre>";
}

echo "<h2>✅ Fix Completed</h2>";
echo "<p><a href='/admin/umkm'>Kembali ke Admin UMKM</a></p>";
echo "<p><a href='/debug-umkm.php'>Run Debug Again</a></p>";

?>
<style>
body { font-family: Arial, sans-serif; margin: 20px; }
.success { color: green; font-weight: bold; background: #d4edda; padding: 5px; border-radius: 3px; margin: 2px 0; }
.error { color: red; font-weight: bold; background: #f8d7da; padding: 5px; border-radius: 3px; margin: 2px 0; }
.info { color: blue; background: #d1ecf1; padding: 5px; border-radius: 3px; margin: 2px 0; }
pre { background: #f8f9fa; padding: 10px; border-radius: 3px; overflow-x: auto; }
</style>
