<?php
/**
 * Simple Image Creator for susukannet.id
 * Creates tourism images without complex checks
 */

// Create directories if they don't exist
$dirs = [
    __DIR__ . '/storage',
    __DIR__ . '/storage/wisata',
    __DIR__ . '/images'
];

foreach ($dirs as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

echo "<h1>Creating Tourism Images...</h1>";

// Create default tourism image
$defaultPath = __DIR__ . '/images/default-tourism.png';
if (!file_exists($defaultPath)) {
    if (extension_loaded('gd')) {
        $img = imagecreate(400, 300);
        $bg = imagecolorallocate($img, 240, 248, 255);
        $text = imagecolorallocate($img, 70, 130, 180);
        
        imagefill($img, 0, 0, $bg);
        imagestring($img, 5, 120, 130, 'Default Tourism', $text);
        imagestring($img, 5, 160, 160, 'Image', $text);
        
        if (imagepng($img, $defaultPath)) {
            echo "<p>✅ Created default-tourism.png</p>";
        }
        imagedestroy($img);
    } else {
        // Create empty file if GD not available
        file_put_contents($defaultPath, '');
        echo "<p>⚠️ Created empty default-tourism.png (GD not available)</p>";
    }
} else {
    echo "<p>ℹ️ default-tourism.png already exists</p>";
}

// Create sample tourism images
$samples = [
    'sample-tourism-1.png' => [52, 152, 219], // Blue
    'sample-tourism-2.png' => [46, 204, 113]  // Green
];

foreach ($samples as $filename => $color) {
    $filepath = __DIR__ . '/storage/wisata/' . $filename;
    
    if (!file_exists($filepath)) {
        if (extension_loaded('gd')) {
            $img = imagecreate(600, 400);
            $bg = imagecolorallocate($img, $color[0], $color[1], $color[2]);
            $white = imagecolorallocate($img, 255, 255, 255);
            
            imagefill($img, 0, 0, $bg);
            imagestring($img, 5, 200, 180, 'Sample Tourism', $white);
            imagestring($img, 5, 250, 210, substr($filename, 0, -4), $white);
            
            if (imagepng($img, $filepath)) {
                echo "<p>✅ Created {$filename}</p>";
            }
            imagedestroy($img);
        } else {
            // Create empty file if GD not available
            file_put_contents($filepath, '');
            echo "<p>⚠️ Created empty {$filename} (GD not available)</p>";
        }
    } else {
        echo "<p>ℹ️ {$filename} already exists</p>";
    }
}

echo "<h2>Testing URLs:</h2>";
$baseUrl = 'https://susukannet.id';
$testUrls = [
    $baseUrl . '/images/default-tourism.png',
    $baseUrl . '/storage/wisata/sample-tourism-1.png',
    $baseUrl . '/storage/wisata/sample-tourism-2.png'
];

foreach ($testUrls as $url) {
    echo "<p><a href='{$url}' target='_blank'>{$url}</a></p>";
    echo "<img src='{$url}?v=" . time() . "' style='max-width:200px; border:1px solid #ddd; margin:5px;' alt='Test Image'><br>";
}

// Clean database if Laravel is available
try {
    if (file_exists(__DIR__ . '/../vendor/autoload.php')) {
        require_once __DIR__ . '/../vendor/autoload.php';
        $app = require_once __DIR__ . '/../bootstrap/app.php';
        $app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();
        
        echo "<h2>Database Cleanup:</h2>";
        
        // Update tourism objects with temp paths
        $updated = \App\Models\ObjekWisata::where('foto_utama', 'like', '%tmp%')
                                        ->orWhere('foto_utama', 'like', '%php%')
                                        ->update(['foto_utama' => 'sample-tourism-1.png']);
        
        if ($updated > 0) {
            echo "<p>✅ Updated {$updated} tourism objects</p>";
        } else {
            echo "<p>ℹ️ No temporary paths found to clean</p>";
        }
    }
} catch (Exception $e) {
    echo "<p>⚠️ Laravel not available: " . $e->getMessage() . "</p>";
}

echo "<h2>✅ Complete!</h2>";
echo "<p>Now test: <a href='{$baseUrl}/admin/objek-wisata' target='_blank'>Admin Tourism Page</a></p>";
?>
