<?php
/**
 * Sync Tourism Images for Shared Hosting
 * This script ensures wisata images work properly on shared hosting
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ObjekWisata;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Sync Wisata Images - Shared Hosting</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 900px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .step { margin: 15px 0; padding: 15px; border-left: 4px solid #007bff; background: #f8f9fa; }
        .code { background: #f1f1f1; padding: 8px; border-radius: 4px; font-family: monospace; margin: 5px 0; }
        .test-image { max-width: 100px; height: 100px; object-fit: cover; border: 1px solid #ddd; margin: 5px; }
        .grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(120px, 1fr)); gap: 10px; margin: 10px 0; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 Sync Wisata Images - Shared Hosting</h1>
        <p><strong>Server:</strong> <?= $_SERVER['HTTP_HOST'] ?? 'localhost' ?></p>
        <p><strong>Script Path:</strong> <?= __FILE__ ?></p>
        
        <?php
        $baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
        echo "<p><strong>Base URL:</strong> {$baseUrl}</p>";
        
        echo "<div class='step'>";
        echo "<h2>📁 Step 1: Directory Setup</h2>";
        
        // Create necessary directories
        $directories = [
            public_path('storage'),
            public_path('storage/wisata'),
            public_path('storage/wisata/galeri'),
            public_path('storage/penduduk'), // For comparison
            storage_path('app/public/wisata'),
            storage_path('app/public/wisata/galeri')
        ];
        
        foreach ($directories as $dir) {
            if (!is_dir($dir)) {
                if (mkdir($dir, 0755, true)) {
                    echo "<div class='success'>✅ Created: {$dir}</div>";
                } else {
                    echo "<div class='error'>❌ Failed to create: {$dir}</div>";
                }
            } else {
                echo "<div class='info'>📁 Exists: {$dir}</div>";
            }
        }
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h2>🔗 Step 2: Symlink Status</h2>";
        
        $publicStorage = public_path('storage');
        $storagePublic = storage_path('app/public');
        
        if (is_link($publicStorage)) {
            $linkTarget = readlink($publicStorage);
            echo "<div class='info'>🔗 Symlink exists: {$publicStorage} → {$linkTarget}</div>";
            
            if (realpath($linkTarget) === realpath($storagePublic)) {
                echo "<div class='success'>✅ Symlink is correct</div>";
            } else {
                echo "<div class='warning'>⚠️ Symlink target is incorrect</div>";
            }
        } else {
            echo "<div class='warning'>⚠️ No symlink detected (typical for shared hosting)</div>";
            echo "<div class='info'>📝 Using manual file sync for shared hosting</div>";
        }
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h2>📋 Step 3: Scan Tourism Data</h2>";
        
        try {
            $tourismObjects = ObjekWisata::all();
            echo "<div class='success'>✅ Found " . $tourismObjects->count() . " tourism objects</div>";
            
            $withImages = $tourismObjects->where('foto_utama', '!=', null)->where('foto_utama', '!=', '');
            echo "<div class='info'>📸 Tourism objects with images: " . $withImages->count() . "</div>";
            
            // Show sample data
            foreach ($tourismObjects->take(3) as $tourism) {
                echo "<div class='code'>";
                echo "<strong>ID {$tourism->id}:</strong> {$tourism->nama_wisata}<br>";
                echo "<strong>foto_utama:</strong> " . ($tourism->foto_utama ?? 'NULL') . "<br>";
                echo "<strong>galeri_foto:</strong> " . (is_array($tourism->galeri_foto) ? implode(', ', $tourism->galeri_foto) : ($tourism->galeri_foto ?? 'NULL'));
                echo "</div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Database Error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h2>🔄 Step 4: Sync Images from Storage to Public</h2>";
        
        $syncCount = 0;
        $sourceDir = storage_path('app/public/wisata');
        $targetDir = public_path('storage/wisata');
        
        if (is_dir($sourceDir)) {
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($sourceDir, RecursiveDirectoryIterator::SKIP_DOTS)
            );
            
            foreach ($iterator as $file) {
                $relativePath = str_replace($sourceDir . DIRECTORY_SEPARATOR, '', $file->getPathname());
                $targetPath = $targetDir . DIRECTORY_SEPARATOR . $relativePath;
                $targetDirPath = dirname($targetPath);
                
                // Create target directory if needed
                if (!is_dir($targetDirPath)) {
                    mkdir($targetDirPath, 0755, true);
                }
                
                // Copy file if newer or doesn't exist
                if (!file_exists($targetPath) || filemtime($file->getPathname()) > filemtime($targetPath)) {
                    if (copy($file->getPathname(), $targetPath)) {
                        echo "<div class='success'>✅ Synced: {$relativePath}</div>";
                        $syncCount++;
                    } else {
                        echo "<div class='error'>❌ Failed to sync: {$relativePath}</div>";
                    }
                } else {
                    echo "<div class='info'>⏭️ Up-to-date: {$relativePath}</div>";
                }
            }
            
            echo "<div class='success'>📊 Sync complete! {$syncCount} files synced.</div>";
        } else {
            echo "<div class='warning'>⚠️ Source directory not found: {$sourceDir}</div>";
        }
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h2>🧪 Step 5: Test Image URLs</h2>";
        
        // Test default images
        $testImages = [
            'Default Tourism' => '/images/default-tourism.png',
            'Sample Tourism 1' => '/storage/wisata/sample-tourism-1.png',
            'Sample Tourism 2' => '/storage/wisata/sample-tourism-2.png'
        ];
        
        echo "<div class='grid'>";
        foreach ($testImages as $name => $path) {
            $fullUrl = $baseUrl . $path;
            echo "<div>";
            echo "<strong>{$name}:</strong><br>";
            echo "<img src='{$fullUrl}' class='test-image' alt='{$name}' onerror=\"this.style.display='none'; this.nextElementSibling.style.display='block';\">";
            echo "<div style='display:none; color: red; font-size: 12px;'>❌ Not found</div>";
            echo "<div class='code' style='font-size: 10px; word-break: break-all;'>{$fullUrl}</div>";
            echo "</div>";
        }
        echo "</div>";
        
        // Test actual tourism images
        echo "<h3>📷 Actual Tourism Images:</h3>";
        echo "<div class='grid'>";
        
        foreach ($withImages->take(6) as $tourism) {
            if ($tourism->foto_utama) {
                $imageUrl = $baseUrl . '/storage/wisata/' . $tourism->foto_utama;
                echo "<div>";
                echo "<strong>{$tourism->nama_wisata}:</strong><br>";
                echo "<img src='{$imageUrl}' class='test-image' alt='{$tourism->nama_wisata}' onerror=\"this.style.display='none'; this.nextElementSibling.style.display='block';\">";
                echo "<div style='display:none; color: red; font-size: 12px;'>❌ Not found</div>";
                echo "<div class='code' style='font-size: 10px; word-break: break-all;'>{$imageUrl}</div>";
                echo "</div>";
            }
        }
        echo "</div>";
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h2>🔧 Step 6: Update Database URLs</h2>";
        
        // Clean any temporary paths
        $tempPathObjects = ObjekWisata::where('foto_utama', 'like', '%tmp%')
                                    ->orWhere('foto_utama', 'like', '%php%')
                                    ->get();
        
        if ($tempPathObjects->count() > 0) {
            foreach ($tempPathObjects as $obj) {
                $oldPath = $obj->foto_utama;
                $obj->update(['foto_utama' => null]);
                echo "<div class='warning'>🧹 Cleaned temporary path for {$obj->nama_wisata}: {$oldPath}</div>";
            }
        } else {
            echo "<div class='success'>✅ No temporary paths found in database</div>";
        }
        
        // Clean gallery
        $galleryObjects = ObjekWisata::whereNotNull('galeri_foto')->get();
        foreach ($galleryObjects as $obj) {
            $galeri = $obj->galeri_foto;
            if (is_array($galeri)) {
                $cleaned = array_filter($galeri, function($foto) {
                    return is_string($foto) && !empty($foto) && 
                           strpos($foto, 'tmp') === false && 
                           strpos($foto, 'php') === false;
                });
                
                if (count($cleaned) !== count($galeri)) {
                    $obj->update(['galeri_foto' => array_values($cleaned)]);
                    echo "<div class='warning'>🧹 Cleaned gallery for {$obj->nama_wisata}</div>";
                }
            }
        }
        echo "</div>";
        
        echo "<div class='step'>";
        echo "<h2>✅ Step 7: Verification</h2>";
        
        echo "<div class='success'>🎉 Sync process completed!</div>";
        echo "<div class='info'>📝 Next steps:</div>";
        echo "<ul>";
        echo "<li>✅ Directories created and synced</li>";
        echo "<li>✅ Database cleaned of temporary paths</li>";
        echo "<li>✅ Image URLs tested</li>";
        echo "<li>📋 Visit <a href='/admin/objek-wisata' target='_blank'>Admin Tourism Page</a> to verify</li>";
        echo "<li>📋 Check <a href='/wisata' target='_blank'>Public Tourism Page</a> for frontend display</li>";
        echo "</ul>";
        
        echo "<div class='code'>";
        echo "<strong>Sample URLs to test manually:</strong><br>";
        echo "• {$baseUrl}/storage/wisata/sample-tourism-1.png<br>";
        echo "• {$baseUrl}/admin/objek-wisata<br>";
        echo "• {$baseUrl}/wisata<br>";
        echo "</div>";
        echo "</div>";
        
        // Add some diagnostic info
        echo "<div class='step'>";
        echo "<h2>🔍 Diagnostic Info</h2>";
        echo "<div class='code'>";
        echo "<strong>PHP Version:</strong> " . PHP_VERSION . "<br>";
        echo "<strong>Laravel Version:</strong> " . app()->version() . "<br>";
        echo "<strong>Storage Path:</strong> " . storage_path('app/public') . "<br>";
        echo "<strong>Public Path:</strong> " . public_path('storage') . "<br>";
        echo "<strong>Base URL:</strong> " . $baseUrl . "<br>";
        echo "<strong>Document Root:</strong> " . ($_SERVER['DOCUMENT_ROOT'] ?? 'N/A') . "<br>";
        echo "</div>";
        echo "</div>";
        ?>
    </div>
</body>
</html>
