<?php
/**
 * Test Admin Authentication & Login Process
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Test Admin Authentication</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1000px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .form-control { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; margin: 5px 0; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; margin: 5px; }
        .btn-success { background: #28a745; }
        .btn-danger { background: #dc3545; }
        .test-form { background: #f8f9ff; border: 2px solid #007bff; padding: 20px; border-radius: 8px; margin: 20px 0; }
        .code { background: #f1f1f1; padding: 8px; border-radius: 4px; font-family: monospace; margin: 5px 0; }
        .user-list { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 15px; }
        .user-card { background: white; border: 1px solid #ddd; padding: 15px; border-radius: 8px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔐 Test Admin Authentication & Login</h1>
        
        <?php
        // Handle login test
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['test_login'])) {
            echo "<div class='section'>";
            echo "<h2>🧪 Login Test Results</h2>";
            
            try {
                $email = $_POST['email'];
                $password = $_POST['password'];
                
                echo "<div class='info'>🔍 Testing login for: {$email}</div>";
                
                // Find user
                $user = User::where('email', $email)->first();
                
                if (!$user) {
                    echo "<div class='error'>❌ User not found with email: {$email}</div>";
                } else {
                    echo "<div class='success'>✅ User found: {$user->name}</div>";
                    echo "<div class='info'>📊 User details:</div>";
                    echo "<div class='code'>";
                    echo "ID: {$user->id}\n";
                    echo "Name: {$user->name}\n";
                    echo "Email: {$user->email}\n";
                    echo "Role: {$user->role}\n";
                    echo "Is Admin: " . ($user->is_admin ? 'YES' : 'NO') . "\n";
                    echo "Is Active: " . ($user->is_active ? 'YES' : 'NO') . "\n";
                    echo "Can Access Admin: " . ($user->canAccessAdmin() ? 'YES' : 'NO') . "\n";
                    echo "Is Super Admin: " . ($user->isSuperAdmin() ? 'YES' : 'NO') . "\n";
                    echo "</div>";
                    
                    // Test password
                    if (Hash::check($password, $user->password)) {
                        echo "<div class='success'>✅ Password is correct</div>";
                        
                        // Test authentication
                        if (Auth::attempt(['email' => $email, 'password' => $password])) {
                            echo "<div class='success'>✅ Authentication successful</div>";
                            
                            $authenticatedUser = Auth::user();
                            echo "<div class='info'>🔐 Authenticated as: {$authenticatedUser->name}</div>";
                            
                            // Test admin access
                            if ($authenticatedUser->canAccessAdmin()) {
                                echo "<div class='success'>✅ User can access admin panel</div>";
                                
                                // Test session
                                $sessionId = Session::getId();
                                echo "<div class='info'>📝 Session ID: {$sessionId}</div>";
                                
                                // Test middleware
                                echo "<div class='info'>🛡️ Middleware tests:</div>";
                                echo "<div class='code'>";
                                echo "Auth::check(): " . (Auth::check() ? 'TRUE' : 'FALSE') . "\n";
                                echo "User role: {$authenticatedUser->role}\n";
                                echo "isAdmin(): " . ($authenticatedUser->isAdmin() ? 'TRUE' : 'FALSE') . "\n";
                                echo "canAccessAdmin(): " . ($authenticatedUser->canAccessAdmin() ? 'TRUE' : 'FALSE') . "\n";
                                echo "</div>";
                                
                            } else {
                                echo "<div class='error'>❌ User cannot access admin panel</div>";
                                echo "<div class='warning'>⚠️ Check: is_active={$user->is_active}, is_admin={$user->is_admin}, role={$user->role}</div>";
                            }
                            
                        } else {
                            echo "<div class='error'>❌ Authentication failed</div>";
                        }
                        
                    } else {
                        echo "<div class='error'>❌ Password is incorrect</div>";
                    }
                }
                
            } catch (Exception $e) {
                echo "<div class='error'>❌ Exception: " . $e->getMessage() . "</div>";
            }
            echo "</div>";
        }
        
        // Handle create admin
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_admin'])) {
            echo "<div class='section'>";
            echo "<h2>👤 Create Admin User</h2>";
            
            try {
                $name = $_POST['admin_name'];
                $email = $_POST['admin_email'];
                $password = $_POST['admin_password'];
                
                // Check if user exists
                $existingUser = User::where('email', $email)->first();
                if ($existingUser) {
                    echo "<div class='warning'>⚠️ User with email {$email} already exists</div>";
                    
                    // Update to admin
                    $existingUser->role = 'admin';
                    $existingUser->is_admin = true;
                    $existingUser->is_active = true;
                    $existingUser->password = Hash::make($password);
                    $existingUser->save();
                    
                    echo "<div class='success'>✅ Updated existing user to admin</div>";
                } else {
                    // Create new admin user
                    $user = User::create([
                        'name' => $name,
                        'email' => $email,
                        'password' => Hash::make($password),
                        'role' => 'admin',
                        'is_admin' => true,
                        'is_active' => true,
                        'email_verified_at' => now(),
                    ]);
                    
                    echo "<div class='success'>✅ Created new admin user: {$user->name}</div>";
                }
                
                echo "<div class='info'>📧 Email: {$email}</div>";
                echo "<div class='info'>🔑 Password: {$password}</div>";
                
            } catch (Exception $e) {
                echo "<div class='error'>❌ Exception: " . $e->getMessage() . "</div>";
            }
            echo "</div>";
        }
        
        // Show current authentication status
        echo "<div class='section'>";
        echo "<h2>🔍 Current Authentication Status</h2>";
        
        if (Auth::check()) {
            $user = Auth::user();
            echo "<div class='success'>✅ User is authenticated</div>";
            echo "<div class='info'>👤 Logged in as: {$user->name} ({$user->email})</div>";
            echo "<div class='info'>🎭 Role: {$user->role}</div>";
            echo "<div class='info'>🛡️ Can access admin: " . ($user->canAccessAdmin() ? 'YES' : 'NO') . "</div>";
            echo "<div class='info'>📝 Session ID: " . Session::getId() . "</div>";
        } else {
            echo "<div class='warning'>⚠️ No user is authenticated</div>";
        }
        echo "</div>";
        
        // Show all admin users
        echo "<div class='section'>";
        echo "<h2>👥 All Admin Users</h2>";
        
        $adminUsers = User::whereIn('role', ['admin', 'superadmin', 'super_admin'])
                         ->orWhere('is_admin', true)
                         ->get();
        
        echo "<div class='info'>📊 Found " . $adminUsers->count() . " admin users</div>";
        
        if ($adminUsers->count() > 0) {
            echo "<div class='user-list'>";
            foreach ($adminUsers as $user) {
                echo "<div class='user-card'>";
                echo "<h4>{$user->name}</h4>";
                echo "<p><strong>Email:</strong> {$user->email}</p>";
                echo "<p><strong>Role:</strong> {$user->role}</p>";
                echo "<p><strong>Is Admin:</strong> " . ($user->is_admin ? 'YES' : 'NO') . "</p>";
                echo "<p><strong>Is Active:</strong> " . ($user->is_active ? 'YES' : 'NO') . "</p>";
                echo "<p><strong>Can Access Admin:</strong> " . ($user->canAccessAdmin() ? 'YES' : 'NO') . "</p>";
                echo "<p><strong>Created:</strong> {$user->created_at}</p>";
                echo "</div>";
            }
            echo "</div>";
        } else {
            echo "<div class='warning'>⚠️ No admin users found. You need to create one.</div>";
        }
        echo "</div>";
        
        // Check session configuration
        echo "<div class='section'>";
        echo "<h2>⚙️ Session Configuration</h2>";
        
        echo "<div class='code'>";
        echo "Session Driver: " . config('session.driver') . "\n";
        echo "Session Lifetime: " . config('session.lifetime') . " minutes\n";
        echo "Session Domain: " . config('session.domain') . "\n";
        echo "Session Path: " . config('session.path') . "\n";
        echo "Session Secure: " . (config('session.secure') ? 'YES' : 'NO') . "\n";
        echo "Session HTTP Only: " . (config('session.http_only') ? 'YES' : 'NO') . "\n";
        echo "Session Same Site: " . config('session.same_site') . "\n";
        echo "</div>";
        echo "</div>";
        ?>
        
        <!-- Login Test Form -->
        <div class="test-form">
            <h2>🧪 Test Login</h2>
            <p><strong>Test authentication with existing admin user:</strong></p>
            
            <form method="POST">
                <input type="hidden" name="test_login" value="1">
                
                <div>
                    <label><strong>Email:</strong></label>
                    <input type="email" name="email" class="form-control" value="admin@aplikasidesa.com" required>
                </div>
                
                <div>
                    <label><strong>Password:</strong></label>
                    <input type="password" name="password" class="form-control" value="admin123" required>
                </div>
                
                <button type="submit" class="btn btn-success">🔐 Test Login</button>
            </form>
        </div>
        
        <!-- Create Admin Form -->
        <div class="test-form">
            <h2>👤 Create/Update Admin User</h2>
            <p><strong>Create a new admin user or update existing one:</strong></p>
            
            <form method="POST">
                <input type="hidden" name="create_admin" value="1">
                
                <div>
                    <label><strong>Name:</strong></label>
                    <input type="text" name="admin_name" class="form-control" value="Test Admin" required>
                </div>
                
                <div>
                    <label><strong>Email:</strong></label>
                    <input type="email" name="admin_email" class="form-control" value="admin@test.com" required>
                </div>
                
                <div>
                    <label><strong>Password:</strong></label>
                    <input type="password" name="admin_password" class="form-control" value="admin123" required>
                </div>
                
                <button type="submit" class="btn btn-success">👤 Create/Update Admin</button>
            </form>
        </div>
        
        <div class="section">
            <h2>🔗 Admin Panel Links</h2>
            <p><strong>After authentication, try these links:</strong></p>
            <ul>
                <li><a href="https://susukannet.id/login" target="_blank">Login Page</a></li>
                <li><a href="https://susukannet.id/admin/dashboard" target="_blank">Admin Dashboard</a></li>
                <li><a href="https://susukannet.id/admin/objek-wisata" target="_blank">Tourism Index</a></li>
                <li><a href="https://susukannet.id/admin/objek-wisata/1/edit" target="_blank">Edit Tourism</a></li>
            </ul>
            
            <h3>🔧 Troubleshooting Steps</h3>
            <ol>
                <li><strong>If no admin users:</strong> Create one using the form above</li>
                <li><strong>If login fails:</strong> Check email/password combination</li>
                <li><strong>If redirect to login:</strong> Check user role and is_active status</li>
                <li><strong>If session issues:</strong> Check session configuration</li>
                <li><strong>If middleware blocks:</strong> Check AdminMiddleware and AdminSessionRefresh</li>
            </ol>
        </div>
    </div>
</body>
</html>
