<?php
/**
 * Live Image Access Test for susukannet.id
 * Test all image URLs and accessibility on shared hosting
 */

header('Content-Type: text/html; charset=utf-8');

// Load Laravel if available
$laravelLoaded = false;
$tourism = null;

try {
    if (file_exists(__DIR__ . '/../vendor/autoload.php')) {
        require_once __DIR__ . '/../vendor/autoload.php';
        $app = require_once __DIR__ . '/../bootstrap/app.php';
        $app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();
        $laravelLoaded = true;
        
        // Get sample tourism data
        $tourism = \App\Models\ObjekWisata::first();
    }
} catch (Exception $e) {
    $laravelError = $e->getMessage();
}

$baseUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'];
$currentTime = date('Y-m-d H:i:s');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Live Image Access Test - <?= $_SERVER['HTTP_HOST'] ?></title>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <style>
        body { 
            font-family: Arial, sans-serif; 
            margin: 20px; 
            background: #f5f5f5; 
            line-height: 1.6;
        }
        .container { 
            max-width: 1200px; 
            margin: 0 auto; 
            background: white; 
            padding: 20px; 
            border-radius: 8px; 
            box-shadow: 0 2px 10px rgba(0,0,0,0.1); 
        }
        .test-section { 
            margin: 25px 0; 
            padding: 20px; 
            border: 1px solid #ddd; 
            border-radius: 8px; 
            background: #f9f9f9; 
        }
        .test-image { 
            max-width: 200px; 
            height: 150px; 
            object-fit: cover; 
            border: 2px solid #ddd; 
            margin: 10px; 
            border-radius: 4px;
        }
        .test-image.loaded { border-color: #28a745; }
        .test-image.error { border-color: #dc3545; }
        .url-box { 
            font-family: monospace; 
            background: #f1f1f1; 
            padding: 8px; 
            border-radius: 4px; 
            margin: 5px 0; 
            word-break: break-all; 
            font-size: 12px;
        }
        .status { 
            padding: 10px; 
            margin: 10px 0; 
            border-radius: 4px; 
            font-weight: bold;
        }
        .success { background: #d4edda; color: #155724; }
        .error { background: #f8d7da; color: #721c24; }
        .warning { background: #fff3cd; color: #856404; }
        .info { background: #d1ecf1; color: #0c5460; }
        .grid { 
            display: grid; 
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)); 
            gap: 15px; 
            margin: 15px 0;
        }
        .test-item {
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 15px;
            background: white;
            text-align: center;
        }
        .test-counter {
            position: fixed;
            top: 10px;
            right: 10px;
            background: #007bff;
            color: white;
            padding: 10px;
            border-radius: 4px;
            font-weight: bold;
            z-index: 1000;
        }
    </style>
    <script>
        let loadedCount = 0;
        let errorCount = 0;
        let totalImages = 0;
        
        function updateCounter() {
            const counter = document.getElementById('counter');
            if (counter) {
                counter.innerHTML = `✅ ${loadedCount} | ❌ ${errorCount} | 📊 ${totalImages}`;
            }
        }
        
        function imageLoaded(img) {
            img.classList.add('loaded');
            img.classList.remove('error');
            loadedCount++;
            updateCounter();
            console.log('Image loaded:', img.src);
        }
        
        function imageError(img) {
            img.classList.add('error');
            img.classList.remove('loaded');
            errorCount++;
            updateCounter();
            console.log('Image error:', img.src);
        }
        
        function testUrl(url, description) {
            return fetch(url, { method: 'HEAD' })
                .then(response => {
                    console.log(`${description}: ${response.status} - ${url}`);
                    return response.ok;
                })
                .catch(error => {
                    console.log(`${description}: ERROR - ${url}`, error);
                    return false;
                });
        }
        
        window.onload = function() {
            totalImages = document.querySelectorAll('.test-image').length;
            updateCounter();
            
            // Test URLs programmatically
            const urls = [
                '<?= $baseUrl ?>/images/default-tourism.png',
                '<?= $baseUrl ?>/storage/wisata/sample-tourism-1.png',
                '<?= $baseUrl ?>/storage/wisata/sample-tourism-2.png'
            ];
            
            urls.forEach(url => {
                testUrl(url, 'Direct test');
            });
        };
    </script>
</head>
<body>
    <div id="counter" class="test-counter">Loading...</div>
    
    <div class="container">
        <h1>🧪 Live Image Access Test</h1>
        
        <div class="status info">
            <strong>Server:</strong> <?= $_SERVER['HTTP_HOST'] ?><br>
            <strong>Time:</strong> <?= $currentTime ?><br>
            <strong>Base URL:</strong> <?= $baseUrl ?><br>
            <strong>Laravel:</strong> <?= $laravelLoaded ? '✅ Loaded' : '❌ Not loaded' ?>
            <?php if (isset($laravelError)): ?>
                <br><strong>Laravel Error:</strong> <?= htmlspecialchars($laravelError) ?>
            <?php endif; ?>
        </div>

        <!-- Default Image Test -->
        <div class="test-section">
            <h2>🖼️ Default Tourism Image Test</h2>
            <?php
            $defaultImageUrl = $baseUrl . '/images/default-tourism.png';
            $defaultImagePath = __DIR__ . '/images/default-tourism.png';
            $imageExists = file_exists($defaultImagePath);
            ?>
            
            <div class="status <?= $imageExists ? 'success' : 'error' ?>">
                File exists on server: <?= $imageExists ? '✅ YES' : '❌ NO' ?>
            </div>
            
            <div class="url-box"><?= $defaultImageUrl ?></div>
            <img src="<?= $defaultImageUrl ?>?v=<?= time() ?>" 
                 class="test-image" 
                 alt="Default Tourism Image"
                 onload="imageLoaded(this)" 
                 onerror="imageError(this)">
            
            <p><a href="<?= $defaultImageUrl ?>" target="_blank">Open in new tab</a></p>
        </div>

        <!-- Storage Images Test -->
        <div class="test-section">
            <h2>📁 Storage Images Test</h2>
            <?php
            $storageImages = [
                'sample-tourism-1.png',
                'sample-tourism-2.png'
            ];
            
            foreach ($storageImages as $filename) {
                $url = $baseUrl . '/storage/wisata/' . $filename;
                $path = __DIR__ . '/storage/wisata/' . $filename;
                $exists = file_exists($path);
                
                echo "<div class='test-item'>";
                echo "<h4>{$filename}</h4>";
                echo "<div class='status " . ($exists ? 'success' : 'error') . "'>";
                echo "File exists: " . ($exists ? '✅ YES' : '❌ NO');
                echo "</div>";
                echo "<div class='url-box'>{$url}</div>";
                echo "<img src='{$url}?v=" . time() . "' class='test-image' alt='{$filename}' onload='imageLoaded(this)' onerror='imageError(this)'>";
                echo "<p><a href='{$url}' target='_blank'>Open in new tab</a></p>";
                echo "</div>";
            }
            ?>
        </div>

        <?php if ($laravelLoaded && $tourism): ?>
        <!-- Laravel Tourism Data Test -->
        <div class="test-section">
            <h2>🎯 Laravel Tourism Data Test</h2>
            <div class="status success">
                <strong>Tourism Object:</strong> <?= htmlspecialchars($tourism->nama_wisata) ?><br>
                <strong>ID:</strong> <?= $tourism->id ?><br>
                <strong>Status:</strong> <?= $tourism->status ?><br>
                <strong>foto_utama:</strong> <?= $tourism->foto_utama ?: 'NULL' ?>
            </div>
            
            <?php
            try {
                $generatedUrl = $tourism->foto_utama_url;
                echo "<div class='url-box'>Generated URL: {$generatedUrl}</div>";
                echo "<img src='{$generatedUrl}?v=" . time() . "' class='test-image' alt='{$tourism->nama_wisata}' onload='imageLoaded(this)' onerror='imageError(this)'>";
                echo "<p><a href='{$generatedUrl}' target='_blank'>Open generated URL</a></p>";
            } catch (Exception $e) {
                echo "<div class='status error'>Error generating URL: " . htmlspecialchars($e->getMessage()) . "</div>";
            }
            ?>
        </div>

        <!-- Multiple Tourism Objects -->
        <div class="test-section">
            <h2>📋 Multiple Tourism Objects</h2>
            <div class="grid">
                <?php
                try {
                    $allTourism = \App\Models\ObjekWisata::limit(6)->get();
                    
                    foreach ($allTourism as $item) {
                        echo "<div class='test-item'>";
                        echo "<h4>" . htmlspecialchars($item->nama_wisata) . "</h4>";
                        echo "<p><strong>ID:</strong> {$item->id}</p>";
                        echo "<p><strong>foto_utama:</strong> " . ($item->foto_utama ?: 'NULL') . "</p>";
                        
                        try {
                            $itemUrl = $item->foto_utama_url;
                            echo "<div class='url-box' style='font-size: 10px;'>{$itemUrl}</div>";
                            echo "<img src='{$itemUrl}?v=" . time() . "' class='test-image' style='width: 100px; height: 75px;' alt='{$item->nama_wisata}' onload='imageLoaded(this)' onerror='imageError(this)'>";
                        } catch (Exception $e) {
                            echo "<div class='status error'>URL Error: " . htmlspecialchars($e->getMessage()) . "</div>";
                        }
                        
                        echo "</div>";
                    }
                } catch (Exception $e) {
                    echo "<div class='status error'>Database Error: " . htmlspecialchars($e->getMessage()) . "</div>";
                }
                ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Directory Check -->
        <div class="test-section">
            <h2>🗂️ Directory Check</h2>
            <?php
            $directories = [
                'public/images' => __DIR__ . '/images',
                'public/storage' => __DIR__ . '/storage',
                'public/storage/wisata' => __DIR__ . '/storage/wisata'
            ];
            
            foreach ($directories as $name => $path) {
                $exists = is_dir($path);
                $readable = $exists && is_readable($path);
                $fileCount = $exists ? count(glob($path . '/*')) : 0;
                
                echo "<div class='status " . ($exists ? 'success' : 'error') . "'>";
                echo "<strong>{$name}:</strong> ";
                echo ($exists ? '✅ EXISTS' : '❌ NOT FOUND') . " | ";
                echo ($readable ? '📖 READABLE' : '🔒 NOT READABLE') . " | ";
                echo "📁 {$fileCount} files<br>";
                echo "<small>{$path}</small>";
                echo "</div>";
            }
            ?>
        </div>

        <!-- Quick Actions -->
        <div class="test-section">
            <h2>⚡ Quick Actions</h2>
            <p><strong>Test these URLs manually:</strong></p>
            <ul>
                <li><a href="<?= $baseUrl ?>/admin/objek-wisata" target="_blank">Admin Tourism Page</a></li>
                <li><a href="<?= $baseUrl ?>/wisata" target="_blank">Public Tourism Page</a></li>
                <li><a href="<?= $baseUrl ?>/images/default-tourism.png" target="_blank">Default Image Direct</a></li>
                <li><a href="<?= $baseUrl ?>/storage/wisata/" target="_blank">Storage Directory</a></li>
            </ul>
            
            <p><strong>Debug Commands:</strong></p>
            <ul>
                <li><code>php artisan shared-hosting:sync-files --force</code></li>
                <li><code>php artisan config:clear && php artisan view:clear</code></li>
                <li><code>php artisan storage:link</code></li>
            </ul>
        </div>

        <!-- Results Summary -->
        <div class="test-section">
            <h2>📊 Test Results Summary</h2>
            <div id="summary">
                <p>Test in progress... Check console for detailed logs.</p>
                <p>Green border = Image loaded successfully ✅</p>
                <p>Red border = Image failed to load ❌</p>
            </div>
        </div>
    </div>

    <script>
        // Additional summary after page load
        setTimeout(() => {
            const summary = document.getElementById('summary');
            summary.innerHTML = `
                <div class="status ${loadedCount > errorCount ? 'success' : 'warning'}">
                    <strong>Final Results:</strong><br>
                    ✅ Images loaded: ${loadedCount}<br>
                    ❌ Images failed: ${errorCount}<br>
                    📊 Total images: ${totalImages}<br>
                    📈 Success rate: ${totalImages > 0 ? Math.round((loadedCount / totalImages) * 100) : 0}%
                </div>
            `;
        }, 3000);
    </script>
</body>
</html>
