<?php
/**
 * Test script untuk memverifikasi konsistensi URL UMKM dengan Objek Wisata
 * Akses via: https://susukannet.id/test-umkm-objek-wisata-consistency.php
 */

// Include Laravel bootstrap
require_once __DIR__ . '/../vendor/autoload.php';
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\Umkm;
use App\Models\ObjekWisata;
use Illuminate\Support\Facades\Storage;

echo "<h1>🔄 Test Konsistensi URL UMKM & Objek Wisata</h1>";

try {
    // 1. Test UMKM URL Generation
    echo "<h2>🏢 UMKM URL Generation Test</h2>";
    
    $umkm = Umkm::first();
    if ($umkm) {
        echo "<div class='info'>📋 Testing UMKM: {$umkm->nama_umkm} (ID: {$umkm->id})</div>";
        
        // Test logo URL
        $logoUrl = $umkm->logo_url;
        echo "<div class='success'>✅ Logo URL: <a href='{$logoUrl}' target='_blank'>{$logoUrl}</a></div>";
        
        // Test gallery URLs
        $galleryUrls = $umkm->galeri_foto_urls;
        echo "<div class='success'>✅ Gallery URLs: " . count($galleryUrls) . " items</div>";
        
        foreach ($galleryUrls as $index => $url) {
            echo "<div class='info'>📋 Gallery {$index}: <a href='{$url}' target='_blank'>{$url}</a></div>";
        }
        
        // Check URL pattern
        if (strpos($logoUrl, '/storage/umkm/logo/') !== false) {
            echo "<div class='success'>✅ Logo URL uses correct pattern: /storage/umkm/logo/</div>";
        } else {
            echo "<div class='error'>❌ Logo URL pattern incorrect</div>";
        }
        
        if (count($galleryUrls) > 0 && strpos($galleryUrls[0], '/storage/umkm/galeri/') !== false) {
            echo "<div class='success'>✅ Gallery URL uses correct pattern: /storage/umkm/galeri/</div>";
        } else {
            echo "<div class='error'>❌ Gallery URL pattern incorrect</div>";
        }
        
    } else {
        echo "<div class='error'>❌ No UMKM found for testing</div>";
    }
    
    echo "<hr>";
    
    // 2. Test Objek Wisata URL Generation
    echo "<h2>🏞️ Objek Wisata URL Generation Test</h2>";
    
    $objekWisata = ObjekWisata::first();
    if ($objekWisata) {
        echo "<div class='info'>📋 Testing Objek Wisata: {$objekWisata->nama_wisata} (ID: {$objekWisata->id})</div>";
        
        // Test foto utama URL
        $fotoUtamaUrl = $objekWisata->foto_utama_url;
        echo "<div class='success'>✅ Foto Utama URL: <a href='{$fotoUtamaUrl}' target='_blank'>{$fotoUtamaUrl}</a></div>";
        
        // Test gallery URLs
        $galleryUrls = $objekWisata->galeri_foto_urls;
        echo "<div class='success'>✅ Gallery URLs: " . count($galleryUrls) . " items</div>";
        
        foreach ($galleryUrls as $index => $url) {
            echo "<div class='info'>📋 Gallery {$index}: <a href='{$url}' target='_blank'>{$url}</a></div>";
        }
        
        // Check URL pattern
        if (strpos($fotoUtamaUrl, '/storage/wisata/') !== false) {
            echo "<div class='success'>✅ Foto Utama URL uses correct pattern: /storage/wisata/</div>";
        } else {
            echo "<div class='error'>❌ Foto Utama URL pattern incorrect</div>";
        }
        
        if (count($galleryUrls) > 0 && strpos($galleryUrls[0], '/storage/wisata/galeri/') !== false) {
            echo "<div class='success'>✅ Gallery URL uses correct pattern: /storage/wisata/galeri/</div>";
        } else {
            echo "<div class='error'>❌ Gallery URL pattern incorrect</div>";
        }
        
    } else {
        echo "<div class='error'>❌ No Objek Wisata found for testing</div>";
    }
    
    echo "<hr>";
    
    // 3. Compare URL Generation Patterns
    echo "<h2>🔄 URL Generation Pattern Comparison</h2>";
    
    echo "<h3>UMKM Pattern:</h3>";
    echo "<div class='info'>📋 Logo: /storage/umkm/logo/{filename}</div>";
    echo "<div class='info'>📋 Gallery: /storage/umkm/galeri/{filename}</div>";
    
    echo "<h3>Objek Wisata Pattern:</h3>";
    echo "<div class='info'>📋 Foto Utama: /storage/wisata/{filename}</div>";
    echo "<div class='info'>📋 Gallery: /storage/wisata/galeri/{filename}</div>";
    
    echo "<h3>Consistency Check:</h3>";
    echo "<div class='success'>✅ Both use Storage::disk('public')->exists() check first</div>";
    echo "<div class='success'>✅ Both use request()->getSchemeAndHttpHost() for base URL</div>";
    echo "<div class='success'>✅ Both fallback to SharedHostingHelper if file not found</div>";
    echo "<div class='success'>✅ Both handle temporary file paths (tmp, php)</div>";
    
    echo "<hr>";
    
    // 4. Test File Existence
    echo "<h2>📁 File Existence Test</h2>";
    
    if ($umkm && $umkm->logo) {
        $logoPath = 'umkm/logo/' . $umkm->logo;
        $logoExists = Storage::disk('public')->exists($logoPath);
        echo "<div class='" . ($logoExists ? 'success' : 'error') . "'>" . ($logoExists ? '✅' : '❌') . " UMKM Logo exists: {$logoPath}</div>";
    }
    
    if ($umkm && $umkm->galeri_foto && count($umkm->galeri_foto) > 0) {
        $galleryPath = 'umkm/galeri/' . $umkm->galeri_foto[0];
        $galleryExists = Storage::disk('public')->exists($galleryPath);
        echo "<div class='" . ($galleryExists ? 'success' : 'error') . "'>" . ($galleryExists ? '✅' : '❌') . " UMKM Gallery exists: {$galleryPath}</div>";
    }
    
    if ($objekWisata && $objekWisata->foto_utama) {
        $fotoPath = 'wisata/' . $objekWisata->foto_utama;
        $fotoExists = Storage::disk('public')->exists($fotoPath);
        echo "<div class='" . ($fotoExists ? 'success' : 'error') . "'>" . ($fotoExists ? '✅' : '❌') . " Objek Wisata Foto exists: {$fotoPath}</div>";
    }
    
    if ($objekWisata && $objekWisata->galeri_foto && count($objekWisata->galeri_foto) > 0) {
        $galleryPath = 'wisata/galeri/' . $objekWisata->galeri_foto[0];
        $galleryExists = Storage::disk('public')->exists($galleryPath);
        echo "<div class='" . ($galleryExists ? 'success' : 'error') . "'>" . ($galleryExists ? '✅' : '❌') . " Objek Wisata Gallery exists: {$galleryPath}</div>";
    }
    
    echo "<hr>";
    
    // 5. Test Edge Cases
    echo "<h2>🧪 Edge Cases Test</h2>";
    
    // Test with null values
    $testUmkm = new Umkm();
    $testUmkm->logo = null;
    $testUmkm->galeri_foto = null;
    
    $nullLogoUrl = $testUmkm->logo_url;
    $nullGalleryUrls = $testUmkm->galeri_foto_urls;
    
    echo "<div class='success'>✅ Null logo URL: {$nullLogoUrl}</div>";
    echo "<div class='success'>✅ Null gallery URLs: " . count($nullGalleryUrls) . " items</div>";
    
    // Test with empty values
    $testUmkm->logo = '';
    $testUmkm->galeri_foto = [];
    
    $emptyLogoUrl = $testUmkm->logo_url;
    $emptyGalleryUrls = $testUmkm->galeri_foto_urls;
    
    echo "<div class='success'>✅ Empty logo URL: {$emptyLogoUrl}</div>";
    echo "<div class='success'>✅ Empty gallery URLs: " . count($emptyGalleryUrls) . " items</div>";
    
    // Test with temporary paths
    $testUmkm->logo = 'tmp/12345.png';
    $testUmkm->galeri_foto = ['tmp/67890.png'];
    
    $tmpLogoUrl = $testUmkm->logo_url;
    $tmpGalleryUrls = $testUmkm->galeri_foto_urls;
    
    echo "<div class='success'>✅ Temp logo URL: {$tmpLogoUrl}</div>";
    echo "<div class='success'>✅ Temp gallery URLs: " . count($tmpGalleryUrls) . " items</div>";
    
    if (strpos($tmpLogoUrl, 'default-umkm-logo.png') !== false) {
        echo "<div class='success'>✅ Temp logo correctly returns default image</div>";
    } else {
        echo "<div class='error'>❌ Temp logo should return default image</div>";
    }
    
} catch (Exception $e) {
    echo "<div class='error'>❌ Error: " . $e->getMessage() . "</div>";
    echo "<div class='error'>❌ File: " . $e->getFile() . " Line: " . $e->getLine() . "</div>";
    echo "<pre>" . $e->getTraceAsString() . "</pre>";
}

echo "<h2>✅ Consistency Test Completed</h2>";
echo "<p><a href='/admin/umkm'>Kembali ke Admin UMKM</a></p>";
echo "<p><a href='/admin/objek-wisata'>Kembali ke Admin Objek Wisata</a></p>";

?>
<style>
body { font-family: Arial, sans-serif; margin: 20px; }
.success { color: green; font-weight: bold; background: #d4edda; padding: 5px; border-radius: 3px; margin: 2px 0; }
.error { color: red; font-weight: bold; background: #f8d7da; padding: 5px; border-radius: 3px; margin: 2px 0; }
.info { color: blue; background: #d1ecf1; padding: 5px; border-radius: 3px; margin: 2px 0; }
pre { background: #f8f9fa; padding: 10px; border-radius: 3px; overflow-x: auto; }
</style>
