<?php
/**
 * Test File Upload for Tourism Images
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ObjekWisata;
use App\Models\User;
use App\Services\FileUploadService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Test File Upload - Tourism</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .test-section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .form-group { margin: 15px 0; }
        .form-control { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; }
        .btn:hover { background: #0056b3; }
        .url-box { font-family: monospace; background: #f1f1f1; padding: 8px; border-radius: 4px; margin: 5px 0; word-break: break-all; }
        .test-image { max-width: 200px; height: 150px; object-fit: cover; border: 2px solid #ddd; margin: 10px; border-radius: 4px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🧪 Test File Upload - Tourism</h1>
        
        <?php
        $baseUrl = 'https://susukannet.id';
        echo "<p><strong>Base URL:</strong> {$baseUrl}</p>";
        echo "<p><strong>Time:</strong> " . date('Y-m-d H:i:s') . "</p>";
        
        // Handle form submission
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['test_image'])) {
            echo "<div class='test-section'>";
            echo "<h2>📤 Upload Test Results</h2>";
            
            try {
                $file = $_FILES['test_image'];
                
                if ($file['error'] === UPLOAD_ERR_OK) {
                    echo "<div class='info'>📁 File uploaded: {$file['name']} ({$file['size']} bytes)</div>";
                    
                    // Test FileUploadService
                    $uploadService = new FileUploadService();
                    
                    // Create a temporary UploadedFile object
                    $uploadedFile = new \Illuminate\Http\UploadedFile(
                        $file['tmp_name'],
                        $file['name'],
                        $file['type'],
                        $file['error'],
                        true
                    );
                    
                    $result = $uploadService->uploadPhotoWithResize($uploadedFile, 'wisata');
                    
                    if ($result['success']) {
                        echo "<div class='success'>✅ Upload successful!</div>";
                        echo "<div class='info'><strong>Filename:</strong> {$result['filename']}</div>";
                        echo "<div class='info'><strong>Path:</strong> {$result['path']}</div>";
                        
                        // Test URL generation
                        $imageUrl = $baseUrl . '/storage/wisata/' . $result['filename'];
                        echo "<div class='url-box'>{$imageUrl}</div>";
                        echo "<img src='{$imageUrl}?v=" . time() . "' class='test-image' alt='Uploaded Image'>";
                        
                        // Update a test tourism object
                        $tourism = ObjekWisata::first();
                        if ($tourism) {
                            $tourism->update(['foto_utama' => $result['filename']]);
                            echo "<div class='success'>✅ Updated tourism '{$tourism->nama_wisata}' with new image</div>";
                        }
                        
                    } else {
                        echo "<div class='error'>❌ Upload failed: {$result['error']}</div>";
                    }
                    
                } else {
                    echo "<div class='error'>❌ File upload error: " . $file['error'] . "</div>";
                }
                
            } catch (Exception $e) {
                echo "<div class='error'>❌ Exception: " . $e->getMessage() . "</div>";
            }
            echo "</div>";
        }
        
        // Test 1: Check Upload Service
        echo "<div class='test-section'>";
        echo "<h2>🔧 Test 1: Upload Service Check</h2>";
        
        try {
            $uploadService = new FileUploadService();
            echo "<div class='success'>✅ FileUploadService loaded successfully</div>";
            
            // Check methods
            $methods = ['uploadPhotoWithResize', 'uploadPhoto', 'uploadDocument'];
            foreach ($methods as $method) {
                if (method_exists($uploadService, $method)) {
                    echo "<div class='info'>📋 Method {$method}: ✅ EXISTS</div>";
                } else {
                    echo "<div class='error'>📋 Method {$method}: ❌ NOT FOUND</div>";
                }
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ FileUploadService error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Test 2: Check Directories
        echo "<div class='test-section'>";
        echo "<h2>📁 Test 2: Directory Check</h2>";
        
        $directories = [
            'storage/app/public' => storage_path('app/public'),
            'storage/app/public/wisata' => storage_path('app/public/wisata'),
            'public/storage' => public_path('storage'),
            'public/storage/wisata' => public_path('storage/wisata')
        ];
        
        foreach ($directories as $name => $path) {
            $exists = is_dir($path);
            $writable = $exists && is_writable($path);
            
            echo "<div class='" . ($exists ? 'success' : 'error') . "'>";
            echo "<strong>{$name}:</strong> " . ($exists ? '✅ EXISTS' : '❌ NOT FOUND');
            if ($exists) {
                echo " | " . ($writable ? '✅ WRITABLE' : '❌ NOT WRITABLE');
            }
            echo "</div>";
        }
        echo "</div>";
        
        // Test 3: Current Tourism Data
        echo "<div class='test-section'>";
        echo "<h2>🎯 Test 3: Current Tourism Data</h2>";
        
        try {
            $tourism = ObjekWisata::first();
            
            if ($tourism) {
                echo "<div class='success'>✅ Found: {$tourism->nama_wisata}</div>";
                echo "<div class='info'><strong>Current foto_utama:</strong> " . ($tourism->foto_utama ?: 'NULL') . "</div>";
                
                if ($tourism->foto_utama) {
                    $currentUrl = $tourism->foto_utama_url;
                    echo "<div class='url-box'>{$currentUrl}</div>";
                    echo "<img src='{$currentUrl}?v=" . time() . "' class='test-image' alt='Current Image'>";
                }
                
            } else {
                echo "<div class='warning'>⚠️ No tourism data found</div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Tourism data error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Test 4: PHP Upload Settings
        echo "<div class='test-section'>";
        echo "<h2>⚙️ Test 4: PHP Upload Settings</h2>";
        
        $settings = [
            'file_uploads' => ini_get('file_uploads'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'post_max_size' => ini_get('post_max_size'),
            'max_execution_time' => ini_get('max_execution_time'),
            'memory_limit' => ini_get('memory_limit')
        ];
        
        foreach ($settings as $name => $value) {
            echo "<div class='info'><strong>{$name}:</strong> {$value}</div>";
        }
        echo "</div>";
        ?>
        
        <!-- Upload Test Form -->
        <div class="test-section">
            <h2>📤 Test 5: Manual Upload Test</h2>
            <form method="POST" enctype="multipart/form-data">
                <div class="form-group">
                    <label for="test_image">Choose Image File:</label>
                    <input type="file" name="test_image" id="test_image" class="form-control" accept="image/*" required>
                    <small>Format: JPG, PNG. Max 2MB</small>
                </div>
                <button type="submit" class="btn">Upload Test Image</button>
            </form>
        </div>
        
        <div class="test-section">
            <h2>🔗 Admin Panel Links</h2>
            <p><strong>Test these after upload:</strong></p>
            <ul>
                <li><a href="<?= $baseUrl ?>/admin/objek-wisata" target="_blank">Tourism Index</a></li>
                <li><a href="<?= $baseUrl ?>/admin/objek-wisata/create" target="_blank">Create Tourism</a></li>
                <?php if (isset($tourism)): ?>
                <li><a href="<?= $baseUrl ?>/admin/objek-wisata/<?= $tourism->id ?>/edit" target="_blank">Edit Tourism</a></li>
                <?php endif; ?>
            </ul>
        </div>
        
        <div class="test-section">
            <h2>📋 Troubleshooting</h2>
            <p><strong>If upload still fails:</strong></p>
            <ol>
                <li>Check file permissions: 755 for directories, 644 for files</li>
                <li>Verify storage disk configuration in config/filesystems.php</li>
                <li>Check .env file for correct storage settings</li>
                <li>Ensure FileUploadService is working correctly</li>
                <li>Test with smaller image files (&lt;1MB)</li>
            </ol>
        </div>
    </div>
</body>
</html>
