<?php
/**
 * Upload and Sync Files for susukannet.id
 * This script helps sync files to the correct locations on shared hosting
 */

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Upload & Sync Files - susukannet.id</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .step { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .code { background: #f1f1f1; padding: 8px; border-radius: 4px; font-family: monospace; margin: 5px 0; }
        .cmd { background: #2d3748; color: #e2e8f0; padding: 10px; border-radius: 4px; font-family: monospace; margin: 10px 0; }
    </style>
</head>
<body>
    <h1>📤 Upload & Sync Files</h1>
    
    <?php
    echo "<div class='info'>";
    echo "<strong>Server Path:</strong> " . __DIR__ . "<br>";
    echo "<strong>Document Root:</strong> " . ($_SERVER['DOCUMENT_ROOT'] ?? 'Unknown') . "<br>";
    echo "<strong>Current URL:</strong> " . (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
    echo "</div>";
    
    // Step 1: Create directory structure
    echo "<div class='step'>";
    echo "<h2>📁 Step 1: Create Directory Structure</h2>";
    
    $directories = [
        __DIR__ . '/images',
        __DIR__ . '/storage',
        __DIR__ . '/storage/wisata',
        __DIR__ . '/storage/wisata/galeri',
        __DIR__ . '/storage/penduduk',
        __DIR__ . '/storage/berita',
        __DIR__ . '/storage/umkm'
    ];
    
    foreach ($directories as $dir) {
        if (!is_dir($dir)) {
            if (mkdir($dir, 0755, true)) {
                echo "<div class='success'>✅ Created: " . basename($dir) . "</div>";
            } else {
                echo "<div class='error'>❌ Failed to create: " . basename($dir) . "</div>";
            }
        } else {
            $fileCount = count(glob($dir . '/*'));
            echo "<div class='info'>📁 Exists: " . basename($dir) . " ({$fileCount} files)</div>";
        }
    }
    echo "</div>";
    
    // Step 2: Create default images
    echo "<div class='step'>";
    echo "<h2>🖼️ Step 2: Create Default Images</h2>";
    
    // Create default tourism image
    $defaultTourismPath = __DIR__ . '/images/default-tourism.png';
    if (!file_exists($defaultTourismPath)) {
        if (function_exists('imagecreate') && function_exists('imagepng')) {
            $img = imagecreate(400, 300);
            $bg = imagecolorallocate($img, 245, 245, 245);
            $text_color = imagecolorallocate($img, 100, 100, 100);
            $border_color = imagecolorallocate($img, 200, 200, 200);
            
            // Draw border
            imagerectangle($img, 0, 0, 399, 299, $border_color);
            
            // Add text
            if (function_exists('imagestring')) {
                imagestring($img, 5, 120, 130, 'Default Tourism', $text_color);
                imagestring($img, 3, 160, 160, 'Image', $text_color);
            }
            
            if (imagepng($img, $defaultTourismPath)) {
                echo "<div class='success'>✅ Created default-tourism.png</div>";
            } else {
                echo "<div class='error'>❌ Failed to create default-tourism.png</div>";
            }
            imagedestroy($img);
        } else {
            echo "<div class='warning'>⚠️ GD library not available for image creation</div>";
        }
    } else {
        echo "<div class='info'>📷 default-tourism.png already exists</div>";
    }
    
    // Create sample tourism images
    $sampleImages = ['sample-tourism-1.png', 'sample-tourism-2.png'];
    foreach ($sampleImages as $index => $filename) {
        $filepath = __DIR__ . '/storage/wisata/' . $filename;
        
        if (!file_exists($filepath)) {
            if (function_exists('imagecreate') && function_exists('imagepng')) {
                $img = imagecreate(600, 400);
                $colors = [
                    [52, 152, 219],   // Blue
                    [46, 204, 113],   // Green
                    [155, 89, 182],   // Purple
                ];
                $color = $colors[$index % count($colors)];
                
                $bg = imagecolorallocate($img, $color[0], $color[1], $color[2]);
                $text_color = imagecolorallocate($img, 255, 255, 255);
                
                if (function_exists('imagestring')) {
                    imagestring($img, 5, 200, 180, 'Sample Tourism', $text_color);
                    imagestring($img, 3, 250, 200, "#" . ($index + 1), $text_color);
                }
                
                if (imagepng($img, $filepath)) {
                    echo "<div class='success'>✅ Created {$filename}</div>";
                } else {
                    echo "<div class='error'>❌ Failed to create {$filename}</div>";
                }
                imagedestroy($img);
            }
        } else {
            echo "<div class='info'>📷 {$filename} already exists</div>";
        }
    }
    echo "</div>";
    
    // Step 3: Test file access
    echo "<div class='step'>";
    echo "<h2>🧪 Step 3: Test File Access</h2>";
    
    $testFiles = [
        'Default Tourism' => '/images/default-tourism.png',
        'Sample Tourism 1' => '/storage/wisata/sample-tourism-1.png',
        'Sample Tourism 2' => '/storage/wisata/sample-tourism-2.png'
    ];
    
    foreach ($testFiles as $name => $path) {
        $fullPath = __DIR__ . $path;
        $url = 'https://susukannet.id' . $path;
        
        if (file_exists($fullPath)) {
            $fileSize = filesize($fullPath);
            echo "<div class='success'>✅ {$name}: " . number_format($fileSize) . " bytes</div>";
            echo "<div class='code'><a href='{$url}' target='_blank'>{$url}</a></div>";
        } else {
            echo "<div class='error'>❌ {$name}: File not found</div>";
        }
    }
    echo "</div>";
    
    // Step 4: Laravel commands to run
    echo "<div class='step'>";
    echo "<h2>⚡ Step 4: Commands to Run</h2>";
    
    echo "<p><strong>Run these commands via SSH or hosting panel:</strong></p>";
    
    $commands = [
        'Clean temporary image paths from database' => 'php artisan tourism:clean-image-paths',
        'Sync storage files to public' => 'php artisan shared-hosting:sync-files --force',
        'Clear application caches' => 'php artisan config:clear && php artisan view:clear',
        'Create storage symlink (if supported)' => 'php artisan storage:link'
    ];
    
    foreach ($commands as $description => $command) {
        echo "<div class='warning'><strong>{$description}:</strong></div>";
        echo "<div class='cmd'>{$command}</div>";
    }
    echo "</div>";
    
    // Step 5: Manual verification
    echo "<div class='step'>";
    echo "<h2>✅ Step 5: Manual Verification</h2>";
    
    echo "<p><strong>Test these URLs to verify everything works:</strong></p>";
    echo "<ul>";
    echo "<li><a href='https://susukannet.id/images/default-tourism.png' target='_blank'>Default Tourism Image</a></li>";
    echo "<li><a href='https://susukannet.id/storage/wisata/sample-tourism-1.png' target='_blank'>Sample Tourism 1</a></li>";
    echo "<li><a href='https://susukannet.id/storage/wisata/sample-tourism-2.png' target='_blank'>Sample Tourism 2</a></li>";
    echo "<li><a href='https://susukannet.id/admin/objek-wisata' target='_blank'>Admin Tourism Page</a></li>";
    echo "<li><a href='https://susukannet.id/wisata' target='_blank'>Public Tourism Page</a></li>";
    echo "<li><a href='https://susukannet.id/test-live-images.php' target='_blank'>Image Test Page</a></li>";
    echo "</ul>";
    
    echo "<p><strong>Expected Results:</strong></p>";
    echo "<ul>";
    echo "<li>All image URLs should return actual images (not 404)</li>";
    echo "<li>Admin page should show tourism objects with images</li>";
    echo "<li>No more temporary file paths (tmp/php...)</li>";
    echo "<li>Generated URLs should use https://susukannet.id domain</li>";
    echo "</ul>";
    echo "</div>";
    
    // Step 6: Troubleshooting
    echo "<div class='step'>";
    echo "<h2>🔧 Step 6: Troubleshooting</h2>";
    
    echo "<p><strong>If images still don't show:</strong></p>";
    echo "<ol>";
    echo "<li>Check file permissions: 755 for directories, 644 for files</li>";
    echo "<li>Verify .htaccess is not blocking image access</li>";
    echo "<li>Check if mod_rewrite is enabled</li>";
    echo "<li>Ensure storage directory is accessible via web</li>";
    echo "<li>Check error logs for any PHP errors</li>";
    echo "</ol>";
    
    echo "<p><strong>Alternative solutions:</strong></p>";
    echo "<ul>";
    echo "<li>Use direct file upload to public/images/ instead of storage/</li>";
    echo "<li>Configure custom storage disk for shared hosting</li>";
    echo "<li>Use CDN or external image hosting</li>";
    echo "</ul>";
    echo "</div>";
    ?>
    
    <div class="step">
        <h2>📞 Support</h2>
        <p>If you need help, you can:</p>
        <ul>
            <li>Check Laravel logs in <code>storage/logs/</code></li>
            <li>Run this script again after making changes</li>
            <li>Contact your hosting provider for storage/symlink support</li>
        </ul>
    </div>
</body>
</html>
