@extends('layouts.app')

@section('title', 'Tambah Penduduk')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.dashboard') }}">Dashboard</a></li>
    <li class="breadcrumb-item"><a href="{{ route('admin.penduduk.index') }}">Data Penduduk</a></li>
    <li class="breadcrumb-item active">Tambah Penduduk</li>
@endsection

@section('content')
<!-- Page Header -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2 class="mb-2">
            <i class="fas fa-user-plus me-2 text-primary"></i>
            Tambah Data Penduduk
        </h2>
        <p class="text-muted mb-0">Tambahkan data penduduk baru ke dalam sistem</p>
        @if(in_array(auth()->user()->role, ['ketua_rt', 'rt', 'ketua_rw', 'rw', 'ketua_dusun', 'dusun']))
            <div class="alert alert-info mt-2 mb-0 py-2">
                <i class="fas fa-map-marker-alt me-2"></i>
                <strong>Wilayah Kerja Anda:</strong> 
                @if(auth()->user()->dusun) Dusun {{ auth()->user()->dusun }} @endif
                @if(auth()->user()->rw) RW {{ auth()->user()->rw }} @endif
                @if(auth()->user()->rt) RT {{ auth()->user()->rt }} @endif
                <br>
                <small class="text-muted">Data penduduk akan otomatis terfilter sesuai wilayah kerja Anda</small>
            </div>
        @endif
        
        <div class="alert alert-success mt-2 mb-0 py-2">
            <i class="fas fa-save me-2"></i>
            <strong>Auto-Save Aktif:</strong> Data form akan otomatis tersimpan setiap 2 detik dan dapat dipulihkan jika halaman di-reload
        </div>
    </div>
    <div>
        <a href="{{ route('admin.penduduk.index') }}" class="btn btn-outline-secondary">
            <i class="fas fa-arrow-left me-2"></i>
            Kembali
        </a>
    </div>
</div>

<form action="{{ route('admin.penduduk.store') }}" method="POST" enctype="multipart/form-data" id="pendudukForm">
    @csrf
    
    <div class="row">
        <!-- Left Column - Main Data -->
        <div class="col-lg-8">
            <!-- Personal Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-user me-2 text-primary"></i>
                        Data Pribadi
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nik" class="form-label required">NIK</label>
                                <input type="text" 
                                       class="form-control @error('nik') is-invalid @enderror" 
                                       id="nik" 
                                       name="nik" 
                                       value="{{ old('nik') }}" 
                                       placeholder="Nomor Induk Kependudukan"
                                       maxlength="16"
                                       required>
                                @error('nik')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">16 digit angka NIK</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nama" class="form-label required">Nama Lengkap</label>
                                <input type="text" 
                                       class="form-control @error('nama') is-invalid @enderror" 
                                       id="nama" 
                                       name="nama" 
                                       value="{{ old('nama') }}" 
                                       placeholder="Nama lengkap sesuai KTP"
                                       required>
                                @error('nama')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="no_kk" class="form-label">Nomor KK</label>
                                <input type="text" 
                                       class="form-control @error('no_kk') is-invalid @enderror" 
                                       id="no_kk" 
                                       name="no_kk" 
                                       value="{{ old('no_kk') }}" 
                                       placeholder="Nomor Kartu Keluarga"
                                       maxlength="16">
                                @error('no_kk')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">16 digit angka KK (opsional)</div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <!-- Placeholder for balance -->
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="jenis_kelamin" class="form-label required">Jenis Kelamin</label>
                                <select class="form-select @error('jenis_kelamin') is-invalid @enderror" 
                                        id="jenis_kelamin" 
                                        name="jenis_kelamin" 
                                        required>
                                    <option value="">Pilih Jenis Kelamin</option>
                                    <option value="L" {{ old('jenis_kelamin') == 'L' ? 'selected' : '' }}>Laki-laki</option>
                                    <option value="P" {{ old('jenis_kelamin') == 'P' ? 'selected' : '' }}>Perempuan</option>
                                </select>
                                @error('jenis_kelamin')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="tempat_lahir" class="form-label required">Tempat Lahir</label>
                                <input type="text" 
                                       class="form-control @error('tempat_lahir') is-invalid @enderror" 
                                       id="tempat_lahir" 
                                       name="tempat_lahir" 
                                       value="{{ old('tempat_lahir') }}" 
                                       placeholder="Kota/Kabupaten tempat lahir"
                                       required>
                                @error('tempat_lahir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="tanggal_lahir" class="form-label required">Tanggal Lahir</label>
                                <input type="date" 
                                       class="form-control @error('tanggal_lahir') is-invalid @enderror" 
                                       id="tanggal_lahir" 
                                       name="tanggal_lahir" 
                                       value="{{ old('tanggal_lahir') }}" 
                                       required>
                                @error('tanggal_lahir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="agama" class="form-label required">Agama</label>
                                <select class="form-select @error('agama') is-invalid @enderror" 
                                        id="agama" 
                                        name="agama" 
                                        required>
                                    <option value="">Pilih Agama</option>
                                    <option value="Islam" {{ old('agama') == 'Islam' ? 'selected' : '' }}>Islam</option>
                                    <option value="Kristen" {{ old('agama') == 'Kristen' ? 'selected' : '' }}>Kristen</option>
                                    <option value="Katolik" {{ old('agama') == 'Katolik' ? 'selected' : '' }}>Katolik</option>
                                    <option value="Hindu" {{ old('agama') == 'Hindu' ? 'selected' : '' }}>Hindu</option>
                                    <option value="Buddha" {{ old('agama') == 'Buddha' ? 'selected' : '' }}>Buddha</option>
                                    <option value="Konghucu" {{ old('agama') == 'Konghucu' ? 'selected' : '' }}>Konghucu</option>
                                </select>
                                @error('agama')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="status_perkawinan" class="form-label required">Status Perkawinan</label>
                                <select class="form-select @error('status_perkawinan') is-invalid @enderror" 
                                        id="status_perkawinan" 
                                        name="status_perkawinan" 
                                        required>
                                    <option value="">Pilih Status Kawin</option>
                                    <option value="Belum Kawin" {{ old('status_perkawinan') == 'Belum Kawin' ? 'selected' : '' }}>Belum Kawin</option>
                                    <option value="Kawin" {{ old('status_perkawinan') == 'Kawin' ? 'selected' : '' }}>Kawin</option>
                                    <option value="Cerai Hidup" {{ old('status_perkawinan') == 'Cerai Hidup' ? 'selected' : '' }}>Cerai Hidup</option>
                                    <option value="Cerai Mati" {{ old('status_perkawinan') == 'Cerai Mati' ? 'selected' : '' }}>Cerai Mati</option>
                                </select>
                                @error('status_perkawinan')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="pekerjaan" class="form-label">Pekerjaan</label>
                                <input type="text" 
                                       class="form-control @error('pekerjaan') is-invalid @enderror" 
                                       id="pekerjaan" 
                                       name="pekerjaan" 
                                       value="{{ old('pekerjaan') }}" 
                                       placeholder="Pekerjaan/Profesi">
                                @error('pekerjaan')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="pendidikan_terakhir" class="form-label">Pendidikan Terakhir</label>
                                <select class="form-select @error('pendidikan_terakhir') is-invalid @enderror" 
                                        id="pendidikan_terakhir" 
                                        name="pendidikan_terakhir">
                                    <option value="">Pilih Pendidikan Terakhir</option>
                                    <option value="Tidak Sekolah" {{ old('pendidikan_terakhir') == 'Tidak Sekolah' ? 'selected' : '' }}>Tidak Sekolah</option>
                                    <option value="SD" {{ old('pendidikan_terakhir') == 'SD' ? 'selected' : '' }}>SD/Sederajat</option>
                                    <option value="SMP" {{ old('pendidikan_terakhir') == 'SMP' ? 'selected' : '' }}>SMP/Sederajat</option>
                                    <option value="SMA" {{ old('pendidikan_terakhir') == 'SMA' ? 'selected' : '' }}>SMA/Sederajat</option>
                                    <option value="D1" {{ old('pendidikan_terakhir') == 'D1' ? 'selected' : '' }}>Diploma 1</option>
                                    <option value="D2" {{ old('pendidikan_terakhir') == 'D2' ? 'selected' : '' }}>Diploma 2</option>
                                    <option value="D3" {{ old('pendidikan_terakhir') == 'D3' ? 'selected' : '' }}>Diploma 3</option>
                                    <option value="S1" {{ old('pendidikan_terakhir') == 'S1' ? 'selected' : '' }}>Sarjana (S1)</option>
                                    <option value="S2" {{ old('pendidikan_terakhir') == 'S2' ? 'selected' : '' }}>Magister (S2)</option>
                                    <option value="S3" {{ old('pendidikan_terakhir') == 'S3' ? 'selected' : '' }}>Doktor (S3)</option>
                                </select>
                                @error('pendidikan_terakhir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="golongan_darah" class="form-label">Golongan Darah</label>
                                <select class="form-select @error('golongan_darah') is-invalid @enderror" 
                                        id="golongan_darah" 
                                        name="golongan_darah">
                                    <option value="">Pilih Golongan Darah</option>
                                    <option value="A" {{ old('golongan_darah') == 'A' ? 'selected' : '' }}>A</option>
                                    <option value="B" {{ old('golongan_darah') == 'B' ? 'selected' : '' }}>B</option>
                                    <option value="AB" {{ old('golongan_darah') == 'AB' ? 'selected' : '' }}>AB</option>
                                    <option value="O" {{ old('golongan_darah') == 'O' ? 'selected' : '' }}>O</option>
                                </select>
                                @error('golongan_darah')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="kewarganegaraan" class="form-label">Kewarganegaraan</label>
                        <select class="form-select @error('kewarganegaraan') is-invalid @enderror" 
                                id="kewarganegaraan" 
                                name="kewarganegaraan">
                            <option value="WNI" {{ old('kewarganegaraan', 'WNI') == 'WNI' ? 'selected' : '' }}>WNI (Warga Negara Indonesia)</option>
                            <option value="WNA" {{ old('kewarganegaraan') == 'WNA' ? 'selected' : '' }}>WNA (Warga Negara Asing)</option>
                        </select>
                        @error('kewarganegaraan')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="no_wa" class="form-label">Nomor WhatsApp</label>
                                <input type="tel" 
                                       class="form-control @error('no_wa') is-invalid @enderror" 
                                       id="no_wa" 
                                       name="no_wa" 
                                       value="{{ old('no_wa') }}" 
                                       placeholder="628123456789"
                                       maxlength="15">
                                @error('no_wa')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">
                                    Format: 628xxx (dimulai dengan 628)
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="status_hidup" class="form-label">Status Hidup</label>
                                <select class="form-select @error('status_hidup') is-invalid @enderror" 
                                        id="status_hidup" 
                                        name="status_hidup" 
                                        required>
                                    <option value="">Pilih Status Hidup</option>
                                    <option value="Hidup" {{ old('status_hidup') == 'Hidup' ? 'selected' : '' }}>Hidup</option>
                                    <option value="Meninggal" {{ old('status_hidup') == 'Meninggal' ? 'selected' : '' }}>Meninggal</option>
                                </select>
                                @error('status_hidup')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Address Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-map-marker-alt me-2 text-success"></i>
                        Alamat & Lokasi
                    </h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="alamat" class="form-label required">Alamat Lengkap</label>
                        <textarea class="form-control @error('alamat') is-invalid @enderror" 
                                  id="alamat" 
                                  name="alamat" 
                                  rows="3" 
                                  placeholder="Alamat lengkap (nama jalan, nomor rumah, dll)"
                                  required>{{ old('alamat') }}</textarea>
                        @error('alamat')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="mb-3">
                        <label for="nama_desa" class="form-label">Nama Desa</label>
                        <input type="text" 
                               class="form-control @error('nama_desa') is-invalid @enderror" 
                               id="nama_desa" 
                               name="nama_desa" 
                               value="{{ old('nama_desa', $setting->nama_desa ?? '') }}" 
                               placeholder="Nama Desa"
                               readonly>
                        @error('nama_desa')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">
                            <i class="fas fa-info-circle me-1"></i>
                            Nama desa otomatis dari setting sistem
                        </div>
                    </div>
                    
                                         <div class="row">
                         <div class="col-md-4">
                             <div class="mb-3">
                                 <label for="dusun" class="form-label">Dusun</label>
                                 <select class="form-select @error('dusun') is-invalid @enderror" 
                                         id="dusun" 
                                         name="dusun" 
                                         {{ in_array(auth()->user()->role, ['ketua_rt', 'rt', 'ketua_rw', 'rw', 'ketua_dusun', 'dusun']) ? 'disabled' : '' }}
                                         onchange="updateRwOptions()">
                                     <option value="">Pilih Dusun</option>
                                     @foreach($dusunList as $dusun)
                                         <option value="{{ $dusun }}" 
                                                 {{ old('dusun', auth()->user()->dusun) == $dusun ? 'selected' : '' }}
                                                 data-dusun="{{ $dusun }}">
                                             {{ $dusun }}
                                         </option>
                                     @endforeach
                                 </select>
                                 @if(in_array(auth()->user()->role, ['ketua_rt', 'rt', 'ketua_rw', 'rw', 'ketua_dusun', 'dusun']))
                                     <input type="hidden" name="dusun" value="{{ auth()->user()->dusun }}">
                                 @endif
                                 @error('dusun')
                                     <div class="invalid-feedback">{{ $message }}</div>
                                 @enderror
                                 @if(in_array(auth()->user()->role, ['ketua_rt', 'rt', 'ketua_rw', 'rw', 'ketua_dusun', 'dusun']))
                                     <div class="form-text text-info">
                                         <i class="fas fa-info-circle"></i> Otomatis terisi sesuai wilayah kerja Anda
                                     </div>
                                     <div class="form-text text-warning">
                                         <i class="fas fa-exclamation-triangle"></i> Hanya bisa menambah data di wilayah kerja Anda
                                     </div>
                                 @endif
                                 <div class="form-text text-muted">
                                     <i class="fas fa-cog me-1"></i> Nama desa otomatis dari setting sistem
                                 </div>
                             </div>
                         </div>
                         <div class="col-md-4">
                             <div class="mb-3">
                                 <label for="rw" class="form-label">RW</label>
                                 <select class="form-select @error('rw') is-invalid @enderror" 
                                         id="rw" 
                                         name="rw" 
                                         {{ in_array(auth()->user()->role, ['ketua_rt', 'rt', 'ketua_rw', 'rw']) ? 'disabled' : '' }}
                                         onchange="updateRtOptions()">
                                     <option value="">Pilih RW</option>
                                     @foreach($rwList as $rw)
                                         <option value="{{ $rw }}" 
                                                 {{ old('rw', auth()->user()->rw) == $rw ? 'selected' : '' }}
                                                 data-rw="{{ $rw }}">
                                             {{ $rw }}
                                         </option>
                                     @endforeach
                                 </select>
                                 @if(in_array(auth()->user()->role, ['ketua_rt', 'rt', 'ketua_rw', 'rw']))
                                     <input type="hidden" name="rw" value="{{ auth()->user()->rw }}">
                                 @endif
                                 @error('rw')
                                     <div class="invalid-feedback">{{ $message }}</div>
                                 @enderror
                                 @if(in_array(auth()->user()->role, ['ketua_rt', 'rt', 'ketua_rw', 'rw']))
                                     <div class="form-text text-info">
                                         <i class="fas fa-info-circle"></i> Otomatis terisi sesuai wilayah kerja Anda
                                     </div>
                                 @endif
                             </div>
                         </div>
                         <div class="col-md-4">
                             <div class="mb-3">
                                 <label for="rt" class="form-label">RT</label>
                                 <select class="form-select @error('rt') is-invalid @enderror" 
                                         id="rt" 
                                         name="rt" 
                                         {{ in_array(auth()->user()->role, ['ketua_rt', 'rt']) ? 'disabled' : '' }}>
                                     <option value="">Pilih RT</option>
                                     @foreach($rtList as $rt)
                                         <option value="{{ $rt }}" 
                                                 {{ old('rt', auth()->user()->rt) == $rt ? 'selected' : '' }}
                                                 data-rt="{{ $rt }}">
                                             {{ $rt }}
                                         </option>
                                     @endforeach
                                 </select>
                                 @if(in_array(auth()->user()->role, ['ketua_rt', 'rt']))
                                     <input type="hidden" name="rt" value="{{ auth()->user()->rt }}">
                                 @endif
                                 @error('rt')
                                     <div class="invalid-feedback">{{ $message }}</div>
                                 @enderror
                                 @if(in_array(auth()->user()->role, ['ketua_rt', 'rt']))
                                     <div class="form-text text-info">
                                         <i class="fas fa-info-circle"></i> Otomatis terisi sesuai wilayah kerja Anda
                                     </div>
                                 @endif
                             </div>
                         </div>
                     </div>
                    
                    <!-- Location Picker -->
                    <div class="mb-3">
                        <label class="form-label">Koordinat Lokasi</label>
                        <div class="row">
                            <div class="col-md-6">
                                <label for="latitude" class="form-label visually-hidden">Latitude</label>
                                <input type="text" 
                                       class="form-control @error('latitude') is-invalid @enderror" 
                                       id="latitude" 
                                       name="latitude" 
                                       value="{{ old('latitude') }}" 
                                       placeholder="Latitude"
                                       readonly>
                                @error('latitude')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="longitude" class="form-label visually-hidden">Longitude</label>
                                <input type="text" 
                                       class="form-control @error('longitude') is-invalid @enderror" 
                                       id="longitude" 
                                       name="longitude" 
                                       value="{{ old('longitude') }}" 
                                       placeholder="Longitude"
                                       readonly>
                                @error('longitude')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="form-text">
                            Klik pada peta di bawah untuk menentukan lokasi rumah
                        </div>
                    </div>
                    
                    <!-- Map -->
                    <div class="mb-3">
                        <div id="locationMap" class="map-container" style="height: 300px;"></div>
                    </div>
                    
                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="getCurrentLocation()">
                        <i class="fas fa-location-arrow me-2"></i>
                        Gunakan Lokasi Saat Ini
                    </button>
                </div>
            </div>
            
            <!-- Birth Certificate Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-certificate me-2 text-info"></i>
                        Data Akta Lahir
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="no_akta_lahir" class="form-label">Nomor Akta Lahir</label>
                                <input type="text" 
                                       class="form-control @error('no_akta_lahir') is-invalid @enderror" 
                                       id="no_akta_lahir" 
                                       name="no_akta_lahir" 
                                       value="{{ old('no_akta_lahir') }}" 
                                       placeholder="Nomor akta kelahiran"
                                       maxlength="50">
                                @error('no_akta_lahir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="tanggal_akta_lahir" class="form-label">Tanggal Akta Lahir</label>
                                <input type="date" 
                                       class="form-control @error('tanggal_akta_lahir') is-invalid @enderror" 
                                       id="tanggal_akta_lahir" 
                                       name="tanggal_akta_lahir" 
                                       value="{{ old('tanggal_akta_lahir') }}">
                                @error('tanggal_akta_lahir')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="tempat_akta_lahir" class="form-label">Tempat Akta Lahir</label>
                        <input type="text" 
                               class="form-control @error('tempat_akta_lahir') is-invalid @enderror" 
                               id="tempat_akta_lahir" 
                               name="tempat_akta_lahir" 
                               value="{{ old('tempat_akta_lahir') }}" 
                               placeholder="Tempat pembuatan akta kelahiran">
                        @error('tempat_akta_lahir')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>

            <!-- Parent Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-users me-2 text-purple"></i>
                        Data Orang Tua
                    </h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="ayah_nik" class="form-label">NIK Ayah</label>
                                <input type="text" 
                                       class="form-control @error('ayah_nik') is-invalid @enderror" 
                                       id="ayah_nik" 
                                       name="ayah_nik" 
                                       value="{{ old('ayah_nik') }}" 
                                       placeholder="NIK Ayah (16 digit)"
                                       maxlength="16">
                                @error('ayah_nik')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nama_ayah" class="form-label">Nama Ayah</label>
                                <input type="text" 
                                       class="form-control @error('nama_ayah') is-invalid @enderror" 
                                       id="nama_ayah" 
                                       name="nama_ayah" 
                                       value="{{ old('nama_ayah') }}" 
                                       placeholder="Nama lengkap ayah">
                                @error('nama_ayah')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">
                                    <i class="fas fa-info-circle me-1"></i>
                                    Sistem akan mencoba mencocokkan NIK dengan nama
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="ibu_nik" class="form-label">NIK Ibu</label>
                                <input type="text" 
                                       class="form-control @error('ibu_nik') is-invalid @enderror" 
                                       id="ibu_nik" 
                                       name="ibu_nik" 
                                       value="{{ old('ibu_nik') }}" 
                                       placeholder="NIK Ibu (16 digit)"
                                       maxlength="16">
                                @error('ibu_nik')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="nama_ibu" class="form-label">Nama Ibu</label>
                                <input type="text" 
                                       class="form-control @error('nama_ibu') is-invalid @enderror" 
                                       id="nama_ibu" 
                                       name="nama_ibu" 
                                       value="{{ old('nama_ibu') }}" 
                                       placeholder="Nama lengkap ibu">
                                @error('nama_ibu')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <div class="form-text">
                                    <i class="fas fa-info-circle me-1"></i>
                                    Sistem akan mencoba mencocokkan NIK dengan nama
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label for="hubungan_keluarga" class="form-label">Hubungan dalam Keluarga</label>
                        <select class="form-select @error('hubungan_keluarga') is-invalid @enderror" 
                                id="hubungan_keluarga" 
                                name="hubungan_keluarga">
                            <option value="">Pilih Hubungan Keluarga</option>
                            <option value="Kepala Keluarga" {{ old('hubungan_keluarga') == 'Kepala Keluarga' ? 'selected' : '' }}>Kepala Keluarga</option>
                            <option value="Istri" {{ old('hubungan_keluarga') == 'Istri' ? 'selected' : '' }}>Istri</option>
                            <option value="Anak" {{ old('hubungan_keluarga') == 'Anak' ? 'selected' : '' }}>Anak</option>
                            <option value="Menantu" {{ old('hubungan_keluarga') == 'Menantu' ? 'selected' : '' }}>Menantu</option>
                            <option value="Cucu" {{ old('hubungan_keluarga') == 'Cucu' ? 'selected' : '' }}>Cucu</option>
                            <option value="Orang Tua" {{ old('hubungan_keluarga') == 'Orang Tua' ? 'selected' : '' }}>Orang Tua</option>
                            <option value="Mertua" {{ old('hubungan_keluarga') == 'Mertua' ? 'selected' : '' }}>Mertua</option>
                            <option value="Saudara" {{ old('hubungan_keluarga') == 'Saudara' ? 'selected' : '' }}>Saudara</option>
                            <option value="Lainnya" {{ old('hubungan_keluarga') == 'Lainnya' ? 'selected' : '' }}>Lainnya</option>
                        </select>
                        @error('hubungan_keluarga')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>
            </div>

            <!-- Family Information -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-home me-2 text-warning"></i>
                        Data Keluarga
                    </h5>
                </div>
                <div class="card-body">
                    <div class="mb-3">
                        <label for="keluarga_id" class="form-label">Keluarga (Nomor KK)</label>
                        <select class="form-select @error('keluarga_id') is-invalid @enderror" 
                                id="keluarga_id" 
                                name="keluarga_id">
                            <option value="">Pilih Keluarga atau Kosongkan untuk Buat Baru</option>
                            @forelse($keluarga ?? [] as $k)
                                <option value="{{ $k->id ?? '' }}" {{ old('keluarga_id') == ($k->id ?? '') ? 'selected' : '' }}>
                                    {{ $k->no_kk ?? 'No KK tidak tersedia' }} - {{ optional($k->kepalaKeluarga)->nama ?? $k->kepala_keluarga_nik ?? 'Kepala keluarga tidak tersedia' }}
                                </option>
                            @empty
                                <option value="" disabled>Belum ada data keluarga</option>
                            @endforelse
                        </select>
                        @error('keluarga_id')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    
                    <div class="form-check">
                        <input class="form-check-input" 
                               type="checkbox" 
                               id="is_kepala_keluarga" 
                               name="is_kepala_keluarga" 
                               value="1" 
                               {{ old('is_kepala_keluarga') ? 'checked' : '' }}>
                        <label class="form-check-label" for="is_kepala_keluarga">
                            Kepala Keluarga
                        </label>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Right Column - Photo & Actions -->
        <div class="col-lg-4">
            <!-- Photo Upload -->
            <div class="card mb-4">
                <div class="card-header bg-light">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-camera me-2 text-info"></i>
                        Foto Penduduk
                    </h5>
                </div>
                <div class="card-body text-center">
                    <div class="photo-preview mb-3">
                        <div id="photoPreview" class="default-avatar-container">
                            <i class="fas fa-user default-avatar-icon"></i>
                            <div class="default-avatar-text">Pilih Foto</div>
                        </div>
                        <img id="photoPreviewImage" 
                             src="" 
                             class="img-fluid rounded-3" 
                             style="max-height: 250px; max-width: 100%; display: none;"
                             alt="Preview Foto">
                    </div>
                    
                    <div class="mb-3">
                        <input type="file" 
                               class="form-control @error('foto') is-invalid @enderror" 
                               id="foto" 
                               name="foto" 
                               accept="image/*"
                               onchange="previewPhoto(this)">
                        @error('foto')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                        <div class="form-text">
                            Format: JPG, PNG, JPEG (Maks. 2MB)
                        </div>
                    </div>
                    
                    <button type="button" class="btn btn-outline-secondary btn-sm" onclick="resetPhoto()">
                        <i class="fas fa-trash me-2"></i>
                        Reset Foto
                    </button>
                </div>
            </div>
            
            <!-- Action Buttons -->
            <div class="card">
                <div class="card-body">
                    <div class="d-grid gap-2">
                        <button type="submit" class="btn btn-gradient-primary" id="submitBtn">
                            <i class="fas fa-save me-2"></i>
                            Simpan Data
                        </button>
                        
                        <button type="button" class="btn btn-outline-warning" onclick="saveDraft()">
                            <i class="fas fa-file-alt me-2"></i>
                            Simpan Sebagai Draft
                        </button>
                        
                        <button type="button" class="btn btn-outline-danger" onclick="clearDraft()">
                            <i class="fas fa-trash me-2"></i>
                            Hapus Draft
                        </button>
                        
                        <a href="{{ route('admin.penduduk.index') }}" class="btn btn-outline-secondary">
                            <i class="fas fa-times me-2"></i>
                            Batal
                        </a>
                    </div>
                    
                    <div class="mt-3">
                        <small class="text-muted">
                            <i class="fas fa-info-circle me-1"></i>
                            Field dengan tanda (*) wajib diisi
                        </small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</form>
@endsection

@push('styles')
<style>
    .required::after {
        content: ' *';
        color: #dc3545;
    }
    
    .photo-preview {
        border: 2px dashed #e9ecef;
        border-radius: 10px;
        padding: 20px;
        transition: all 0.3s ease;
    }
    
    .photo-preview:hover {
        border-color: #667eea;
        background: rgba(102, 126, 234, 0.05);
    }
    
    .map-container {
        border: 2px solid #e9ecef;
        border-radius: 10px;
        overflow: hidden;
    }
    
    .form-control:focus, .form-select:focus {
        border-color: #667eea;
        box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
    }
    
    .card-header {
        border-bottom: 2px solid #e9ecef;
    }
    
    .btn-gradient-primary {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        border: none;
        color: white;
        font-weight: 600;
    }
    
    .btn-gradient-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 10px 25px rgba(102, 126, 234, 0.3);
        color: white;
    }
    
    .leaflet-container {
        font-family: inherit;
    }
    
    .default-avatar-container {
        width: 100%;
        max-width: 250px;
        height: 250px;
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        border: 2px dashed #dee2e6;
        border-radius: 12px;
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        margin: 0 auto;
        transition: all 0.3s ease;
    }
    
    .default-avatar-container:hover {
        border-color: #667eea;
        background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(102, 126, 234, 0.05) 100%);
    }
    
    .default-avatar-icon {
        font-size: 3rem;
        color: #9ca3af;
        margin-bottom: 10px;
    }
    
    .default-avatar-text {
        color: #6b7280;
        font-size: 0.9rem;
        font-weight: 500;
    }
    
    /* Auto-save notification styles */
    .alert-auto-save {
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 9999;
        min-width: 300px;
        animation: slideInRight 0.3s ease-out;
    }
    
    @keyframes slideInRight {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    /* Loading overlay */
    #loadingOverlay {
        backdrop-filter: blur(2px);
    }
    
    /* Toast container */
    .toast-container {
        z-index: 10000;
    }
    
    /* Form persistence indicator */
    .form-saved-indicator {
        position: fixed;
        bottom: 20px;
        right: 20px;
        background: rgba(40, 167, 69, 0.9);
        color: white;
        padding: 8px 16px;
        border-radius: 20px;
        font-size: 0.8rem;
        z-index: 9998;
        opacity: 0;
        transition: opacity 0.3s ease;
    }
    
    .form-saved-indicator.show {
        opacity: 1;
    }
</style>
@endpush

@push('scripts')
<script>
let map;
let marker;

document.addEventListener('DOMContentLoaded', function() {
    initializeMap();
    initializeForm();
});

// Initialize map
function initializeMap() {
    // Default coordinates (Bogor)
    const defaultLat = -6.5971;
    const defaultLng = 106.8060;
    
    map = L.map('locationMap').setView([defaultLat, defaultLng], 15);
    
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
    }).addTo(map);
    
    // Add click event to map
    map.on('click', function(e) {
        const lat = e.latlng.lat;
        const lng = e.latlng.lng;
        
        setMapLocation(lat, lng);
    });
    
    // Set existing coordinates if available
    const existingLat = document.getElementById('latitude').value;
    const existingLng = document.getElementById('longitude').value;
    
    if (existingLat && existingLng) {
        setMapLocation(parseFloat(existingLat), parseFloat(existingLng));
        map.setView([existingLat, existingLng], 17);
    }
}

// Set location on map
function setMapLocation(lat, lng) {
    // Remove existing marker
    if (marker) {
        map.removeLayer(marker);
    }
    
    // Add new marker
    marker = L.marker([lat, lng]).addTo(map);
    
    // Update input fields
    document.getElementById('latitude').value = lat.toFixed(6);
    document.getElementById('longitude').value = lng.toFixed(6);
    
    // Reverse geocoding (optional - get address from coordinates)
    reverseGeocode(lat, lng);
}

// Loading and notification functions
function showLoading() {
    // Create loading overlay
    const loading = document.createElement('div');
    loading.id = 'loadingOverlay';
    loading.className = 'position-fixed w-100 h-100 d-flex align-items-center justify-content-center';
    loading.style.cssText = 'top: 0; left: 0; background: rgba(0,0,0,0.5); z-index: 9999;';
    loading.innerHTML = `
        <div class="spinner-border text-light" role="status">
            <span class="visually-hidden">Loading...</span>
        </div>
    `;
    document.body.appendChild(loading);
}

function hideLoading() {
    const loading = document.getElementById('loadingOverlay');
    if (loading) {
        loading.remove();
    }
}

function showToast(message, type = 'info') {
    // Create toast notification
    const toast = document.createElement('div');
    toast.className = `toast align-items-center text-white bg-${type === 'error' ? 'danger' : type === 'success' ? 'success' : 'info'} border-0`;
    toast.setAttribute('role', 'alert');
    toast.setAttribute('aria-live', 'assertive');
    toast.setAttribute('aria-atomic', 'true');
    toast.innerHTML = `
        <div class="d-flex">
            <div class="toast-body">
                ${message}
            </div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
        </div>
    `;
    
    // Add to toast container
    let toastContainer = document.getElementById('toastContainer');
    if (!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.id = 'toastContainer';
        toastContainer.className = 'toast-container position-fixed top-0 end-0 p-3';
        toastContainer.style.cssText = 'z-index: 10000;';
        document.body.appendChild(toastContainer);
    }
    
    toastContainer.appendChild(toast);
    
    // Show toast
    const bsToast = new bootstrap.Toast(toast);
    bsToast.show();
    
    // Auto-remove after hide
    toast.addEventListener('hidden.bs.toast', () => {
        toast.remove();
    });
}

// Get current location
function getCurrentLocation() {
    if (navigator.geolocation) {
        showLoading();
        
        navigator.geolocation.getCurrentPosition(
            function(position) {
                const lat = position.coords.latitude;
                const lng = position.coords.longitude;
                
                setMapLocation(lat, lng);
                map.setView([lat, lng], 17);
                
                hideLoading();
                showToast('Lokasi berhasil didapatkan', 'success');
            },
            function(error) {
                hideLoading();
                let errorMessage = 'Gagal mendapatkan lokasi';
                
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        errorMessage = 'Izin lokasi ditolak';
                        break;
                    case error.POSITION_UNAVAILABLE:
                        errorMessage = 'Lokasi tidak tersedia';
                        break;
                    case error.TIMEOUT:
                        errorMessage = 'Timeout mendapatkan lokasi';
                        break;
                }
                
                showToast(errorMessage, 'error');
            },
            {
                enableHighAccuracy: true,
                timeout: 5000,
                maximumAge: 0
            }
        );
    } else {
        showToast('Geolocation tidak didukung browser', 'error');
    }
}

// Reverse geocoding
function reverseGeocode(lat, lng) {
    fetch(`https://nominatim.openstreetmap.org/reverse?format=json&lat=${lat}&lon=${lng}&zoom=18&addressdetails=1`)
        .then(response => response.json())
        .then(data => {
            if (data.display_name) {
                // Optionally populate address field
                const addressField = document.getElementById('alamat');
                if (!addressField.value) {
                    addressField.value = data.display_name;
                }
            }
        })
        .catch(error => {
            console.error('Reverse geocoding error:', error);
        });
}

// Photo preview
function previewPhoto(input) {
    if (input.files && input.files[0]) {
        const file = input.files[0];
        
        // Validate file size (2MB)
        if (file.size > 2 * 1024 * 1024) {
            showToast('Ukuran file terlalu besar. Maksimal 2MB.', 'error');
            input.value = '';
            return;
        }
        
        // Validate file type
        if (!file.type.match('image.*')) {
            showToast('File harus berupa gambar.', 'error');
            input.value = '';
            return;
        }
        
        const reader = new FileReader();
        reader.onload = function(e) {
            // Hide default avatar container and show image
            document.getElementById('photoPreview').style.display = 'none';
            const previewImg = document.getElementById('photoPreviewImage');
            previewImg.src = e.target.result;
            previewImg.style.display = 'block';
            
            // Add resize info
            showToast('Foto akan diresize otomatis ke ukuran 400x400 pixel untuk konsistensi tampilan.', 'info');
        };
        reader.readAsDataURL(file);
    }
}

// Reset photo
function resetPhoto() {
    document.getElementById('foto').value = '';
    // Show default avatar container and hide image
    document.getElementById('photoPreview').style.display = 'flex';
    document.getElementById('photoPreviewImage').style.display = 'none';
    document.getElementById('photoPreviewImage').src = '';
}

// Initialize form
function initializeForm() {
    // NIK validation
    document.getElementById('nik').addEventListener('input', function() {
        this.value = this.value.replace(/\D/g, '');
        
        if (this.value.length === 16) {
            // Auto-fill some data based on NIK if needed
            validateNIK(this.value);
        }
    });
    
    // Phone number formatting
    const phoneInputs = document.querySelectorAll('input[type="tel"]');
    phoneInputs.forEach(input => {
        input.addEventListener('input', function() {
            this.value = this.value.replace(/\D/g, '');
        });
    });
    
    // Date validation
    document.getElementById('tanggal_lahir').addEventListener('change', function() {
        const today = new Date();
        const birthDate = new Date(this.value);
        
        if (birthDate > today) {
            showToast('Tanggal lahir tidak boleh di masa depan', 'error');
            this.value = '';
        }
    });
    
    // Kepala keluarga logic
    document.getElementById('is_kepala_keluarga').addEventListener('change', function() {
        const keluargaSelect = document.getElementById('keluarga_id');
        
        if (this.checked) {
            keluargaSelect.value = '';
            keluargaSelect.disabled = true;
            showToast('Jika kepala keluarga, sistem akan membuat kartu keluarga baru', 'info');
        } else {
            keluargaSelect.disabled = false;
        }
    });
    
    // Restore form data if available
    setTimeout(() => {
        restoreFormData();
    }, 500);
}

// Validate NIK
function validateNIK(nik) {
    if (nik.length !== 16) {
        return false;
    }
    
    // Basic NIK validation
    const provinceCodes = ['31', '32', '33', '34', '35', '36']; // Add more as needed
    const provinceCode = nik.substring(0, 2);
    
    if (!provinceCodes.includes(provinceCode)) {
        showToast('Kode provinsi NIK tidak valid', 'warning');
    }
    
    return true;
}

// Save as draft
function saveDraft() {
    const formData = new FormData(document.getElementById('pendudukForm'));
    formData.append('is_draft', '1');
    
    showLoading();
    
    fetch('{{ route("admin.penduduk.store") }}', {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast('Data berhasil disimpan sebagai draft', 'success');
        } else {
            showToast(data.message || 'Gagal menyimpan draft', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Terjadi kesalahan saat menyimpan draft', 'error');
    })
    .finally(() => {
        hideLoading();
    });
}

// Dynamic wilayah filtering functions
function updateRwOptions() {
    const selectedDusun = document.getElementById('dusun').value;
    const rwSelect = document.getElementById('rw');
    const rtSelect = document.getElementById('rt');
    
    // Reset RW and RT
    rwSelect.innerHTML = '<option value="">Pilih RW</option>';
    rtSelect.innerHTML = '<option value="">Pilih RT</option>';
    
    if (!selectedDusun) return;
    
    // Get RW data from wilayah_administratif table via AJAX
    fetch(`/admin/wilayah-administratif/jenis/rw?parent_dusun=${encodeURIComponent(selectedDusun)}`, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data.length > 0) {
            data.data.forEach(rw => {
                const option = document.createElement('option');
                option.value = rw.nama_wilayah;
                option.textContent = rw.nama_wilayah;
                rwSelect.appendChild(option);
            });
        } else {
            // Fallback to static list if AJAX fails
            const rwOptions = @json($rwList);
            rwOptions.forEach(rw => {
                const option = document.createElement('option');
                option.value = rw;
                option.textContent = rw;
                rwSelect.appendChild(option);
            });
        }
    })
    .catch(error => {
        console.error('Error loading RW data:', error);
        // Fallback to static list
        const rwOptions = @json($rwList);
        rwOptions.forEach(rw => {
            const option = document.createElement('option');
            option.value = rw;
            option.textContent = rw;
            rwSelect.appendChild(option);
        });
    });
}

function updateRtOptions() {
    const selectedDusun = document.getElementById('dusun').value;
    const selectedRw = document.getElementById('rw').value;
    const rtSelect = document.getElementById('rt');
    
    // Reset RT
    rtSelect.innerHTML = '<option value="">Pilih RT</option>';
    
    if (!selectedDusun || !selectedRw) return;
    
    // Get RT data from wilayah_administratif table via AJAX
    fetch(`/admin/wilayah-administratif/jenis/rt?parent_rw=${encodeURIComponent(selectedRw)}&parent_dusun=${encodeURIComponent(selectedDusun)}`, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data.length > 0) {
            data.data.forEach(rt => {
                const option = document.createElement('option');
                option.value = rt.nama_wilayah;
                option.textContent = rt.nama_wilayah;
                rtSelect.appendChild(option);
            });
        } else {
            // Fallback to static list if AJAX fails
            const rtOptions = @json($rtList);
            rtOptions.forEach(rt => {
                const option = document.createElement('option');
                option.value = rt;
                option.textContent = rt;
                rtSelect.appendChild(option);
            });
        }
    })
    .catch(error => {
        console.error('Error loading RT data:', error);
        // Fallback to static list
        const rtOptions = @json($rtList);
        rtOptions.forEach(rt => {
            const option = document.createElement('option');
            option.value = rt;
            option.textContent = rt;
            rtSelect.appendChild(option);
        });
    });
}

// Validate wilayah kerja based on user role
function validateWilayahKerja() {
    const userRole = '{{ auth()->user()->role }}';
    const userDusun = '{{ auth()->user()->dusun }}';
    const userRt = '{{ auth()->user()->rt }}';
    const userRw = '{{ auth()->user()->rw }}';
    
    const inputDusun = document.getElementById('dusun').value;
    const inputRt = document.getElementById('rt').value;
    const inputRw = document.getElementById('rw').value;
    
    // RT level validation
    if (['ketua_rt', 'rt'].includes(userRole)) {
        if (inputDusun !== userDusun || inputRt !== userRt || inputRw !== userRw) {
            showToast('Anda hanya bisa menambah data penduduk di wilayah kerja Anda sendiri', 'error');
            return false;
        }
    }
    
    // RW level validation
    if (['ketua_rw', 'rw'].includes(userRole)) {
        if (inputDusun !== userDusun || inputRw !== userRw) {
            showToast('Anda hanya bisa menambah data penduduk di RW Anda', 'error');
            return false;
        }
    }
    
    // Dusun level validation
    if (['ketua_dusun', 'dusun'].includes(userRole)) {
        if (inputDusun !== userDusun) {
            showToast('Anda hanya bisa menambah data penduduk di dusun Anda', 'error');
            return false;
        }
    }
    
    return true;
}

// Auto-refresh session before form submission
function refreshSessionBeforeSubmit() {
    if (window.adminSessionManager) {
        return window.adminSessionManager.forceRefresh();
    }
    return Promise.resolve(true);
}

// Form submission
document.getElementById('pendudukForm').addEventListener('submit', function(e) {
    // Prevent default submission first
    e.preventDefault();
    
    // Validate wilayah kerja first
    if (!validateWilayahKerja()) {
        return false;
    }
    
    // Enable disabled fields before submit to ensure values are sent
    enableDisabledFields();
    
    const submitBtn = document.getElementById('submitBtn');
    
    // Disable submit button
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Menyimpan...';
    
    // Refresh session before submission
    refreshSessionBeforeSubmit().then(success => {
        if (!success) {
            alert('Sesi Anda telah berakhir. Silakan refresh halaman dan login kembali.');
            window.location.reload();
            return;
        }
        
        // Ensure CSRF token is up to date
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
        const formToken = this.querySelector('input[name="_token"]');
        
        if (csrfToken && formToken) {
            formToken.value = csrfToken;
            console.log('CSRF token updated:', csrfToken);
        } else {
            console.error('CSRF token not found!');
            alert('CSRF token tidak ditemukan. Silakan refresh halaman dan coba lagi.');
            submitBtn.disabled = false;
            submitBtn.innerHTML = '<i class="fas fa-save me-2"></i>Simpan Data';
            return;
        }
        
        // Now submit the form
        this.submit();
    }).catch(error => {
        console.error('Error refreshing session:', error);
        alert('Terjadi kesalahan saat memperbarui sesi. Silakan coba lagi.');
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fas fa-save me-2"></i>Simpan Data';
    });
    
    // Add success handler to clear saved data
    this.addEventListener('submit', function() {
        // This will be called after successful submission
        setTimeout(() => {
            clearSavedFormData();
        }, 1000);
    });
});

// Enable disabled fields before form submission
function enableDisabledFields() {
    const userRole = '{{ auth()->user()->role }}';
    
    // Enable dusun field if user has RT/RW/Dusun role
    if (['ketua_rt', 'rt', 'ketua_rw', 'rw', 'ketua_dusun', 'dusun'].includes(userRole)) {
        const dusunSelect = document.getElementById('dusun');
        if (dusunSelect.disabled) {
            dusunSelect.disabled = false;
        }
    }
    
    // Enable RW field if user has RT/RW role
    if (['ketua_rt', 'rt', 'ketua_rw', 'rw'].includes(userRole)) {
        const rwSelect = document.getElementById('rw');
        if (rwSelect.disabled) {
            rwSelect.disabled = false;
        }
    }
    
    // Enable RT field if user has RT role
    if (['ketua_rt', 'rt'].includes(userRole)) {
        const rtSelect = document.getElementById('rt');
        if (rtSelect.disabled) {
            rtSelect.disabled = false;
        }
    }
}

// Auto-save and form persistence system
let autoSaveTimer;
let isAutoSaving = false;

// Initialize auto-save system
function enableAutoSave() {
    const form = document.getElementById('pendudukForm');
    const inputs = form.querySelectorAll('input, select, textarea');
    
    inputs.forEach(input => {
        input.addEventListener('input', function() {
            clearTimeout(autoSaveTimer);
            autoSaveTimer = setTimeout(() => {
                autoSaveForm();
            }, 2000); // Save after 2 seconds of inactivity
        });
        
        input.addEventListener('change', function() {
            // Save immediately on change for important fields
            if (['nik', 'nama', 'dusun', 'rw', 'rt'].includes(this.name)) {
                clearTimeout(autoSaveTimer);
                autoSaveTimer = setTimeout(() => {
                    autoSaveForm();
                }, 500);
            }
        });
    });
    
    // Save on page unload
    window.addEventListener('beforeunload', function() {
        if (hasFormData()) {
            autoSaveForm(true); // Force save
        }
    });
    
    // Save on form focus/blur
    form.addEventListener('focusin', function() {
        clearTimeout(autoSaveTimer);
    });
    
    form.addEventListener('focusout', function() {
        autoSaveTimer = setTimeout(() => {
            autoSaveForm();
        }, 1000);
    });
}

// Auto-save form data
function autoSaveForm(force = false) {
    if (isAutoSaving && !force) return;
    
    const form = document.getElementById('pendudukForm');
    const formData = new FormData(form);
    const data = {};
    
    // Convert FormData to object
    for (let [key, value] of formData.entries()) {
        if (value !== '' && value !== null) {
            data[key] = value;
        }
    }
    
    // Add timestamp
    data.timestamp = new Date().toISOString();
    data.form_id = 'penduduk_create';
    
    // Save to localStorage
    try {
        localStorage.setItem('penduduk_form_draft', JSON.stringify(data));
        console.log('Form auto-saved at:', new Date().toLocaleTimeString());
        
        // Show subtle notification
        showAutoSaveNotification('Data tersimpan otomatis', 'success');
        
    } catch (error) {
        console.error('Auto-save error:', error);
        // Fallback to sessionStorage if localStorage fails
        try {
            sessionStorage.setItem('penduduk_form_draft', JSON.stringify(data));
        } catch (fallbackError) {
            console.error('Fallback save also failed:', fallbackError);
        }
    }
    
    isAutoSaving = false;
}

// Check if form has data
function hasFormData() {
    const form = document.getElementById('pendudukForm');
    const inputs = form.querySelectorAll('input, select, textarea');
    
    for (let input of inputs) {
        if (input.value && input.value.trim() !== '') {
            return true;
        }
    }
    return false;
}

// Restore form data
function restoreFormData() {
    try {
        // Try localStorage first
        let savedData = localStorage.getItem('penduduk_form_draft');
        
        if (!savedData) {
            // Fallback to sessionStorage
            savedData = sessionStorage.getItem('penduduk_form_draft');
        }
        
        if (savedData) {
            const data = JSON.parse(savedData);
            
            // Check if data is recent (within 24 hours)
            const savedTime = new Date(data.timestamp);
            const now = new Date();
            const hoursDiff = (now - savedTime) / (1000 * 60 * 60);
            
            if (hoursDiff < 24) {
                populateForm(data);
                showAutoSaveNotification('Data sebelumnya dipulihkan', 'info');
                return true;
            } else {
                // Clear old data
                clearSavedFormData();
            }
        }
    } catch (error) {
        console.error('Error restoring form data:', error);
        clearSavedFormData();
    }
    
    return false;
}

// Populate form with saved data
function populateForm(data) {
    const form = document.getElementById('pendudukForm');
    
    // Populate input fields
    Object.keys(data).forEach(key => {
        if (key === 'timestamp' || key === 'form_id') return;
        
        const element = form.querySelector(`[name="${key}"]`);
        if (element) {
            if (element.type === 'checkbox') {
                element.checked = data[key] === '1' || data[key] === true;
            } else if (element.type === 'file') {
                // Skip file inputs
                return;
            } else {
                element.value = data[key];
            }
            
            // Trigger change event for dependent fields
            if (['dusun', 'rw', 'rt'].includes(key)) {
                element.dispatchEvent(new Event('change'));
            }
        }
    });
    
    // Handle special cases
    if (data.foto && data.foto !== '') {
        // Restore photo preview if it was a data URL
        if (data.foto.startsWith('data:')) {
            document.getElementById('photoPreviewImage').src = data.foto;
            document.getElementById('photoPreviewImage').style.display = 'block';
            document.getElementById('photoPreview').style.display = 'none';
        }
    }
    
    // Update dependent dropdowns
    if (data.dusun) {
        updateRwOptions();
    }
    if (data.rw) {
        updateRtOptions();
    }
}

// Clear saved form data
function clearSavedFormData() {
    try {
        localStorage.removeItem('penduduk_form_draft');
        sessionStorage.removeItem('penduduk_form_draft');
    } catch (error) {
        console.error('Error clearing saved data:', error);
    }
}

// Show auto-save notification
function showAutoSaveNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : 'info-circle'} me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto-remove after 3 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 3000);
}

// Manual save draft function
function saveDraft() {
    const formData = new FormData(document.getElementById('pendudukForm'));
    formData.append('is_draft', '1');
    
    showLoading();
    
    fetch('{{ route("admin.penduduk.store") }}', {
        method: 'POST',
        body: formData,
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast('Data berhasil disimpan sebagai draft', 'success');
            // Clear saved form data after successful save
            clearSavedFormData();
        } else {
            showToast(data.message || 'Gagal menyimpan draft', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Terjadi kesalahan saat menyimpan draft', 'error');
    })
    .finally(() => {
        hideLoading();
    });
}

// Clear draft button
function clearDraft() {
    if (confirm('Apakah Anda yakin ingin menghapus data draft yang tersimpan?')) {
        clearSavedFormData();
        document.getElementById('pendudukForm').reset();
        showToast('Data draft berhasil dihapus', 'success');
    }
}

// Enable auto-save
enableAutoSave();
</script>
@endpush 
