@extends('layouts.app')

@section('title', 'Data Penduduk')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.dashboard') }}">Dashboard</a></li>
    <li class="breadcrumb-item active">Data Penduduk</li>
@endsection

@section('content')
<!-- Page Header -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2 class="mb-2">
            <i class="fas fa-users me-2 text-primary"></i>
            Data Penduduk
        </h2>
        <p class="text-muted mb-0">Kelola data penduduk desa secara lengkap dan akurat</p>
    </div>
    <div>
        <button type="button" class="btn btn-outline-success me-2" onclick="exportData()">
            <i class="fas fa-download me-2"></i>
            Export CSV
        </button>
        <button type="button" class="btn btn-outline-info me-2" onclick="toggleImportForm()">
            <i class="fas fa-upload me-2"></i>
            Import CSV
        </button>
        <button type="button" class="btn btn-outline-warning me-2" onclick="showBulkPasswordModal()">
            <i class="fas fa-key me-2"></i>
            Generate Password Massal
        </button>
        <a href="{{ route('admin.penduduk.create') }}" class="btn btn-gradient-primary">
            <i class="fas fa-user-plus me-2"></i>
            Tambah Penduduk
        </a>
    </div>
</div>

@if(session('error'))
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        <i class="fas fa-exclamation-triangle me-2"></i>
        {{ session('error') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
@endif

@if(session('success'))
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        <i class="fas fa-check-circle me-2"></i>
        {{ session('success') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
@endif

@if(session('import_errors') && count(session('import_errors')) > 0)
    <div class="alert alert-warning alert-dismissible fade show" role="alert">
        <i class="fas fa-exclamation-triangle me-2"></i>
        <strong>Beberapa data gagal diimport:</strong>
        <ul class="mt-2 mb-0">
            @foreach(array_slice(session('import_errors'), 0, 10) as $error)
            <li>{{ $error }}</li>
            @endforeach
            @if(count(session('import_errors')) > 10)
            <li><em>Dan {{ count(session('import_errors')) - 10 }} error lainnya...</em></li>
            @endif
        </ul>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
@endif

<!-- Statistics Cards -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card bg-primary text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="me-3">
                        <i class="fas fa-users fa-2x"></i>
                    </div>
                    <div>
                        <h4 class="mb-0">{{ $stats['total'] ?? 0 }}</h4>
                        <small>Total Penduduk</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-info text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="me-3">
                        <i class="fas fa-male fa-2x"></i>
                    </div>
                    <div>
                        <h4 class="mb-0">{{ $stats['laki_laki'] ?? 0 }}</h4>
                        <small>Laki-laki</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-pink text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="me-3">
                        <i class="fas fa-female fa-2x"></i>
                    </div>
                    <div>
                        <h4 class="mb-0">{{ $stats['perempuan'] ?? 0 }}</h4>
                        <small>Perempuan</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-warning text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="me-3">
                        <i class="fas fa-child fa-2x"></i>
                    </div>
                    <div>
                        <h4 class="mb-0">{{ $stats['anak'] ?? 0 }}</h4>
                        <small>Anak-anak</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Password Statistics Cards -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card bg-success text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="me-3">
                        <i class="fas fa-check-circle fa-2x"></i>
                    </div>
                    <div>
                        <h4 class="mb-0">{{ $stats['password_aktif'] ?? 0 }}</h4>
                        <small>Portal Aktif</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-warning text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="me-3">
                        <i class="fas fa-exclamation-triangle fa-2x"></i>
                    </div>
                    <div>
                        <h4 class="mb-0">{{ $stats['password_harus_ganti'] ?? 0 }}</h4>
                        <small>Harus Ganti Password</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-danger text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="me-3">
                        <i class="fas fa-times-circle fa-2x"></i>
                    </div>
                    <div>
                        <h4 class="mb-0">{{ $stats['password_belum_diatur'] ?? 0 }}</h4>
                        <small>Belum Diatur</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-info text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="me-3">
                        <i class="fas fa-sign-in-alt fa-2x"></i>
                    </div>
                    <div>
                        <h4 class="mb-0">{{ $stats['login_hari_ini'] ?? 0 }}</h4>
                        <small>Login Hari Ini</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" action="{{ route('admin.penduduk.index') }}">
            <div class="row">
                <div class="col-md-3">
                    <label class="form-label">Pencarian</label>
                    <input type="text" class="form-control" name="search" value="{{ request('search') }}" placeholder="Nama, NIK, atau No KK">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Dusun</label>
                    <select class="form-select" name="dusun">
                        <option value="">Semua Dusun</option>
                        @foreach($dusunList as $dusun)
                            <option value="{{ $dusun }}" {{ request('dusun') == $dusun ? 'selected' : '' }}>{{ $dusun }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">RT</label>
                    <select class="form-select" name="rt">
                        <option value="">Semua RT</option>
                        @foreach($rtList as $rt)
                            <option value="{{ $rt }}" {{ request('rt') == $rt ? 'selected' : '' }}>{{ $rt }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">RW</label>
                    <select class="form-select" name="rw">
                        <option value="">Semua RW</option>
                        @foreach($rwList as $rw)
                            <option value="{{ $rw }}" {{ request('rw') == $rw ? 'selected' : '' }}>{{ $rw }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Status Portal</label>
                    <select class="form-select" name="portal_status">
                        <option value="">Semua Status</option>
                        <option value="aktif" {{ request('portal_status') == 'aktif' ? 'selected' : '' }}>Portal Aktif</option>
                        <option value="harus_ganti" {{ request('portal_status') == 'harus_ganti' ? 'selected' : '' }}>Harus Ganti Password</option>
                        <option value="belum_diatur" {{ request('portal_status') == 'belum_diatur' ? 'selected' : '' }}>Belum Diatur</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">&nbsp;</label>
                    <div class="d-grid">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-search me-2"></i>Filter
                        </button>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Data Table -->
<div class="card">
    <div class="card-header bg-light">
        <h5 class="card-title mb-0">
            <i class="fas fa-table me-2"></i>
            Daftar Penduduk
        </h5>
    </div>
    <div class="card-body">
        @if(count($penduduk) > 0)
            <div class="table-responsive">
                <table class="table table-hover table-striped">
                    <thead class="table-light">
                        <tr>
                            <th width="5%" class="text-center">No</th>
                            <th width="15%">NIK</th>
                            <th width="18%">Nama Lengkap</th>
                            <th width="8%" class="text-center">L/P</th>
                            <th width="15%">Tempat/Tgl Lahir</th>
                            <th width="6%" class="text-center">Umur</th>
                            <th width="10%" class="text-center">Portal</th>
                            <th width="13%">Alamat</th>
                            <th width="10%" class="text-center">Aksi</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($penduduk as $index => $p)
                        <tr>
                            <td class="text-center">{{ $penduduk->firstItem() + $index }}</td>
                            <td>
                                <span class="text-monospace">{{ $p->nik ?? '-' }}</span>
                            </td>
                            <td>
                                <div class="d-flex align-items-center">
                                    <img src="{{ \App\Helpers\ImageHelper::getAvatarUrl($p->foto, $p->nama) }}" 
                                         class="rounded-circle me-2" 
                                         width="32" height="32"
                                         alt="{{ $p->nama }}"
                                         style="object-fit: cover;">
                                    <div>
                                        <strong>{{ $p->nama ?? '-' }}</strong>
                                        @if($p->no_kk)
                                            <br><small class="text-muted">KK: {{ $p->no_kk }}</small>
                                        @endif
                                    </div>
                                </div>
                            </td>
                            <td class="text-center">
                                @if($p->jenis_kelamin == 'L')
                                    <span class="badge bg-primary">L</span>
                                @elseif($p->jenis_kelamin == 'P')
                                    <span class="badge bg-pink">P</span>
                                @else
                                    <span class="badge bg-secondary">-</span>
                                @endif
                            </td>
                            <td>
                                @if($p->tempat_lahir || $p->tanggal_lahir)
                                    {{ $p->tempat_lahir ?? '-' }}
                                    @if($p->tanggal_lahir)
                                        <br><small class="text-muted">{{ $p->tanggal_lahir->format('d/m/Y') }}</small>
                                    @endif
                                @else
                                    -
                                @endif
                            </td>
                            <td class="text-center">
                                @if($p->tanggal_lahir)
                                    {{ $p->tanggal_lahir->age }} th
                                @else
                                    -
                                @endif
                            </td>
                            <td class="text-center">
                                @if($p->hasPassword())
                                    @if($p->mustChangePassword())
                                        <span class="badge bg-warning" data-bs-toggle="tooltip" title="Password harus diganti">
                                            <i class="fas fa-exclamation-triangle me-1"></i>
                                            Harus Ganti
                                        </span>
                                    @else
                                        <span class="badge bg-success" data-bs-toggle="tooltip" title="Password aktif - Login: {{ $p->last_login_at ? $p->last_login_at->diffForHumans() : 'Belum pernah' }}">
                                            <i class="fas fa-check-circle me-1"></i>
                                            Aktif
                                        </span>
                                    @endif
                                @else
                                    <span class="badge bg-danger" data-bs-toggle="tooltip" title="Belum bisa login portal">
                                        <i class="fas fa-times-circle me-1"></i>
                                        Belum Diatur
                                    </span>
                                @endif
                            </td>
                            <td>
                                <div>
                                    {{ $p->alamat ?? '-' }}
                                    @if($p->rt || $p->rw || $p->dusun)
                                        <br>
                                        <small class="text-muted">
                                            @if($p->rt)RT {{ $p->rt }}@endif
                                            @if($p->rw) RW {{ $p->rw }}@endif
                                            @if($p->dusun) {{ $p->dusun }}@endif
                                        </small>
                                    @endif
                                </div>
                            </td>
                            <td class="text-center">
                                <div class="btn-group btn-group-sm" role="group">
                                    <button type="button" class="btn btn-outline-info" 
                                            onclick="showDetail({{ $p->id }})"
                                            data-bs-toggle="tooltip" title="Lihat Detail">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    <a href="{{ route('admin.penduduk.edit', $p->id) }}" 
                                       class="btn btn-outline-warning"
                                       data-bs-toggle="tooltip" title="Edit">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                    <button type="button" class="btn btn-outline-danger" 
                                            onclick="deletePenduduk({{ $p->id }})"
                                            data-bs-toggle="tooltip" title="Hapus">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <div class="d-flex justify-content-between align-items-center mt-3">
                <div>
                    <small class="text-muted">
                        Menampilkan {{ $penduduk->firstItem() }} sampai {{ $penduduk->lastItem() }} 
                        dari {{ $penduduk->total() }} data
                    </small>
                </div>
                <div>
                    {{ $penduduk->appends(request()->query())->links() }}
                </div>
            </div>
        @else
            <div class="text-center py-5">
                <i class="fas fa-users fa-3x text-muted mb-3"></i>
                <h5 class="text-muted">Tidak ada data penduduk</h5>
                <p class="text-muted">Belum ada data penduduk yang tersedia atau sesuai dengan filter yang dipilih.</p>
                <a href="{{ route('admin.penduduk.create') }}" class="btn btn-primary">
                    <i class="fas fa-plus me-2"></i>Tambah Penduduk Pertama
                </a>
            </div>
        @endif
    </div>
</div>

<!-- Import Form (Hidden by default) -->
<div class="card mb-4" id="importForm" style="display: none;">
    <div class="card-header">
        <h5 class="card-title mb-0">
            <i class="fas fa-upload me-2"></i>
            Import Data Penduduk
        </h5>
    </div>
    <div class="card-body">
        <form action="{{ route('admin.penduduk.import') }}" method="POST" enctype="multipart/form-data">
            @csrf
            
            <!-- Baris 1: File Input -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label fw-bold">
                        <i class="fas fa-file-csv me-2 text-primary"></i>
                        Pilih File CSV
                    </label>
                    <input type="file" class="form-control" name="file" accept=".csv" required>
                    <div class="form-text">
                        <i class="fas fa-info-circle me-1"></i>
                        Format: .csv | Ukuran maksimal: 2MB
                    </div>
                </div>
                <div class="col-md-6">
                    <label class="form-label fw-bold">
                        <i class="fas fa-download me-2 text-success"></i>
                        Template & Panduan
                    </label>
                    <div class="d-grid">
                        <button type="button" class="btn btn-outline-success" onclick="downloadTemplate()">
                            <i class="fas fa-download me-2"></i>Download Template CSV
                        </button>
                    </div>
                    <div class="form-text">
                        <i class="fas fa-lightbulb me-1"></i>
                        Template berisi contoh format yang benar
                    </div>
                </div>
            </div>

            <!-- Baris 2: Format Field -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <label class="form-label fw-bold text-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        Field Wajib (6 kolom)
                    </label>
                    <div class="alert alert-light border-danger">
                        <div class="row">
                            <div class="col-6">
                                <small class="d-block">• <strong>nik</strong></small>
                                <small class="d-block">• <strong>nama</strong></small>
                                <small class="d-block">• <strong>jenis_kelamin</strong></small>
                            </div>
                            <div class="col-6">
                                <small class="d-block">• <strong>tempat_lahir</strong></small>
                                <small class="d-block">• <strong>tanggal_lahir</strong></small>
                                <small class="d-block">• <strong>alamat</strong></small>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <label class="form-label fw-bold text-info">
                        <i class="fas fa-plus-circle me-2"></i>
                        Field Opsional (17 kolom)
                    </label>
                    <div class="alert alert-light border-info">
                        <div class="row">
                            <div class="col-6">
                                <small class="d-block">• no_kk</small>
                                <small class="d-block">• agama</small>
                                <small class="d-block">• status_kawin</small>
                                <small class="d-block">• pekerjaan</small>
                                <small class="d-block">• pendidikan_terakhir</small>
                                <small class="d-block">• golongan_darah</small>
                                <small class="d-block">• kewarganegaraan</small>
                                <small class="d-block">• no_wa</small>
                                <small class="d-block">• status_hidup</small>
                            </div>
                            <div class="col-6">
                                <small class="d-block">• rt</small>
                                <small class="d-block">• rw</small>
                                <small class="d-block">• dusun</small>
                                <small class="d-block">• desa</small>
                                <small class="d-block">• nama_ayah</small>
                                <small class="d-block">• nama_ibu</small>
                                <small class="d-block">• no_akta_lahir</small>
                                <small class="d-block">• tanggal_akta_lahir</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Baris 3: Action Buttons -->
            <div class="row">
                <div class="col-12">
                    <div class="d-flex justify-content-between align-items-center">
                        <div class="text-muted">
                            <i class="fas fa-info-circle me-2"></i>
                            <small>Field wajib harus diisi, field opsional akan menggunakan nilai default jika kosong</small>
                        </div>
                        <div class="d-flex gap-2">
                            <button type="button" class="btn btn-outline-secondary" onclick="toggleImportForm()">
                                <i class="fas fa-times me-2"></i>Batal
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-upload me-2"></i>Import Data Penduduk
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Detail Modal -->
<div class="modal fade" id="detailModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Detail Penduduk</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="detailContent">
                <div class="text-center">
                    <div class="spinner-border" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
            </div>
        </div>
    </div>
</div>

<!-- Bulk Password Generation Modal -->
<div class="modal fade" id="bulkPasswordModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-key me-2"></i>
                    Generate Password Massal
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-info">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Informasi:</strong> Fitur ini akan menggenerate password default (6 digit terakhir NIK) untuk penduduk yang belum memiliki password.
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label class="form-label">Filter Berdasarkan</label>
                        <select class="form-select" id="bulkFilterType">
                            <option value="all">Semua Penduduk Tanpa Password</option>
                            <option value="dusun">Berdasarkan Dusun</option>
                            <option value="rt_rw">Berdasarkan RT/RW</option>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <div id="filterOptions" style="display: none;">
                            <label class="form-label">Pilih Filter</label>
                            <select class="form-select" id="filterValue">
                                <!-- Options akan diisi via JavaScript -->
                            </select>
                        </div>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label class="form-label">Jumlah Maksimal</label>
                        <input type="number" class="form-control" id="maxCount" value="50" min="1" max="500">
                        <small class="text-muted">Maksimal 500 penduduk per batch</small>
                    </div>
                    <div class="col-md-6">
                        <div class="form-check mt-4">
                            <input class="form-check-input" type="checkbox" id="forcePasswordChange" checked>
                            <label class="form-check-label" for="forcePasswordChange">
                                Paksa ganti password saat login pertama
                            </label>
                        </div>
                    </div>
                </div>
                
                <div id="previewContainer" style="display: none;">
                    <h6>Preview Data yang Akan Diproses:</h6>
                    <div class="table-responsive">
                        <table class="table table-sm table-striped">
                            <thead>
                                <tr>
                                    <th>NIK</th>
                                    <th>Nama</th>
                                    <th>Password</th>
                                </tr>
                            </thead>
                            <tbody id="previewTable">
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <div id="resultContainer" style="display: none;">
                    <h6>Hasil Generate Password:</h6>
                    <div id="resultContent"></div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
                <button type="button" class="btn btn-info" id="previewBtn" onclick="previewBulkPassword()">
                    <i class="fas fa-eye me-2"></i>Preview
                </button>
                <button type="button" class="btn btn-warning" id="generateBtn" onclick="generateBulkPassword()" style="display: none;">
                    <i class="fas fa-key me-2"></i>Generate Password
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
.bg-pink {
    background-color: #e91e63 !important;
}

.text-monospace {
    font-family: 'Courier New', Courier, monospace;
    font-size: 0.9em;
}

.table th {
    border-top: none;
    font-weight: 600;
    background: #f8f9fa;
}

#importForm {
    border: 2px dashed #dee2e6;
    border-radius: 0.5rem;
}

#importForm .card-header {
    background: #f8f9fa;
    border-bottom: 1px solid #dee2e6;
}

#importForm.show {
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.table td {
    vertical-align: middle;
}

.btn-group-sm .btn {
    padding: 0.25rem 0.5rem;
    font-size: 0.875rem;
}

.badge {
    font-size: 0.75rem;
    font-weight: 500;
}

/* Import/Export styling */
.btn-group .dropdown-menu {
    border-radius: 8px;
    box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
}

/* Import Form Styling */
#importForm .alert {
    margin-bottom: 0;
    padding: 0.75rem;
}

#importForm .alert small {
    margin-bottom: 0.25rem;
    line-height: 1.4;
}

#importForm .form-text {
    font-size: 0.8rem;
    margin-top: 0.25rem;
}

#importForm .btn {
    border-radius: 6px;
    font-weight: 500;
}

#importForm .border-danger {
    border-color: #dc3545 !important;
    border-width: 2px !important;
}

#importForm .border-info {
    border-color: #0dcaf0 !important;
    border-width: 2px !important;
}

#importForm .fw-bold {
    font-weight: 600 !important;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    #importForm .d-flex.justify-content-between {
        flex-direction: column;
        gap: 1rem;
    }
    
    #importForm .d-flex.gap-2 {
        justify-content: center;
    }
}

.dropdown-item {
    padding: 8px 16px;
    transition: all 0.2s ease;
}

.dropdown-item:hover {
    background-color: #f8f9fa;
    transform: translateX(2px);
}

.dropdown-item i {
    width: 20px;
    text-align: center;
}

/* Import modal styling */
#importModal .modal-content {
    border-radius: 12px;
    border: none;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
}

#importModal .modal-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 12px 12px 0 0;
}

#importModal .alert-info {
    border-left: 4px solid #17a2b8;
    background-color: #f0f9ff;
}

#importModal .form-control:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
}
</style>
@endpush

@push('scripts')
<script>
$(document).ready(function() {
    console.log('Penduduk index page loaded');
    
    // Force clean modal backdrops on page load
    if (typeof window.forceCleanModalBackdrops === 'function') {
        window.forceCleanModalBackdrops();
    }
    
    // Initialize tooltips
    $('[data-bs-toggle="tooltip"]').tooltip();
    
    // Show import form if there are session messages
    @if(session('success') || session('error'))
        // Auto show form if there was an import attempt
        setTimeout(() => {
            const importForm = document.getElementById('importForm');
            if (importForm) {
                importForm.style.display = 'block';
                importForm.classList.add('show');
                importForm.scrollIntoView({ behavior: 'smooth', block: 'start' });
            }
        }, 500);
    @endif
});

function showDetail(id) {
    $('#detailModal').modal('show');
    
    fetch(`/admin/penduduk/${id}`, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const penduduk = data.data;
            $('#detailContent').html(`
                <div class="row">
                    <div class="col-md-4 text-center">
                        <img src="${penduduk.foto}" 
                             class="img-fluid rounded-circle mb-3" 
                             style="max-height: 200px; max-width: 200px; object-fit: cover;"
                             alt="${penduduk.nama}">
                        <h5>${penduduk.nama}</h5>
                        <p class="text-muted">${penduduk.nik}</p>
                    </div>
                    <div class="col-md-8">
                        <table class="table table-borderless">
                            <tr><td><strong>Jenis Kelamin:</strong></td><td>${penduduk.jenis_kelamin === 'L' ? 'Laki-laki' : 'Perempuan'}</td></tr>
                            <tr><td><strong>Tempat/Tgl Lahir:</strong></td><td>${penduduk.tempat_lahir}, ${penduduk.tanggal_lahir}</td></tr>
                            <tr><td><strong>Alamat:</strong></td><td>${penduduk.alamat}</td></tr>
                            <tr><td><strong>RT/RW:</strong></td><td>${penduduk.rt}/${penduduk.rw}</td></tr>
                            <tr><td><strong>Agama:</strong></td><td>${penduduk.agama}</td></tr>
                            <tr><td><strong>Status Kawin:</strong></td><td>${penduduk.status_kawin}</td></tr>
                            <tr><td><strong>Pekerjaan:</strong></td><td>${penduduk.pekerjaan || '-'}</td></tr>
                            <tr><td><strong>Kewarganegaraan:</strong></td><td>${penduduk.kewarganegaraan}</td></tr>
                        </table>
                    </div>
                </div>
            `);
        } else {
            $('#detailContent').html('<div class="alert alert-danger">Gagal memuat detail penduduk</div>');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        $('#detailContent').html('<div class="alert alert-danger">Terjadi kesalahan saat memuat data</div>');
    });
}

function deletePenduduk(id) {
    if (confirm('Apakah Anda yakin ingin menghapus data penduduk ini?')) {
        fetch(`/admin/penduduk/${id}`, {
            method: 'DELETE',
            headers: {
                'Accept': 'application/json',
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Data penduduk berhasil dihapus');
                location.reload();
            } else {
                alert('Gagal menghapus data penduduk');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Terjadi kesalahan saat menghapus data');
        });
    }
}

// Export function
function exportData() {
    // Get current filters
    const search = $('input[name="search"]').val() || '';
    const dusun = $('select[name="dusun"]').val() || '';
    
    // Build URL with filters
    let url = '{{ route("admin.penduduk.export") }}';
    const params = new URLSearchParams();
    
    if (search) params.append('search', search);
    if (dusun) params.append('dusun', dusun);
    
    if (params.toString()) {
        url += '?' + params.toString();
    }
    
    // Download file
    window.location.href = url;
}

// Download template function
function downloadTemplate() {
    window.location.href = '{{ route("admin.penduduk.template") }}';
}

// Toggle import form
function toggleImportForm() {
    const form = document.getElementById('importForm');
    if (form.style.display === 'none' || form.style.display === '') {
        form.style.display = 'block';
        form.classList.add('show');
        // Scroll to form
        setTimeout(() => {
            form.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }, 100);
    } else {
        form.classList.remove('show');
        setTimeout(() => {
            form.style.display = 'none';
        }, 300);
    }
}

// Handle import form
$('form[action="{{ route("admin.penduduk.import") }}"]').on('submit', function(e) {
    const fileInput = $(this).find('input[type="file"]')[0];
    const file = fileInput ? fileInput.files[0] : null;
    
    if (!file) {
        e.preventDefault();
        alert('Silakan pilih file CSV terlebih dahulu.');
        return false;
    }
    
    if (file.size > 2 * 1024 * 1024) { // 2MB
        e.preventDefault();
        alert('Ukuran file terlalu besar. Maksimal 2MB.');
        return false;
    }
    
    // Validate file extension
    const fileExtension = file.name.split('.').pop().toLowerCase();
    if (fileExtension !== 'csv') {
        e.preventDefault();
        alert('Format file tidak didukung. Gunakan file .csv');
        return false;
    }
    
    // Show loading
    const submitBtn = $(this).find('button[type="submit"]');
    const originalText = submitBtn.html();
    submitBtn.html('<i class="fas fa-spinner fa-spin me-2"></i>Mengimpor...');
    submitBtn.prop('disabled', true);
    
    return true;
});

// Reset form when page loads
$(document).ready(function() {
    // Reset import form
    const importForm = document.getElementById('importForm');
    if (importForm) {
        const form = importForm.querySelector('form');
        if (form) {
            form.reset();
        }
    }
});

// Bulk Password Generation Functions
function showBulkPasswordModal() {
    $('#bulkPasswordModal').modal('show');
    
    // Reset form
    document.getElementById('bulkFilterType').value = 'all';
    document.getElementById('maxCount').value = '50';
    document.getElementById('forcePasswordChange').checked = true;
    document.getElementById('filterOptions').style.display = 'none';
    document.getElementById('previewContainer').style.display = 'none';
    document.getElementById('resultContainer').style.display = 'none';
    document.getElementById('previewBtn').style.display = 'inline-block';
    document.getElementById('generateBtn').style.display = 'none';
}

// Handle filter type change
document.getElementById('bulkFilterType').addEventListener('change', function() {
    const filterType = this.value;
    const filterOptions = document.getElementById('filterOptions');
    const filterValue = document.getElementById('filterValue');
    
    if (filterType === 'all') {
        filterOptions.style.display = 'none';
    } else {
        filterOptions.style.display = 'block';
        
        // Clear existing options
        filterValue.innerHTML = '';
        
        if (filterType === 'dusun') {
            // Populate dusun options
            @foreach($dusunList as $dusun)
                filterValue.innerHTML += '<option value="{{ $dusun }}">{{ $dusun }}</option>';
            @endforeach
        } else if (filterType === 'rt_rw') {
            // Populate RT/RW options
            @foreach($rtList as $rt)
                @foreach($rwList as $rw)
                    filterValue.innerHTML += '<option value="{{ $rt }}/{{ $rw }}">RT {{ $rt }} RW {{ $rw }}</option>';
                @endforeach
            @endforeach
        }
    }
});

function previewBulkPassword() {
    const filterType = document.getElementById('bulkFilterType').value;
    const filterValue = document.getElementById('filterValue').value;
    const maxCount = document.getElementById('maxCount').value;
    
    const data = {
        filter_type: filterType,
        filter_value: filterValue,
        max_count: maxCount,
        preview: true
    };
    
    fetch('/admin/penduduk/bulk-password-preview', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const previewTable = document.getElementById('previewTable');
            previewTable.innerHTML = '';
            
            data.penduduk.forEach(p => {
                const password = p.nik.substr(-6);
                previewTable.innerHTML += `
                    <tr>
                        <td>${p.nik}</td>
                        <td>${p.nama}</td>
                        <td><code>${password}</code></td>
                    </tr>
                `;
            });
            
            document.getElementById('previewContainer').style.display = 'block';
            document.getElementById('generateBtn').style.display = 'inline-block';
            
            if (data.penduduk.length === 0) {
                previewTable.innerHTML = '<tr><td colspan="3" class="text-center text-muted">Tidak ada penduduk yang memenuhi kriteria</td></tr>';
                document.getElementById('generateBtn').style.display = 'none';
            }
        } else {
            alert('Gagal memuat preview: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Terjadi kesalahan saat memuat preview');
    });
}

function generateBulkPassword() {
    if (!confirm('Apakah Anda yakin ingin generate password untuk semua penduduk yang dipilih?')) {
        return;
    }
    
    const filterType = document.getElementById('bulkFilterType').value;
    const filterValue = document.getElementById('filterValue').value;
    const maxCount = document.getElementById('maxCount').value;
    const forceChange = document.getElementById('forcePasswordChange').checked;
    
    const generateBtn = document.getElementById('generateBtn');
    const originalText = generateBtn.innerHTML;
    generateBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Memproses...';
    generateBtn.disabled = true;
    
    const data = {
        filter_type: filterType,
        filter_value: filterValue,
        max_count: maxCount,
        force_change: forceChange,
        preview: false
    };
    
    fetch('/admin/penduduk/bulk-password-generate', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            let resultHtml = '';
            
            if (data.success_count > 0) {
                resultHtml = `
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle me-2"></i>
                        <strong>Berhasil!</strong> ${data.success_count} password berhasil digenerate.
                    </div>
                `;
            } else {
                resultHtml = `
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Informasi:</strong> Tidak ada penduduk yang memerlukan password baru. Semua penduduk sudah memiliki password.
                    </div>
                `;
            }
            
            if (data.results && data.results.length > 0) {
                resultHtml += `
                    <div class="table-responsive">
                        <table class="table table-sm table-striped">
                            <thead>
                                <tr>
                                    <th>NIK</th>
                                    <th>Nama</th>
                                    <th>Password</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                `;
                
                data.results.forEach(result => {
                    const statusClass = result.status === 'Success' ? 'text-success' : 'text-danger';
                    resultHtml += `
                        <tr>
                            <td>${result.nik}</td>
                            <td>${result.nama}</td>
                            <td><code>${result.password || '-'}</code></td>
                            <td class="${statusClass}">${result.status}</td>
                        </tr>
                    `;
                });
                
                resultHtml += '</tbody></table></div>';
            }
            
            document.getElementById('resultContent').innerHTML = resultHtml;
            document.getElementById('resultContainer').style.display = 'block';
            document.getElementById('previewContainer').style.display = 'none';
            document.getElementById('previewBtn').style.display = 'none';
            document.getElementById('generateBtn').style.display = 'none';
            
            // Reload page after 3 seconds
            setTimeout(() => {
                location.reload();
            }, 3000);
            
        } else {
            alert('Gagal generate password: ' + data.message);
        }
        
        generateBtn.innerHTML = originalText;
        generateBtn.disabled = false;
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Terjadi kesalahan saat generate password');
        generateBtn.innerHTML = originalText;
        generateBtn.disabled = false;
    });
}
</script>
@endpush 