@extends('layouts.app')

@section('title', 'Visualisasi Pohon Keluarga - ' . $penduduk->nama)

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.pohon-keluarga.index') }}">Pohon Keluarga</a></li>
    <li class="breadcrumb-item"><a href="{{ route('admin.pohon-keluarga.show', $penduduk->nik) }}">{{ $penduduk->nama }}</a></li>
    <li class="breadcrumb-item active">Visualisasi</li>
@endsection

@section('content')
<div class="row">
    <!-- Header Card -->
    <div class="col-12 mb-4">
        <div class="card border-0 shadow">
            <div class="card-header bg-gradient-primary text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-project-diagram me-2"></i>
                        Visualisasi Pohon Keluarga
                    </h5>
                    <div class="d-flex gap-2">
                        <button class="btn btn-light btn-sm" onclick="resetZoom()">
                            <i class="fas fa-search-minus me-1"></i>Reset Zoom
                        </button>
                        <button class="btn btn-light btn-sm" onclick="centerView()">
                            <i class="fas fa-crosshairs me-1"></i>Center
                        </button>
                        <button class="btn btn-light btn-sm" onclick="exportDiagram()">
                            <i class="fas fa-download me-1"></i>Export PNG
                        </button>
                    </div>
                </div>
            </div>
            <div class="card-body p-2">
                <div class="row align-items-center">
                    <div class="col-md-8">
                        <div class="d-flex align-items-center">
                            <div class="avatar-sm {{ $penduduk->jenis_kelamin == 'L' ? 'bg-primary' : 'bg-pink' }} me-3">
                                <i class="fas fa-{{ $penduduk->jenis_kelamin == 'L' ? 'male' : 'female' }} text-white"></i>
                            </div>
                            <div>
                                <h6 class="mb-1">{{ $penduduk->nama }}</h6>
                                <small class="text-muted">
                                    NIK: {{ $penduduk->nik }} | 
                                    {{ $penduduk->jenis_kelamin == 'L' ? 'Laki-laki' : 'Perempuan' }} | 
                                    {{ $penduduk->umur }} tahun
                                </small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4 text-end">
                        <div class="btn-group btn-group-sm" role="group">
                            <button type="button" class="btn btn-outline-secondary" onclick="changeView('ancestors')">
                                <i class="fas fa-arrow-up me-1"></i>Garis Keturunan
                            </button>
                            <button type="button" class="btn btn-outline-secondary active" onclick="changeView('descendants')">
                                <i class="fas fa-arrow-down me-1"></i>Pohon Keluarga
                            </button>
                            <button type="button" class="btn btn-outline-secondary" onclick="changeView('full')">
                                <i class="fas fa-sitemap me-1"></i>Lengkap
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Visualization Container -->
    <div class="col-12 mb-4">
        <div class="card">
            <div class="card-header bg-light d-flex justify-content-between align-items-center">
                <h6 class="mb-0">
                    <i class="fas fa-eye text-primary me-2"></i>
                    Diagram Interaktif
                </h6>
                <div class="d-flex align-items-center gap-3">
                    <!-- Legend -->
                    <div class="d-flex align-items-center gap-3">
                        <small class="text-muted">Legenda:</small>
                        <span class="badge bg-primary">
                            <i class="fas fa-male me-1"></i>Laki-laki
                        </span>
                        <span class="badge bg-pink">
                            <i class="fas fa-female me-1"></i>Perempuan
                        </span>
                        <span class="badge bg-warning">
                            <i class="fas fa-star me-1"></i>Fokus
                        </span>
                    </div>
                </div>
            </div>
            <div class="card-body p-0">
                <div id="family-tree-visualization" style="min-height: 600px; background: #f8f9fa;">
                    <div class="d-flex align-items-center justify-content-center h-100">
                        <div class="text-center py-5">
                            <div class="spinner-border text-primary mb-3" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <p class="text-muted">Memuat visualisasi pohon keluarga...</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Family Information Panel -->
    <div class="col-12 mb-4">
        <div class="card">
            <div class="card-header bg-light">
                <h6 class="mb-0">
                    <i class="fas fa-info-circle text-info me-2"></i>
                    Informasi Keluarga
                </h6>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="card bg-light border-0 h-100">
                            <div class="card-body">
                                <h6 class="card-title">
                                    <i class="fas fa-chart-bar text-primary me-2"></i>
                                    Statistik Keluarga
                                </h6>
                                <div id="family-stats" class="mt-3">
                                    <div class="d-flex justify-content-between mb-2">
                                        <span>Total Anggota:</span>
                                        <span class="fw-bold" id="total-members">-</span>
                                    </div>
                                    <div class="d-flex justify-content-between mb-2">
                                        <span>Generasi:</span>
                                        <span class="fw-bold" id="total-generations">-</span>
                                    </div>
                                    <div class="d-flex justify-content-between mb-2">
                                        <span>Laki-laki:</span>
                                        <span class="fw-bold text-primary" id="male-count">-</span>
                                    </div>
                                    <div class="d-flex justify-content-between">
                                        <span>Perempuan:</span>
                                        <span class="fw-bold text-pink" id="female-count">-</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card bg-light border-0 h-100">
                            <div class="card-body">
                                <h6 class="card-title">
                                    <i class="fas fa-mouse-pointer text-success me-2"></i>
                                    Kontrol Interaktif
                                </h6>
                                <div class="mt-3">
                                    <p class="small text-muted mb-2">
                                        <i class="fas fa-hand-pointer me-2"></i>
                                        Klik pada node untuk melihat detail
                                    </p>
                                    <p class="small text-muted mb-2">
                                        <i class="fas fa-search-plus me-2"></i>
                                        Scroll untuk zoom in/out
                                    </p>
                                    <p class="small text-muted mb-2">
                                        <i class="fas fa-arrows-alt me-2"></i>
                                        Drag untuk menggeser diagram
                                    </p>
                                    <p class="small text-muted mb-0">
                                        <i class="fas fa-expand me-2"></i>
                                        Double-click untuk focus pada node
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Action Buttons -->
    <div class="col-12">
        <div class="d-flex justify-content-between">
            <a href="{{ route('admin.pohon-keluarga.show', $penduduk->nik) }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Kembali ke Detail
            </a>
            <div class="d-flex gap-2">
                <a href="{{ route('admin.pohon-keluarga.edit', $penduduk->nik) }}" 
                   class="btn btn-warning">
                    <i class="fas fa-edit me-2"></i>Edit Relasi
                </a>
                <a href="{{ route('admin.pohon-keluarga.index') }}" 
                   class="btn btn-info">
                    <i class="fas fa-list me-2"></i>Daftar Keluarga
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Node Detail Modal -->
<div class="modal fade" id="nodeDetailModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-user me-2"></i>
                    Detail Anggota Keluarga
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="nodeDetailContent">
                <!-- Content will be loaded dynamically -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
                <button type="button" class="btn btn-primary" onclick="focusOnNode()">
                    <i class="fas fa-crosshairs me-2"></i>Fokus pada Node
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/d3@7/dist/d3.min.css">
<style>
.avatar-sm {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 16px;
}

.bg-pink {
    background-color: #e91e63 !important;
}

.text-pink {
    color: #e91e63 !important;
}

.card-gradient-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

#family-tree-visualization {
    position: relative;
    overflow: hidden;
    cursor: grab;
}

#family-tree-visualization:active {
    cursor: grabbing;
}

.family-node {
    cursor: pointer;
    transition: all 0.3s ease;
}

.family-node:hover {
    transform: scale(1.1);
}

.family-node.focus {
    stroke: #ffc107;
    stroke-width: 3px;
}

.family-link {
    fill: none;
    stroke: #999;
    stroke-width: 1.5px;
    stroke-opacity: 0.6;
}

.family-text {
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    font-size: 12px;
    text-anchor: middle;
    fill: #333;
    pointer-events: none;
}

.zoom-controls {
    position: absolute;
    top: 10px;
    right: 10px;
    z-index: 1000;
}

.btn-group-sm .btn {
    font-size: 0.75rem;
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/d3@7/dist/d3.min.js"></script>
<script>
let familyTreeData = @json($treeData ?? []);
let currentView = 'descendants';
let selectedNodeId = null;
let svg, g, zoom;

$(document).ready(function() {
    initializeFamilyTree();
    updateFamilyStats();
});

function initializeFamilyTree() {
    // Clear existing content
    d3.select("#family-tree-visualization").selectAll("*").remove();
    
    // Set dimensions
    const container = document.getElementById('family-tree-visualization');
    const width = container.clientWidth;
    const height = container.clientHeight;
    
    // Create SVG
    svg = d3.select("#family-tree-visualization")
        .append("svg")
        .attr("width", width)
        .attr("height", height);
    
    // Create zoom behavior
    zoom = d3.zoom()
        .scaleExtent([0.1, 3])
        .on("zoom", function(event) {
            g.attr("transform", event.transform);
        });
    
    svg.call(zoom);
    
    // Create main group
    g = svg.append("g");
    
    // Load family tree
    if (familyTreeData && Object.keys(familyTreeData).length > 0) {
        renderFamilyTree(familyTreeData);
    } else {
        showNoDataMessage();
    }
}

function renderFamilyTree(data) {
    // Create tree layout
    const treeLayout = d3.tree()
        .size([600, 400])
        .separation((a, b) => (a.parent === b.parent ? 1 : 2) / a.depth);
    
    // Convert data to hierarchy
    const root = d3.hierarchy(data);
    
    // Apply tree layout
    treeLayout(root);
    
    // Center the tree
    const nodes = root.descendants();
    const links = root.links();
    
    // Adjust positions
    nodes.forEach(d => {
        d.x += 300; // Center horizontally
        d.y += 50;  // Add top margin
    });
    
    // Draw links
    g.selectAll('.family-link')
        .data(links)
        .enter()
        .append('path')
        .attr('class', 'family-link')
        .attr('d', d3.linkVertical()
            .x(d => d.x)
            .y(d => d.y)
        );
    
    // Draw nodes
    const nodeGroups = g.selectAll('.family-node')
        .data(nodes)
        .enter()
        .append('g')
        .attr('class', 'family-node')
        .attr('transform', d => `translate(${d.x},${d.y})`)
        .on('click', function(event, d) {
            showNodeDetail(d.data);
        })
        .on('dblclick', function(event, d) {
            focusOnSpecificNode(d);
        });
    
    // Add circles for nodes
    nodeGroups.append('circle')
        .attr('r', 25)
        .attr('fill', d => {
            if (d.data.id === '{{ $penduduk->nik }}') return '#ffc107'; // Focus node
            return d.data.gender === 'L' ? '#007bff' : '#e91e63';
        })
        .attr('stroke', '#fff')
        .attr('stroke-width', 2);
    
    // Add icons
    nodeGroups.append('text')
        .attr('class', 'family-text')
        .attr('y', 5)
        .style('font-family', 'FontAwesome')
        .style('font-size', '16px')
        .style('fill', 'white')
        .text(d => d.data.gender === 'L' ? '\uf222' : '\uf221'); // Male/Female icons
    
    // Add names
    nodeGroups.append('text')
        .attr('class', 'family-text')
        .attr('y', 45)
        .style('font-size', '11px')
        .style('font-weight', 'bold')
        .text(d => {
            const name = d.data.name || 'Tidak diketahui';
            return name.length > 15 ? name.substring(0, 15) + '...' : name;
        });
    
    // Add age
    nodeGroups.append('text')
        .attr('class', 'family-text')
        .attr('y', 58)
        .style('font-size', '9px')
        .style('fill', '#666')
        .text(d => d.data.age ? `${d.data.age} tahun` : '');
    
    // Center initial view
    centerView();
}

function showNodeDetail(nodeData) {
    selectedNodeId = nodeData.id;
    
    const content = `
        <div class="row">
            <div class="col-md-4 text-center">
                <div class="avatar-large ${nodeData.gender === 'L' ? 'bg-primary' : 'bg-pink'} mb-3">
                    <i class="fas fa-${nodeData.gender === 'L' ? 'male' : 'female'} text-white"></i>
                </div>
            </div>
            <div class="col-md-8">
                <h5>${nodeData.name || 'Tidak diketahui'}</h5>
                <table class="table table-sm">
                    <tr>
                        <td><strong>NIK:</strong></td>
                        <td>${nodeData.id || '-'}</td>
                    </tr>
                    <tr>
                        <td><strong>Jenis Kelamin:</strong></td>
                        <td>${nodeData.gender === 'L' ? 'Laki-laki' : 'Perempuan'}</td>
                    </tr>
                    <tr>
                        <td><strong>Umur:</strong></td>
                        <td>${nodeData.age || '-'} tahun</td>
                    </tr>
                    <tr>
                        <td><strong>Tanggal Lahir:</strong></td>
                        <td>${nodeData.birth_date || '-'}</td>
                    </tr>
                    <tr>
                        <td><strong>Status Keluarga:</strong></td>
                        <td>${nodeData.status_keluarga || '-'}</td>
                    </tr>
                    <tr>
                        <td><strong>No. KK:</strong></td>
                        <td>${nodeData.no_kk || '-'}</td>
                    </tr>
                </table>
            </div>
        </div>
        <hr>
        <div class="text-center">
            <a href="/admin/pohon-keluarga/${nodeData.id}" class="btn btn-primary btn-sm me-2">
                <i class="fas fa-eye me-1"></i>Lihat Detail
            </a>
            <a href="/admin/pohon-keluarga/${nodeData.id}/edit" class="btn btn-warning btn-sm">
                <i class="fas fa-edit me-1"></i>Edit Relasi
            </a>
        </div>
    `;
    
    document.getElementById('nodeDetailContent').innerHTML = content;
    new bootstrap.Modal(document.getElementById('nodeDetailModal')).show();
}

function updateFamilyStats() {
    if (!familyTreeData || Object.keys(familyTreeData).length === 0) {
        document.getElementById('total-members').textContent = '0';
        document.getElementById('total-generations').textContent = '0';
        document.getElementById('male-count').textContent = '0';
        document.getElementById('female-count').textContent = '0';
        return;
    }
    
    // Count statistics from tree data
    const stats = calculateTreeStats(familyTreeData);
    
    document.getElementById('total-members').textContent = stats.totalMembers;
    document.getElementById('total-generations').textContent = stats.generations;
    document.getElementById('male-count').textContent = stats.maleCount;
    document.getElementById('female-count').textContent = stats.femaleCount;
}

function calculateTreeStats(data, depth = 0, stats = { totalMembers: 0, generations: 0, maleCount: 0, femaleCount: 0 }) {
    stats.totalMembers++;
    stats.generations = Math.max(stats.generations, depth + 1);
    
    if (data.gender === 'L') {
        stats.maleCount++;
    } else {
        stats.femaleCount++;
    }
    
    // Process children
    if (data.children && data.children.length > 0) {
        data.children.forEach(child => {
            calculateTreeStats(child, depth + 1, stats);
        });
    }
    
    return stats;
}

function changeView(viewType) {
    currentView = viewType;
    
    // Update button states
    document.querySelectorAll('.btn-group .btn').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.classList.add('active');
    
    // Re-render tree based on view
    initializeFamilyTree();
}

function resetZoom() {
    svg.transition().duration(750).call(
        zoom.transform,
        d3.zoomIdentity
    );
}

function centerView() {
    const bounds = g.node().getBBox();
    const parent = g.node().parentElement;
    const fullWidth = parent.clientWidth || parent.parentNode.clientWidth;
    const fullHeight = parent.clientHeight || parent.parentNode.clientHeight;
    const width = bounds.width;
    const height = bounds.height;
    const midX = bounds.x + width / 2;
    const midY = bounds.y + height / 2;
    
    if (width == 0 || height == 0) return; // nothing to fit
    
    const scale = 0.8 / Math.max(width / fullWidth, height / fullHeight);
    const translate = [fullWidth / 2 - scale * midX, fullHeight / 2 - scale * midY];
    
    svg.transition().duration(750).call(
        zoom.transform,
        d3.zoomIdentity.translate(translate[0], translate[1]).scale(scale)
    );
}

function focusOnNode() {
    if (selectedNodeId) {
        window.location.href = `/admin/pohon-keluarga/${selectedNodeId}`;
    }
}

function focusOnSpecificNode(node) {
    // Highlight the node
    g.selectAll('.family-node').classed('focus', false);
    d3.select(node).classed('focus', true);
    
    // Center on the node
    const transform = d3.zoomTransform(svg.node());
    const x = transform.invertX(node.x);
    const y = transform.invertY(node.y);
    
    svg.transition().duration(750).call(
        zoom.transform,
        d3.zoomIdentity.translate(300 - node.x, 200 - node.y).scale(1.5)
    );
}

function exportDiagram() {
    // Get SVG element
    const svgElement = document.querySelector('#family-tree-visualization svg');
    
    // Create canvas
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    
    // Set canvas size
    canvas.width = svgElement.getAttribute('width');
    canvas.height = svgElement.getAttribute('height');
    
    // Convert SVG to image
    const data = new XMLSerializer().serializeToString(svgElement);
    const DOMURL = window.URL || window.webkitURL || window;
    const img = new Image();
    const svgBlob = new Blob([data], { type: 'image/svg+xml;charset=utf-8' });
    const url = DOMURL.createObjectURL(svgBlob);
    
    img.onload = function() {
        ctx.drawImage(img, 0, 0);
        DOMURL.revokeObjectURL(url);
        
        // Download image
        const imgURI = canvas.toDataURL('image/png').replace('image/png', 'image/octet-stream');
        const evt = new MouseEvent('click', {
            view: window,
            bubbles: false,
            cancelable: true
        });
        
        const a = document.createElement('a');
        a.setAttribute('download', `pohon-keluarga-${Date.now()}.png`);
        a.setAttribute('href', imgURI);
        a.setAttribute('target', '_blank');
        a.dispatchEvent(evt);
    };
    
    img.src = url;
}

function showNoDataMessage() {
    g.append('text')
        .attr('x', 300)
        .attr('y', 200)
        .attr('text-anchor', 'middle')
        .style('font-size', '18px')
        .style('fill', '#666')
        .text('Data pohon keluarga tidak tersedia');
    
    g.append('text')
        .attr('x', 300)
        .attr('y', 230)
        .attr('text-anchor', 'middle')
        .style('font-size', '14px')
        .style('fill', '#999')
        .text('Silakan edit relasi keluarga untuk membuat pohon keluarga');
}

// Handle window resize
window.addEventListener('resize', function() {
    setTimeout(initializeFamilyTree, 100);
});
</script>
@endpush
