<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

class ImageHelper
{
    /**
     * Get image URL with fallback
     */
    public static function getImageUrl(?string $path, string $fallback = '/img/no-image.png'): string
    {
        if (!$path) {
            return asset($fallback);
        }

        // Check if file exists in storage
        if (Storage::disk('public')->exists($path)) {
            // Use request URL or fallback to localhost
            $baseUrl = request() ? request()->getSchemeAndHttpHost() : 'http://127.0.0.1:8000';
            return $baseUrl . '/storage/' . $path;
        }

        // Check if it's an absolute URL
        if (filter_var($path, FILTER_VALIDATE_URL)) {
            return $path;
        }

        // Return fallback if file not found
        return asset($fallback);
    }

    /**
     * Get avatar URL with fallback
     */
    public static function getAvatarUrl(?string $path, string $name = 'User'): string
    {
        if (!$path) {
            return self::generateAvatarUrl($name);
        }

        if (Storage::disk('public')->exists($path)) {
            // Use request URL or fallback to localhost
            $baseUrl = request() ? request()->getSchemeAndHttpHost() : 'http://127.0.0.1:8000';
            return $baseUrl . '/storage/' . $path;
        }

        return self::generateAvatarUrl($name);
    }

    /**
     * Generate avatar URL using UI Avatars service
     */
    public static function generateAvatarUrl(string $name, int $size = 200): string
    {
        $initials = self::getInitials($name);
        $backgroundColor = self::generateColor($name);
        
        return "https://ui-avatars.com/api/?" . http_build_query([
            'name' => $initials,
            'size' => $size,
            'background' => $backgroundColor,
            'color' => 'ffffff',
            'bold' => true,
            'rounded' => true
        ]);
    }

    /**
     * Get initials from name
     */
    public static function getInitials(string $name): string
    {
        $words = explode(' ', trim($name));
        $initials = '';
        
        foreach ($words as $word) {
            if (!empty($word)) {
                $initials .= strtoupper($word[0]);
                if (strlen($initials) >= 2) break;
            }
        }
        
        return $initials ?: 'U';
    }

    /**
     * Generate consistent color based on name
     */
    public static function generateColor(string $name): string
    {
        $colors = [
            '3498db', '9b59b6', 'e74c3c', 'f39c12', 
            '2ecc71', '1abc9c', '34495e', 'e67e22',
            '95a5a6', '8e44ad', 'c0392b', 'd35400'
        ];
        
        $hash = crc32($name);
        $index = abs($hash) % count($colors);
        
        return $colors[$index];
    }

    /**
     * Check if image file exists
     */
    public static function exists(?string $path): bool
    {
        if (!$path) {
            return false;
        }

        return Storage::disk('public')->exists($path);
    }

    /**
     * Get file size in human readable format
     */
    public static function getFileSize(?string $path): ?string
    {
        if (!$path || !Storage::disk('public')->exists($path)) {
            return null;
        }

        $size = Storage::disk('public')->size($path);
        return self::formatBytes($size);
    }

    /**
     * Format bytes to human readable format
     */
    public static function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }

    /**
     * Get image dimensions
     */
    public static function getImageDimensions(?string $path): ?array
    {
        if (!$path || !Storage::disk('public')->exists($path)) {
            return null;
        }

        $fullPath = Storage::disk('public')->path($path);
        $imageInfo = getimagesize($fullPath);
        
        if ($imageInfo === false) {
            return null;
        }

        return [
            'width' => $imageInfo[0],
            'height' => $imageInfo[1],
            'type' => $imageInfo[2],
            'mime' => $imageInfo['mime']
        ];
    }

    /**
     * Create thumbnail (requires intervention/image package)
     */
    public static function createThumbnail(string $path, int $width = 150, int $height = 150): ?string
    {
        if (!Storage::disk('public')->exists($path)) {
            return null;
        }

        // This would require intervention/image package
        // For now, return original path
        return $path;
    }

    /**
     * Validate image file
     */
    public static function validateImage($file): array
    {
        $errors = [];
        
        if (!$file) {
            $errors[] = 'No file provided';
            return $errors;
        }

        // Check if it's a valid uploaded file
        if (!$file->isValid()) {
            $errors[] = 'Invalid file upload';
            return $errors;
        }

        // Check file size (2MB max)
        $maxSize = config('app.max_upload_size', 2048) * 1024;
        if ($file->getSize() > $maxSize) {
            $errors[] = 'File size exceeds maximum allowed size (' . self::formatBytes($maxSize) . ')';
        }

        // Check mime type
        $allowedMimes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif', 'image/webp'];
        if (!in_array($file->getMimeType(), $allowedMimes)) {
            $errors[] = 'Invalid image format. Allowed: JPEG, PNG, JPG, GIF, WebP';
        }

        // Check image dimensions if needed
        $imageInfo = getimagesize($file->getPathname());
        if ($imageInfo === false) {
            $errors[] = 'Invalid image file';
        }

        return $errors;
    }
} 