<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ObjekWisata;
use App\Models\TiketWisata;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class ObjekWisataController extends Controller
{
    public function index(Request $request)
    {
        $query = ObjekWisata::with(['user', 'lastUpdatedBy']);
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_wisata', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhere('alamat_lengkap', 'like', "%{$search}%")
                  ->orWhere('desa_kelurahan', 'like', "%{$search}%");
            });
        }
        
        // Filter by category
        if ($request->filled('kategori')) {
            $query->where('kategori_wisata', $request->kategori);
        }
        
        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        // Filter by featured
        if ($request->filled('featured')) {
            $query->where('is_featured', $request->featured === '1');
        }
        
        $objekWisata = $query->paginate(20)->withQueryString();
        
        // Get filter options
        $kategoriList = $this->getKategoriList();
        
        // Statistics
        $stats = [
            'total' => ObjekWisata::count(),
            'aktif' => ObjekWisata::aktif()->count(),
            'featured' => ObjekWisata::featured()->count(),
            'recommended' => ObjekWisata::recommended()->count(),
        ];
        
        return view('admin.objek-wisata.index', compact('objekWisata', 'kategoriList', 'stats'));
    }
    
    public function create()
    {
        $kategoriList = $this->getKategoriList();
        
        return view('admin.objek-wisata.create', compact('kategoriList'));
    }
    
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nama_wisata' => 'required|string|max:255',
            'kategori_wisata' => 'required|string',
            'deskripsi' => 'required|string',
            'alamat_lengkap' => 'required|string',
            'harga_tiket_dewasa' => 'required|numeric|min:0',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                           ->withErrors($validator)
                           ->withInput();
        }
        
        $data = $request->all();
        $data['user_id'] = auth()->id();
        $data['status'] = 'aktif';
        
        ObjekWisata::create($data);
        
        return redirect()->route('admin.objek-wisata.index')
                        ->with('success', 'Objek wisata berhasil ditambahkan.');
    }
    
    public function show(ObjekWisata $objekWisata)
    {
        $objekWisata->load(['user', 'tiketWisata']);
        
        return view('admin.objek-wisata.show', compact('objekWisata'));
    }
    
    public function edit(ObjekWisata $objekWisata)
    {
        $kategoriList = $this->getKategoriList();
        
        return view('admin.objek-wisata.edit', compact('objekWisata', 'kategoriList'));
    }
    
    public function update(Request $request, ObjekWisata $objekWisata)
    {
        $validator = Validator::make($request->all(), [
            'nama_wisata' => 'required|string|max:255',
            'kategori_wisata' => 'required|string',
            'deskripsi' => 'required|string',
            'alamat_lengkap' => 'required|string',
            'harga_tiket_dewasa' => 'required|numeric|min:0',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                           ->withErrors($validator)
                           ->withInput();
        }
        
        $data = $request->all();
        $data['last_updated_by'] = auth()->id();
        
        $objekWisata->update($data);
        
        return redirect()->route('admin.objek-wisata.index')
                        ->with('success', 'Objek wisata berhasil diperbarui.');
    }
    
    public function destroy(ObjekWisata $objekWisata)
    {
        $objekWisata->delete();
        
        return redirect()->route('admin.objek-wisata.index')
                        ->with('success', 'Objek wisata berhasil dihapus.');
    }
    
    public function toggleStatus(ObjekWisata $objekWisata)
    {
        $newStatus = $objekWisata->status === 'aktif' ? 'nonaktif' : 'aktif';
        $objekWisata->update(['status' => $newStatus]);
        
        return response()->json([
            'success' => true,
            'message' => 'Status objek wisata berhasil diubah.',
            'status' => $newStatus
        ]);
    }
    
    public function toggleFeatured(ObjekWisata $objekWisata)
    {
        $objekWisata->update(['is_featured' => !$objekWisata->is_featured]);
        
        return response()->json([
            'success' => true,
            'message' => 'Status featured berhasil diubah.',
            'is_featured' => $objekWisata->is_featured
        ]);
    }
    
    public function statistik(Request $request)
    {
        $stats = [];
        
        // Monthly statistics for current year
        for ($month = 1; $month <= 12; $month++) {
            $monthlyPengunjung = TiketWisata::whereYear('tanggal_kunjungan', now()->year)
                                           ->whereMonth('tanggal_kunjungan', $month)
                                           ->whereNotNull('check_in_at')
                                           ->sum('jumlah_tiket');
            
            $monthlyPendapatan = TiketWisata::whereYear('tanggal_kunjungan', now()->year)
                                           ->whereMonth('tanggal_kunjungan', $month)
                                           ->where('status_tiket', 'paid')
                                           ->sum('total_bayar');
            
            $stats[] = [
                'bulan' => $month,
                'label' => now()->setMonth($month)->format('M'),
                'pengunjung' => $monthlyPengunjung,
                'pendapatan' => $monthlyPendapatan
            ];
        }
        
        // Top tourism objects
        $topObjek = ObjekWisata::orderBy('total_pengunjung_keseluruhan', 'desc')
                              ->limit(10)
                              ->get(['nama_wisata', 'total_pengunjung_keseluruhan', 'pendapatan_keseluruhan']);
        
        return view('admin.objek-wisata.statistik', compact('stats', 'topObjek'));
    }
    
    public function statistikKunjungan(Request $request)
    {
        $year = $request->get('year', now()->year);
        $month = $request->get('month');
        
        // Build query for statistics
        $query = TiketWisata::with('objekWisata');
        
        // Filter by year
        $query->whereYear('tanggal_kunjungan', $year);
        
        // Filter by month if specified
        if ($month) {
            $query->whereMonth('tanggal_kunjungan', $month);
        }
        
        // Monthly visitor statistics
        $monthlyStats = [];
        for ($m = 1; $m <= 12; $m++) {
            $pengunjung = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                    ->whereMonth('tanggal_kunjungan', $m)
                                    ->where('status_tiket', 'paid')
                                    ->sum('jumlah_tiket');
            
            $pendapatan = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                    ->whereMonth('tanggal_kunjungan', $m)
                                    ->where('status_tiket', 'paid')
                                    ->sum('total_bayar');
            
            $monthlyStats[] = [
                'bulan' => $m,
                'nama_bulan' => now()->setMonth($m)->format('F'),
                'singkatan_bulan' => now()->setMonth($m)->format('M'),
                'pengunjung' => $pengunjung,
                'pendapatan' => $pendapatan
            ];
        }
        
        // Tourism object statistics
        $objekWisataStats = ObjekWisata::with(['tiketWisata' => function($query) use ($year, $month) {
                                         $query->whereYear('tanggal_kunjungan', $year)
                                               ->where('status_tiket', 'paid');
                                         if ($month) {
                                             $query->whereMonth('tanggal_kunjungan', $month);
                                         }
                                     }])
                                     ->get()
                                     ->map(function($objek) {
                                         return [
                                             'id' => $objek->id,
                                             'nama' => $objek->nama_wisata,
                                             'kategori' => $objek->kategori_wisata,
                                             'total_pengunjung' => $objek->tiketWisata->sum('jumlah_tiket'),
                                             'total_pendapatan' => $objek->tiketWisata->sum('total_bayar'),
                                             'total_tiket' => $objek->tiketWisata->count()
                                         ];
                                     })
                                     ->sortByDesc('total_pengunjung')
                                     ->values();
        
        // Overall statistics
        $totalPengunjung = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                     ->when($month, function($query) use ($month) {
                                         return $query->whereMonth('tanggal_kunjungan', $month);
                                     })
                                     ->where('status_tiket', 'paid')
                                     ->sum('jumlah_tiket');
        
        $totalPendapatan = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                     ->when($month, function($query) use ($month) {
                                         return $query->whereMonth('tanggal_kunjungan', $month);
                                     })
                                     ->where('status_tiket', 'paid')
                                     ->sum('total_bayar');
        
        $totalTiket = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                ->when($month, function($query) use ($month) {
                                    return $query->whereMonth('tanggal_kunjungan', $month);
                                })
                                ->where('status_tiket', 'paid')
                                ->count();
        
        $objekWisataAktif = ObjekWisata::where('status', 'aktif')->count();
        
        $overallStats = [
            'total_pengunjung' => $totalPengunjung,
            'total_pendapatan' => $totalPendapatan,
            'total_tiket' => $totalTiket,
            'objek_wisata_aktif' => $objekWisataAktif
        ];
        
        return view('admin.objek-wisata.statistik', compact(
            'monthlyStats', 
            'objekWisataStats', 
            'overallStats', 
            'year', 
            'month'
        ));
    }
    
    private function getKategoriList()
    {
        return [
            'alam' => 'Wisata Alam',
            'budaya' => 'Wisata Budaya',
            'sejarah' => 'Wisata Sejarah',
            'religi' => 'Wisata Religi',
            'kuliner' => 'Wisata Kuliner',
            'adventure' => 'Wisata Adventure',
            'edukasi' => 'Wisata Edukasi',
            'agrowisata' => 'Agrowisata',
            'pantai' => 'Wisata Pantai',
            'gunung' => 'Wisata Gunung',
            'air_terjun' => 'Air Terjun',
            'danau' => 'Wisata Danau',
            'goa' => 'Wisata Goa',
            'taman' => 'Taman Wisata',
            'museum' => 'Museum',
            'candi' => 'Candi',
            'lainnya' => 'Lainnya'
        ];
    }
} 
