<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ProdukUmkm;
use App\Models\Umkm;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class ProdukUmkmController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = ProdukUmkm::with(['umkm', 'user']);
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_produk', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhereHas('umkm', function($umkmQuery) use ($search) {
                      $umkmQuery->where('nama_umkm', 'like', "%{$search}%");
                  });
            });
        }
        
        // Filter by UMKM
        if ($request->filled('umkm_id')) {
            $query->where('umkm_id', $request->umkm_id);
        }
        
        // Filter by category
        if ($request->filled('kategori')) {
            $query->where('kategori_produk', $request->kategori);
        }
        
        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        // Filter by featured
        if ($request->filled('featured')) {
            $query->where('is_featured', $request->featured === '1');
        }
        
        // Filter by bestseller
        if ($request->filled('bestseller')) {
            $query->where('is_bestseller', $request->bestseller === '1');
        }
        
        // Filter by availability
        if ($request->filled('availability')) {
            if ($request->availability === 'available') {
                $query->available();
            } elseif ($request->availability === 'out_of_stock') {
                $query->where('stok', '<=', 0)->where('unlimited_stok', false);
            }
        }
        
        // Sort
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        
        if (in_array($sortBy, ['nama_produk', 'harga_jual', 'stok', 'total_terjual', 'rating_rata_rata', 'created_at'])) {
            $query->orderBy($sortBy, $sortOrder);
        }
        
        $produkUmkm = $query->paginate(20)->withQueryString();
        
        // Get filter options
        $umkmList = Umkm::aktif()->orderBy('nama_umkm')->get();
        $kategoriList = $this->getKategoriList();
        
        // Statistics
        $stats = [
            'total' => ProdukUmkm::count(),
            'aktif' => ProdukUmkm::aktif()->count(),
            'featured' => ProdukUmkm::featured()->count(),
            'bestseller' => ProdukUmkm::bestseller()->count(),
            'out_of_stock' => ProdukUmkm::where('stok', '<=', 0)->where('unlimited_stok', false)->count(),
        ];
        
        return view('admin.produk-umkm.index', compact('produkUmkm', 'umkmList', 'kategoriList', 'stats'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $umkmList = Umkm::aktif()->orderBy('nama_umkm')->get();
        $kategoriList = $this->getKategoriList();
        
        return view('admin.produk-umkm.create', compact('umkmList', 'kategoriList'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'umkm_id' => 'required|exists:umkm,id',
            'nama_produk' => 'required|string|max:255',
            'kategori_produk' => 'required|string',
            'deskripsi' => 'required|string',
            'harga_jual' => 'required|numeric|min:0',
            'harga_modal' => 'nullable|numeric|min:0',
            'stok' => 'required_if:unlimited_stok,false|nullable|integer|min:0',
            'minimal_stok' => 'nullable|integer|min:0',
            'satuan' => 'required|string|max:50',
            'berat' => 'nullable|numeric|min:0',
            'foto_utama' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'galeri_foto.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'video_produk' => 'nullable|url',
            'unlimited_stok' => 'boolean',
            'bisa_dikirim' => 'boolean',
            'hanya_cod' => 'boolean',
            'is_featured' => 'boolean',
            'is_bestseller' => 'boolean',
            'is_new' => 'boolean',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                           ->withErrors($validator)
                           ->withInput();
        }
        
        $data = $request->all();
        $data['user_id'] = auth()->user()->id;
        
        // Handle foto utama upload
        if ($request->hasFile('foto_utama')) {
            $file = $request->file('foto_utama');
            $filename = time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
            $file->storeAs('public/produk', $filename);
            $data['foto_utama'] = $filename;
        }
        
        // Handle galeri foto upload
        if ($request->hasFile('galeri_foto')) {
            $galeriFiles = [];
            foreach ($request->file('galeri_foto') as $file) {
                $filename = time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
                $file->storeAs('public/produk/galeri', $filename);
                $galeriFiles[] = $filename;
            }
            $data['galeri_foto'] = $galeriFiles;
        }
        
        // Handle tags
        if ($request->filled('tags')) {
            $data['tags'] = array_map('trim', explode(',', $request->tags));
        }
        
        // Handle keyword
        if ($request->filled('keyword')) {
            $data['keyword'] = array_map('trim', explode(',', $request->keyword));
        }
        
        // Set default status
        $data['status'] = $data['status'] ?? 'aktif';
        
        ProdukUmkm::create($data);
        
        return redirect()->route('admin.produk-umkm.index')
                        ->with('success', 'Produk UMKM berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(ProdukUmkm $produkUmkm)
    {
        $produkUmkm->load(['umkm', 'user', 'transaksi']);
        $produkUmkm->incrementViewCount();
        
        return view('admin.produk-umkm.show', compact('produkUmkm'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ProdukUmkm $produkUmkm)
    {
        $umkmList = Umkm::aktif()->orderBy('nama_umkm')->get();
        $kategoriList = $this->getKategoriList();
        
        return view('admin.produk-umkm.edit', compact('produkUmkm', 'umkmList', 'kategoriList'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ProdukUmkm $produkUmkm)
    {
        $validator = Validator::make($request->all(), [
            'umkm_id' => 'required|exists:umkm,id',
            'nama_produk' => 'required|string|max:255',
            'kategori_produk' => 'required|string',
            'deskripsi' => 'required|string',
            'harga_jual' => 'required|numeric|min:0',
            'harga_modal' => 'nullable|numeric|min:0',
            'stok' => 'required_if:unlimited_stok,false|nullable|integer|min:0',
            'minimal_stok' => 'nullable|integer|min:0',
            'satuan' => 'required|string|max:50',
            'berat' => 'nullable|numeric|min:0',
            'foto_utama' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'galeri_foto.*' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'video_produk' => 'nullable|url',
            'unlimited_stok' => 'boolean',
            'bisa_dikirim' => 'boolean',
            'hanya_cod' => 'boolean',
            'is_featured' => 'boolean',
            'is_bestseller' => 'boolean',
            'is_new' => 'boolean',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                           ->withErrors($validator)
                           ->withInput();
        }
        
        $data = $request->all();
        
        // Handle foto utama upload
        if ($request->hasFile('foto_utama')) {
            // Delete old photo
            if ($produkUmkm->foto_utama) {
                Storage::delete('public/produk/' . $produkUmkm->foto_utama);
            }
            
            $file = $request->file('foto_utama');
            $filename = time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
            $file->storeAs('public/produk', $filename);
            $data['foto_utama'] = $filename;
        }
        
        // Handle galeri foto upload
        if ($request->hasFile('galeri_foto')) {
            // Delete old gallery photos
            if ($produkUmkm->galeri_foto) {
                foreach ($produkUmkm->galeri_foto as $oldFile) {
                    Storage::delete('public/produk/galeri/' . $oldFile);
                }
            }
            
            $galeriFiles = [];
            foreach ($request->file('galeri_foto') as $file) {
                $filename = time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
                $file->storeAs('public/produk/galeri', $filename);
                $galeriFiles[] = $filename;
            }
            $data['galeri_foto'] = $galeriFiles;
        }
        
        // Handle tags
        if ($request->filled('tags')) {
            $data['tags'] = array_map('trim', explode(',', $request->tags));
        }
        
        // Handle keyword
        if ($request->filled('keyword')) {
            $data['keyword'] = array_map('trim', explode(',', $request->keyword));
        }
        
        $produkUmkm->update($data);
        
        return redirect()->route('admin.produk-umkm.index')
                        ->with('success', 'Produk UMKM berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ProdukUmkm $produkUmkm)
    {
        // Delete photos
        if ($produkUmkm->foto_utama) {
            Storage::delete('public/produk/' . $produkUmkm->foto_utama);
        }
        
        if ($produkUmkm->galeri_foto) {
            foreach ($produkUmkm->galeri_foto as $file) {
                Storage::delete('public/produk/galeri/' . $file);
            }
        }
        
        $produkUmkm->delete();
        
        return redirect()->route('admin.produk-umkm.index')
                        ->with('success', 'Produk UMKM berhasil dihapus.');
    }

    public function toggleStatus(ProdukUmkm $produkUmkm)
    {
        $newStatus = $produkUmkm->status === 'aktif' ? 'nonaktif' : 'aktif';
        $produkUmkm->update(['status' => $newStatus]);
        
        return response()->json([
            'success' => true,
            'message' => 'Status produk berhasil diubah.',
            'status' => $newStatus
        ]);
    }

    public function toggleFeatured(ProdukUmkm $produkUmkm)
    {
        $produkUmkm->update(['is_featured' => !$produkUmkm->is_featured]);
        
        return response()->json([
            'success' => true,
            'message' => 'Status featured berhasil diubah.',
            'is_featured' => $produkUmkm->is_featured
        ]);
    }

    public function updateStock(Request $request, ProdukUmkm $produkUmkm)
    {
        $validator = Validator::make($request->all(), [
            'stok' => 'required|integer|min:0',
            'operasi' => 'required|in:set,tambah,kurang'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid.',
                'errors' => $validator->errors()
            ], 422);
        }
        
        $stok = $request->stok;
        $operasi = $request->operasi;
        
        if ($operasi === 'set') {
            $produkUmkm->update(['stok' => $stok]);
        } elseif ($operasi === 'tambah') {
            $produkUmkm->updateStok($stok, 'tambah');
        } elseif ($operasi === 'kurang') {
            if (!$produkUmkm->updateStok($stok, 'kurang')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Stok tidak mencukupi.'
                ], 400);
            }
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Stok berhasil diperbarui.',
            'stok' => $produkUmkm->fresh()->stok
        ]);
    }

    public function bulkAction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'action' => 'required|in:delete,activate,deactivate,feature,unfeature',
            'ids' => 'required|array',
            'ids.*' => 'exists:produk_umkm,id'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid.',
                'errors' => $validator->errors()
            ], 422);
        }
        
        $produkList = ProdukUmkm::whereIn('id', $request->ids);
        $count = $produkList->count();
        
        switch ($request->action) {
            case 'delete':
                foreach ($produkList->get() as $produk) {
                    // Delete photos
                    if ($produk->foto_utama) {
                        Storage::delete('public/produk/' . $produk->foto_utama);
                    }
                    if ($produk->galeri_foto) {
                        foreach ($produk->galeri_foto as $file) {
                            Storage::delete('public/produk/galeri/' . $file);
                        }
                    }
                }
                $produkList->delete();
                $message = "{$count} produk berhasil dihapus.";
                break;
                
            case 'activate':
                $produkList->update(['status' => 'aktif']);
                $message = "{$count} produk berhasil diaktifkan.";
                break;
                
            case 'deactivate':
                $produkList->update(['status' => 'nonaktif']);
                $message = "{$count} produk berhasil dinonaktifkan.";
                break;
                
            case 'feature':
                $produkList->update(['is_featured' => true]);
                $message = "{$count} produk berhasil dijadikan featured.";
                break;
                
            case 'unfeature':
                $produkList->update(['is_featured' => false]);
                $message = "{$count} produk berhasil dihapus dari featured.";
                break;
        }
        
        return response()->json([
            'success' => true,
            'message' => $message
        ]);
    }

    public function export(Request $request)
    {
        // Implementation for exporting products data
        // This could be Excel, CSV, or PDF export
        
        return response()->json([
            'success' => true,
            'message' => 'Export akan segera dimulai.'
        ]);
    }

    private function getKategoriList()
    {
        return [
            'makanan_segar' => 'Makanan Segar',
            'makanan_olahan' => 'Makanan Olahan',
            'minuman' => 'Minuman',
            'kerajinan_kayu' => 'Kerajinan Kayu',
            'kerajinan_bambu' => 'Kerajinan Bambu',
            'kerajinan_tekstil' => 'Kerajinan Tekstil',
            'fashion_pakaian' => 'Fashion Pakaian',
            'fashion_aksesoris' => 'Fashion Aksesoris',
            'elektronik' => 'Elektronik',
            'peralatan_rumah' => 'Peralatan Rumah',
            'kosmetik_kecantikan' => 'Kosmetik & Kecantikan',
            'obat_herbal' => 'Obat Herbal',
            'tanaman_hias' => 'Tanaman Hias',
            'hasil_pertanian' => 'Hasil Pertanian',
            'hasil_peternakan' => 'Hasil Peternakan',
            'jasa_digital' => 'Jasa Digital',
            'jasa_fisik' => 'Jasa Fisik',
            'lainnya' => 'Lainnya'
        ];
    }
}
