<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SettingController extends Controller
{
    public function index()
    {
        $setting = Setting::getSettings();
        return view('admin.settings.index', compact('setting'));
    }

    public function update(Request $request)
    {
        $request->validate([
            'nama_desa' => 'required|string|max:255',
            'kecamatan' => 'required|string|max:255',
            'kabupaten' => 'required|string|max:255',
            'provinsi' => 'required|string|max:255',
            'alamat_kantor' => 'required|string',
            'kepala_desa' => 'required|string|max:255',
            'telepon' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            'luas_wilayah' => 'nullable|numeric|min:0',
            'jumlah_penduduk' => 'integer|min:0',
            'jumlah_kk' => 'integer|min:0',
            'jumlah_rt' => 'integer|min:0',
            'jumlah_rw' => 'integer|min:0',
            'jumlah_dusun' => 'integer|min:0',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'tripay_api_key' => 'nullable|string|max:255',
            'tripay_private_key' => 'nullable|string|max:255',
            'tripay_merchant_code' => 'nullable|string|max:255',
            'tripay_mode' => 'nullable|in:sandbox,production',
            'tripay_expiry_time' => 'nullable|integer|min:1|max:168',
            'tripay_fee_customer' => 'nullable|boolean',
            'tripay_channels' => 'nullable|array',
            'ottopay_api_key' => 'nullable|string|max:255',
            'ottopay_secret_key' => 'nullable|string|max:255',
            'ottopay_merchant_code' => 'nullable|string|max:255',
            'ottopay_mode' => 'nullable|in:sandbox,production',
            'ottopay_expiry_time' => 'nullable|integer|min:1|max:168',
            'ottopay_fee_customer' => 'nullable|boolean',
            'ottopay_channels' => 'nullable|array'
        ]);

        $data = $request->except(['logo', '_token', '_method']);

        // Handle logo upload
        if ($request->hasFile('logo')) {
            $setting = Setting::first();
            
            // Delete old logo if exists
            if ($setting && $setting->logo) {
                Storage::disk('public')->delete($setting->logo);
            }
            
            $logoPath = $request->file('logo')->store('settings', 'public');
            $data['logo'] = $logoPath;
        }

        // Handle jam_operasional
        if ($request->has('jam_operasional')) {
            $jamOperasional = [];
            $hari = ['senin', 'selasa', 'rabu', 'kamis', 'jumat', 'sabtu', 'minggu'];
            
            foreach ($hari as $day) {
                if ($request->has("jam_{$day}_mulai") && $request->has("jam_{$day}_selesai")) {
                    $jamOperasional[$day] = [
                        'mulai' => $request->input("jam_{$day}_mulai"),
                        'selesai' => $request->input("jam_{$day}_selesai"),
                        'tutup' => $request->has("tutup_{$day}")
                    ];
                }
            }
            $data['jam_operasional'] = $jamOperasional;
        }

        // Handle media_sosial
        if ($request->has('media_sosial')) {
            $mediaSosial = [];
            $platforms = ['facebook', 'instagram', 'twitter', 'youtube', 'whatsapp'];
            
            foreach ($platforms as $platform) {
                if ($request->filled("media_{$platform}")) {
                    $mediaSosial[$platform] = $request->input("media_{$platform}");
                }
            }
            $data['media_sosial'] = $mediaSosial;
        }

        // Handle tripay_channels
        if ($request->has('tripay_channels')) {
            $data['tripay_channels'] = $request->input('tripay_channels', []);
        }

        // Handle tripay_fee_customer
        if ($request->has('tripay_fee_customer')) {
            $data['tripay_fee_customer'] = $request->boolean('tripay_fee_customer');
        }

        // Handle ottopay_channels
        if ($request->has('ottopay_channels')) {
            $data['ottopay_channels'] = $request->input('ottopay_channels', []);
        }

        // Handle ottopay_fee_customer
        if ($request->has('ottopay_fee_customer')) {
            $data['ottopay_fee_customer'] = $request->boolean('ottopay_fee_customer');
        }

        $setting = Setting::updateSettings($data);

        return response()->json([
            'success' => true,
            'message' => 'Pengaturan berhasil disimpan',
            'data' => $setting
        ]);
    }

    public function uploadLogo(Request $request)
    {
        $request->validate([
            'logo' => 'required|image|mimes:jpeg,png,jpg|max:2048'
        ]);

        $setting = Setting::first();
        
        // Delete old logo if exists
        if ($setting && $setting->logo) {
            Storage::disk('public')->delete($setting->logo);
        }

        $logoPath = $request->file('logo')->store('settings', 'public');
        
        if ($setting) {
            $setting->update(['logo' => $logoPath]);
        } else {
            Setting::create(['logo' => $logoPath]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Logo berhasil diupload',
            'logo_url' => Storage::url($logoPath)
        ]);
    }

    public function deleteLogo()
    {
        $setting = Setting::first();
        
        if ($setting && $setting->logo) {
            Storage::disk('public')->delete($setting->logo);
            $setting->update(['logo' => null]);
            
            return response()->json([
                'success' => true,
                'message' => 'Logo berhasil dihapus'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => 'Logo tidak ditemukan'
        ]);
    }

    public function testTripay(Request $request)
    {
        $request->validate([
            'api_key' => 'required|string',
            'private_key' => 'required|string',
            'merchant_code' => 'required|string',
            'mode' => 'required|in:sandbox,production'
        ]);

        try {
            $baseUrl = $request->mode === 'production' 
                ? 'https://tripay.co.id/api' 
                : 'https://tripay.co.id/api-sandbox';

            $response = \Illuminate\Support\Facades\Http::withHeaders([
                'Authorization' => 'Bearer ' . $request->api_key,
            ])->get($baseUrl . '/merchant/payment-channel');

            if ($response->successful()) {
                $data = $response->json();
                $channelCount = count($data['data'] ?? []);
                
                return response()->json([
                    'success' => true,
                    'message' => "Berhasil terhubung dengan {$channelCount} channel pembayaran tersedia"
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal terhubung: ' . ($response->json()['message'] ?? 'Invalid credentials')
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    public function testOttopay(Request $request)
    {
        $request->validate([
            'api_key' => 'required|string',
            'secret_key' => 'required|string',
            'merchant_code' => 'required|string',
            'mode' => 'required|in:sandbox,production'
        ]);

        try {
            $baseUrl = $request->mode === 'production' 
                ? 'https://api.ottopay.id' 
                : 'https://sandbox-api.ottopay.id';

            // Test dengan endpoint merchant info
            $response = \Illuminate\Support\Facades\Http::withHeaders([
                'Authorization' => 'Bearer ' . $request->api_key,
                'Content-Type' => 'application/json',
                'X-Merchant-Code' => $request->merchant_code
            ])->get($baseUrl . '/v1/merchant/info');

            if ($response->successful()) {
                $data = $response->json();
                
                return response()->json([
                    'success' => true,
                    'message' => 'Koneksi berhasil! Merchant: ' . ($data['data']['name'] ?? 'Unknown')
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Kredensial tidak valid atau merchant tidak ditemukan'
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal terhubung ke OttoPay: ' . $e->getMessage()
            ]);
        }
    }
} 
