<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Umkm;
use App\Models\Penduduk;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class UmkmController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Umkm::with(['pemilik', 'user']);
        
        // Filter berdasarkan kategori
        if ($request->filled('kategori')) {
            $query->where('kategori', $request->kategori);
        }
        
        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        // Filter berdasarkan verifikasi
        if ($request->filled('verified')) {
            $query->where('is_verified', $request->verified == '1');
        }
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_umkm', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhereHas('pemilik', function($q) use ($search) {
                      $q->where('nama', 'like', "%{$search}%");
                  });
            });
        }
        
        $umkm = $query->latest()->paginate(10);
        
        // Data untuk filter
        $kategoris = [
            'makanan_minuman' => 'Makanan & Minuman',
            'kerajinan_tangan' => 'Kerajinan Tangan',
            'fashion_tekstil' => 'Fashion & Tekstil',
            'jasa_layanan' => 'Jasa & Layanan',
            'pertanian_perkebunan' => 'Pertanian & Perkebunan',
            'peternakan_perikanan' => 'Peternakan & Perikanan',
            'teknologi_digital' => 'Teknologi Digital',
            'perdagangan_retail' => 'Perdagangan & Retail',
            'pariwisata_budaya' => 'Pariwisata & Budaya',
            'lainnya' => 'Lainnya'
        ];
        
        $penduduk = Penduduk::orderBy('nama')->get();
        
        return view('admin.umkm.index', compact('umkm', 'kategoris', 'penduduk'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $kategoris = [
            'makanan_minuman' => 'Makanan & Minuman',
            'kerajinan_tangan' => 'Kerajinan Tangan',
            'fashion_tekstil' => 'Fashion & Tekstil',
            'jasa_layanan' => 'Jasa & Layanan',
            'pertanian_perkebunan' => 'Pertanian & Perkebunan',
            'peternakan_perikanan' => 'Peternakan & Perikanan',
            'teknologi_digital' => 'Teknologi Digital',
            'perdagangan_retail' => 'Perdagangan & Retail',
            'pariwisata_budaya' => 'Pariwisata & Budaya',
            'lainnya' => 'Lainnya'
        ];
        
        $penduduk = Penduduk::orderBy('nama')->get();
        
        return view('admin.umkm.create', compact('kategoris', 'penduduk'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'nama_umkm' => 'required|string|max:255',
            'kategori' => 'required|in:makanan_minuman,kerajinan_tangan,fashion_tekstil,jasa_layanan,pertanian_perkebunan,peternakan_perikanan,teknologi_digital,perdagangan_retail,pariwisata_budaya,lainnya',
            'pemilik_id' => 'required|exists:penduduk,id',
            'deskripsi' => 'required|string',
            'alamat' => 'required|string',
            'nomor_telepon' => 'nullable|string|max:15',
            'email' => 'nullable|email',
            'website' => 'nullable|url',
            'skala_usaha' => 'required|in:mikro,kecil,menengah',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'galeri_foto.*' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $data = $request->all();
        $data['user_id'] = Auth::id();
        $data['slug'] = Str::slug($request->nama_umkm);
        
        // Handle logo upload
        if ($request->hasFile('logo')) {
            $logo = $request->file('logo');
            $logoName = time() . '_logo_' . $logo->getClientOriginalName();
            $logo->storeAs('public/umkm/logo', $logoName);
            $data['logo'] = $logoName;
        }
        
        // Handle galeri foto upload
        if ($request->hasFile('galeri_foto')) {
            $galeriFiles = [];
            foreach ($request->file('galeri_foto') as $file) {
                $fileName = time() . '_galeri_' . $file->getClientOriginalName();
                $file->storeAs('public/umkm/galeri', $fileName);
                $galeriFiles[] = $fileName;
            }
            $data['galeri_foto'] = $galeriFiles;
        }
        
        // Handle jam operasional
        if ($request->filled('jam_operasional')) {
            $data['jam_operasional'] = $request->jam_operasional;
        }
        
        // Handle hari libur
        if ($request->filled('hari_libur')) {
            $data['hari_libur'] = $request->hari_libur;
        }
        
        // Handle tags
        if ($request->filled('tags')) {
            $data['tags'] = explode(',', $request->tags);
        }

        Umkm::create($data);

        return redirect()->route('admin.umkm.index')
            ->with('success', 'Data UMKM berhasil ditambahkan');
    }

    /**
     * Display the specified resource.
     */
    public function show(Umkm $umkm)
    {
        $umkm->load(['pemilik', 'produk', 'user']);
        $umkm->incrementViewCount();
        
        return view('admin.umkm.show', compact('umkm'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Umkm $umkm)
    {
        $kategoris = [
            'makanan_minuman' => 'Makanan & Minuman',
            'kerajinan_tangan' => 'Kerajinan Tangan',
            'fashion_tekstil' => 'Fashion & Tekstil',
            'jasa_layanan' => 'Jasa & Layanan',
            'pertanian_perkebunan' => 'Pertanian & Perkebunan',
            'peternakan_perikanan' => 'Peternakan & Perikanan',
            'teknologi_digital' => 'Teknologi Digital',
            'perdagangan_retail' => 'Perdagangan & Retail',
            'pariwisata_budaya' => 'Pariwisata & Budaya',
            'lainnya' => 'Lainnya'
        ];
        
        $penduduk = Penduduk::orderBy('nama')->get();
        
        return view('admin.umkm.edit', compact('umkm', 'kategoris', 'penduduk'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Umkm $umkm)
    {
        $request->validate([
            'nama_umkm' => 'required|string|max:255',
            'kategori' => 'required|in:makanan_minuman,kerajinan_tangan,fashion_tekstil,jasa_layanan,pertanian_perkebunan,peternakan_perikanan,teknologi_digital,perdagangan_retail,pariwisata_budaya,lainnya',
            'pemilik_id' => 'required|exists:penduduk,id',
            'deskripsi' => 'required|string',
            'alamat' => 'required|string',
            'nomor_telepon' => 'nullable|string|max:15',
            'email' => 'nullable|email',
            'website' => 'nullable|url',
            'skala_usaha' => 'required|in:mikro,kecil,menengah',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'galeri_foto.*' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $data = $request->all();
        
        // Handle logo upload
        if ($request->hasFile('logo')) {
            // Delete old logo
            if ($umkm->logo) {
                Storage::delete('public/umkm/logo/' . $umkm->logo);
            }
            
            $logo = $request->file('logo');
            $logoName = time() . '_logo_' . $logo->getClientOriginalName();
            $logo->storeAs('public/umkm/logo', $logoName);
            $data['logo'] = $logoName;
        }
        
        // Handle galeri foto upload
        if ($request->hasFile('galeri_foto')) {
            // Delete old galeri
            if ($umkm->galeri_foto) {
                foreach ($umkm->galeri_foto as $oldFile) {
                    Storage::delete('public/umkm/galeri/' . $oldFile);
                }
            }
            
            $galeriFiles = [];
            foreach ($request->file('galeri_foto') as $file) {
                $fileName = time() . '_galeri_' . $file->getClientOriginalName();
                $file->storeAs('public/umkm/galeri', $fileName);
                $galeriFiles[] = $fileName;
            }
            $data['galeri_foto'] = $galeriFiles;
        }
        
        // Handle jam operasional
        if ($request->filled('jam_operasional')) {
            $data['jam_operasional'] = $request->jam_operasional;
        }
        
        // Handle hari libur
        if ($request->filled('hari_libur')) {
            $data['hari_libur'] = $request->hari_libur;
        }
        
        // Handle tags
        if ($request->filled('tags')) {
            $data['tags'] = explode(',', $request->tags);
        }

        $umkm->update($data);

        return redirect()->route('admin.umkm.index')
            ->with('success', 'Data UMKM berhasil diperbarui');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Umkm $umkm)
    {
        // Delete files
        if ($umkm->logo) {
            Storage::delete('public/umkm/logo/' . $umkm->logo);
        }
        
        if ($umkm->galeri_foto) {
            foreach ($umkm->galeri_foto as $file) {
                Storage::delete('public/umkm/galeri/' . $file);
            }
        }
        
        $umkm->delete();

        return redirect()->route('admin.umkm.index')
            ->with('success', 'Data UMKM berhasil dihapus');
    }

    public function verify(Request $request, Umkm $umkm)
    {
        $request->validate([
            'status' => 'required|in:aktif,nonaktif',
            'catatan_verifikasi' => 'nullable|string'
        ]);

        $umkm->update([
            'status' => $request->status,
            'is_verified' => $request->status === 'aktif',
            'verified_at' => now(),
            'verified_by' => Auth::id(),
            'catatan_verifikasi' => $request->catatan_verifikasi
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Status UMKM berhasil diperbarui'
        ]);
    }

    public function toggleMarketplace(Umkm $umkm)
    {
        $umkm->update([
            'aktif_marketplace' => !$umkm->aktif_marketplace
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Status marketplace berhasil diperbarui',
            'aktif_marketplace' => $umkm->aktif_marketplace
        ]);
    }

    public function statistik()
    {
        $totalUmkm = Umkm::count();
        $umkmAktif = Umkm::where('status', 'aktif')->count();
        $umkmVerified = Umkm::where('is_verified', true)->count();
        $umkmMarketplace = Umkm::where('aktif_marketplace', true)->count();
        
        $umkmByKategori = Umkm::selectRaw('kategori, COUNT(*) as total')
            ->groupBy('kategori')
            ->get();
            
        $umkmBySkala = Umkm::selectRaw('skala_usaha, COUNT(*) as total')
            ->groupBy('skala_usaha')
            ->get();

        return view('admin.umkm.statistik', compact(
            'totalUmkm', 'umkmAktif', 'umkmVerified', 'umkmMarketplace',
            'umkmByKategori', 'umkmBySkala'
        ));
    }

    public function laporan(Request $request)
    {
        $query = Umkm::with(['pemilik', 'produk']);
        
        if ($request->filled('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        
        if ($request->filled('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
        
        if ($request->filled('kategori')) {
            $query->where('kategori', $request->kategori);
        }
        
        $umkm = $query->get();
        
        return view('admin.umkm.laporan', compact('umkm'));
    }
}
