<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Penduduk;
use App\Models\PelayananSurat;
use App\Models\Pengaduan;
use App\Models\Berita;
use App\Models\AgendaDesa;
use App\Models\Transaksi;
use App\Models\TiketWisata;
use App\Models\AnggotaKeluarga;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class PortalWargaController extends Controller
{
    /**
     * Portal dashboard web view (for browser access)
     */
    public function dashboardWeb(Request $request)
    {
        // Check if user is "logged in" via localStorage token
        // For web access, we'll use session-based authentication
        if (!$request->session()->has('portal_user') || !$request->session()->has('portal_nik')) {
            return redirect('/')->with('error', 'Silakan login terlebih dahulu');
        }

        $userData = $request->session()->get('portal_user');
        $sessionNik = $request->session()->get('portal_nik');
        
        // CRITICAL FIX: Validate NIK consistency
        if (!isset($userData['nik']) || $userData['nik'] !== $sessionNik) {
            $request->session()->flush();
            return redirect('/')->with('error', 'Session tidak valid, silakan login ulang');
        }

        $penduduk = Penduduk::where('nik', $sessionNik)->first();

        if (!$penduduk) {
            $request->session()->flush();
            return redirect('/')->with('error', 'Data penduduk tidak ditemukan');
        }
        
        // Additional security: Verify the penduduk NIK matches session
        if ($penduduk->nik !== $sessionNik) {
            $request->session()->flush();
            return redirect('/')->with('error', 'Akses tidak sah, silakan login ulang');
        }

        try {
            // Get dashboard data - ALWAYS use $penduduk->nik, not session data
            $layananTerbaru = PelayananSurat::where('nik', $penduduk->nik)
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get();

            $pengaduanTerbaru = Pengaduan::where('nik', $penduduk->nik)
                ->orderBy('created_at', 'desc')
                ->limit(3)
                ->get();

            $beritaTerbaru = Berita::published()
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get();

            $agendaMendatang = AgendaDesa::where('tanggal', '>=', now())
                ->orderBy('tanggal', 'asc')
                ->limit(5)
                ->get();

            $statistik = [
                'surat_diproses' => PelayananSurat::where('nik', $penduduk->nik)
                    ->whereIn('status', ['Menunggu', 'Diproses'])
                    ->count(),
                'surat_selesai' => PelayananSurat::where('nik', $penduduk->nik)
                    ->where('status', 'Selesai')
                    ->count(),
                'pengaduan_aktif' => Pengaduan::where('nik', $penduduk->nik)
                    ->whereIn('status', ['Menunggu', 'Diproses'])
                    ->count(),
                'total_layanan' => PelayananSurat::where('nik', $penduduk->nik)->count()
            ];

            return view('portal.dashboard', compact(
                'penduduk',
                'layananTerbaru',
                'pengaduanTerbaru', 
                'beritaTerbaru',
                'agendaMendatang',
                'statistik'
            ));

        } catch (\Exception $e) {
            Log::error('Portal Dashboard Web Error', [
                'nik' => $penduduk->nik,
                'error' => $e->getMessage()
            ]);

            return redirect('/')->with('error', 'Terjadi kesalahan saat memuat dashboard');
        }
    }

    /**
     * Set session for web portal access
     */
    public function setSession(Request $request)
    {
        try {
            $user = $request->input('user');
            $token = $request->input('token');
            
            if ($user && $token && isset($user['nik'])) {
                // Start session manually if not started
                if (!$request->hasSession()) {
                    // Try to start session manually
                    $request->setLaravelSession(app('session.store'));
                }
                
                // CRITICAL FIX: Clear existing session first to prevent NIK mixing
                $request->session()->flush();
                
                // Regenerate session ID to ensure complete separation
                $request->session()->regenerate(true);
                
                // Set new session data with NIK validation
                $request->session()->put('portal_user', $user);
                $request->session()->put('portal_token', $token);
                $request->session()->put('portal_nik', $user['nik']); // Explicit NIK tracking
                
                // Save session
                $request->session()->save();
                
                // Log for debugging
                Log::info('Portal Session Set', [
                    'nik' => $user['nik'],
                    'session_id' => $request->session()->getId(),
                    'timestamp' => now()
                ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Session berhasil diset',
                    'debug' => [
                        'nik' => $user['nik'],
                        'session_id' => $request->session()->getId()
                    ]
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid'
            ], 400);
            
        } catch (\Exception $e) {
            Log::error('Set Session Error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Logout from web portal
     */
    public function logoutWeb(Request $request)
    {
        try {
            $sessionNik = $request->session()->get('portal_nik');
            
            // Log logout for debugging
            Log::info('Portal Logout', [
                'nik' => $sessionNik,
                'session_id' => $request->session()->getId(),
                'timestamp' => now()
            ]);
            
            // Complete session cleanup
            $request->session()->flush();
            $request->session()->regenerate(true);
            
            return response()->json([
                'success' => true,
                'message' => 'Logout berhasil'
            ]);
            
        } catch (\Exception $e) {
            Log::error('Logout Error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat logout'
            ], 500);
        }
    }

    /**
     * Get warga dashboard data
     */
    public function dashboard(Request $request)
    {
        try {
            $user = $request->user();
            $penduduk = Penduduk::where('nik', $user->nik)->first();

            if (!$penduduk) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data penduduk tidak ditemukan'
                ], 404);
            }

            // Get latest services
            $layananTerbaru = PelayananSurat::where('nik', $penduduk->nik)
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get(['id', 'jenis_surat', 'status', 'tanggal_pengajuan', 'nomor_antrian']);

            // Get latest complaints
            $pengaduanTerbaru = Pengaduan::where('nik', $penduduk->nik)
                ->orderBy('created_at', 'desc')
                ->limit(3)
                ->get(['id', 'judul', 'status', 'tanggal_pengaduan']);

            // Get latest news
            $beritaTerbaru = Berita::published()
                ->orderBy('created_at', 'desc')
                ->limit(5)
                ->get(['id', 'judul', 'slug', 'excerpt', 'created_at', 'gambar']);

            // Get upcoming agenda
            $agendaMendatang = AgendaDesa::where('tanggal', '>=', now())
                ->orderBy('tanggal', 'asc')
                ->limit(5)
                ->get(['id', 'nama_kegiatan', 'tanggal', 'waktu', 'tempat']);

            // Get statistics
            $statistik = [
                'surat_diproses' => PelayananSurat::where('nik', $penduduk->nik)
                    ->whereIn('status', ['Menunggu', 'Diproses'])
                    ->count(),
                'surat_selesai' => PelayananSurat::where('nik', $penduduk->nik)
                    ->where('status', 'Selesai')
                    ->count(),
                'pengaduan_aktif' => Pengaduan::where('nik', $penduduk->nik)
                    ->whereIn('status', ['Menunggu', 'Diproses'])
                    ->count(),
                'transaksi_pending' => Transaksi::where('nik_pembeli', $penduduk->nik)
                    ->where('status', 'UNPAID')
                    ->count()
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'penduduk' => [
                        'nik' => $penduduk->nik,
                        'nama' => $penduduk->nama,
                        'alamat' => $penduduk->alamat,
                        'dusun' => $penduduk->dusun,
                        'rt' => $penduduk->rt,
                        'rw' => $penduduk->rw,
                        'foto' => $penduduk->foto
                    ],
                    'layanan_terbaru' => $layananTerbaru,
                    'pengaduan_terbaru' => $pengaduanTerbaru,
                    'berita_terbaru' => $beritaTerbaru,
                    'agenda_mendatang' => $agendaMendatang,
                    'statistik' => $statistik
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Warga Dashboard Error', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat memuat dashboard'
            ], 500);
        }
    }

    /**
     * Get layanan surat for warga
     */
    public function layananSurat(Request $request)
    {
        try {
            $user = $request->user();
            $penduduk = Penduduk::where('nik', $user->nik)->first();

            if (!$penduduk) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data penduduk tidak ditemukan'
                ], 404);
            }

            $query = PelayananSurat::where('nik', $penduduk->nik);

            // Filter by status
            if ($request->status) {
                $query->where('status', $request->status);
            }

            // Filter by date range
            if ($request->dari_tanggal && $request->sampai_tanggal) {
                $query->whereBetween('tanggal_pengajuan', [
                    $request->dari_tanggal, 
                    $request->sampai_tanggal
                ]);
            }

            $layananSurat = $query->orderBy('tanggal_pengajuan', 'desc')
                ->paginate($request->get('per_page', 10));

            return response()->json([
                'success' => true,
                'data' => $layananSurat
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Warga Layanan Surat Error', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat memuat data surat'
            ], 500);
        }
    }

    /**
     * Submit pengajuan surat
     */
    public function ajukanSurat(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'jenis_surat' => 'required|string',
            'keperluan' => 'required|string',
            'keterangan' => 'nullable|string',
            'berkas_pendukung' => 'nullable|array',
            'berkas_pendukung.*' => 'file|mimes:pdf,jpg,jpeg,png|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = $request->user();
            $penduduk = Penduduk::where('nik', $user->nik)->first();

            if (!$penduduk) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data penduduk tidak ditemukan'
                ], 404);
            }

            // Handle file uploads
            $berkasPendukung = [];
            if ($request->hasFile('berkas_pendukung')) {
                foreach ($request->file('berkas_pendukung') as $file) {
                    $filename = time() . '_' . $file->getClientOriginalName();
                    $path = $file->storeAs('berkas-surat', $filename, 'public');
                    $berkasPendukung[] = $path;
                }
            }

            // Generate nomor antrian
            $nomorAntrian = 'ANT-' . date('Ymd') . '-' . str_pad(
                PelayananSurat::whereDate('created_at', today())->count() + 1, 
                3, '0', STR_PAD_LEFT
            );

            $surat = PelayananSurat::create([
                'nik' => $penduduk->nik,
                'jenis_surat' => $request->jenis_surat,
                'keperluan' => $request->keperluan,
                'keterangan' => $request->keterangan,
                'nomor_antrian' => $nomorAntrian,
                'status' => 'Menunggu',
                'tanggal_pengajuan' => now(),
                'berkas_pendukung' => json_encode($berkasPendukung)
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Pengajuan surat berhasil dikirim',
                'data' => [
                    'id' => $surat->id,
                    'nomor_antrian' => $surat->nomor_antrian,
                    'jenis_surat' => $surat->jenis_surat,
                    'status' => $surat->status
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Warga Ajukan Surat Error', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat mengajukan surat'
            ], 500);
        }
    }

    /**
     * Track status surat
     */
    public function trackSurat(Request $request, $nomorAntrian)
    {
        try {
            $user = $request->user();
            $penduduk = Penduduk::where('nik', $user->nik)->first();

            $surat = PelayananSurat::where('nomor_antrian', $nomorAntrian)
                ->where('nik', $penduduk->nik)
                ->first();

            if (!$surat) {
                return response()->json([
                    'success' => false,
                    'message' => 'Surat tidak ditemukan'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'nomor_antrian' => $surat->nomor_antrian,
                    'jenis_surat' => $surat->jenis_surat,
                    'status' => $surat->status,
                    'tanggal_pengajuan' => $surat->tanggal_pengajuan,
                    'tanggal_selesai' => $surat->tanggal_selesai,
                    'keterangan' => $surat->keterangan,
                    'file_pdf' => $surat->file_pdf,
                    'timeline' => [
                        'diajukan' => $surat->tanggal_pengajuan,
                        'diproses' => $surat->status === 'Diproses' ? now() : null,
                        'selesai' => $surat->tanggal_selesai
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Warga Track Surat Error', [
                'user_id' => $request->user()->id,
                'nomor_antrian' => $nomorAntrian,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat melacak surat'
            ], 500);
        }
    }

    /**
     * Get pengaduan for warga
     */
    public function pengaduan(Request $request)
    {
        try {
            $user = $request->user();
            $penduduk = Penduduk::where('nik', $user->nik)->first();

            $query = Pengaduan::where('nik', $penduduk->nik);

            // Filter by status
            if ($request->status) {
                $query->where('status', $request->status);
            }

            $pengaduan = $query->orderBy('tanggal_pengaduan', 'desc')
                ->paginate($request->get('per_page', 10));

            return response()->json([
                'success' => true,
                'data' => $pengaduan
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Warga Pengaduan Error', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat memuat pengaduan'
            ], 500);
        }
    }

    /**
     * Submit pengaduan
     */
    public function submitPengaduan(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'judul' => 'required|string|max:255',
            'isi_pengaduan' => 'required|string',
            'kategori' => 'required|string',
            'lokasi' => 'nullable|string',
            'foto' => 'nullable|array',
            'foto.*' => 'image|mimes:jpg,jpeg,png|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = $request->user();
            $penduduk = Penduduk::where('nik', $user->nik)->first();

            // Handle photo uploads
            $fotos = [];
            if ($request->hasFile('foto')) {
                foreach ($request->file('foto') as $foto) {
                    $filename = time() . '_' . $foto->getClientOriginalName();
                    $path = $foto->storeAs('pengaduan', $filename, 'public');
                    $fotos[] = $path;
                }
            }

            $pengaduan = Pengaduan::create([
                'nik' => $penduduk->nik,
                'judul' => $request->judul,
                'isi_pengaduan' => $request->isi_pengaduan,
                'kategori' => $request->kategori,
                'lokasi' => $request->lokasi,
                'foto' => json_encode($fotos),
                'status' => 'Menunggu',
                'tanggal_pengaduan' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Pengaduan berhasil dikirim',
                'data' => [
                    'id' => $pengaduan->id,
                    'judul' => $pengaduan->judul,
                    'status' => $pengaduan->status
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Warga Submit Pengaduan Error', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat mengirim pengaduan'
            ], 500);
        }
    }

    /**
     * Get profile warga
     */
    public function profile(Request $request)
    {
        try {
            $user = $request->user();
            $penduduk = Penduduk::where('nik', $user->nik)->first();

            if (!$penduduk) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data penduduk tidak ditemukan'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'nik' => $penduduk->nik,
                    'nama' => $penduduk->nama,
                    'alamat' => $penduduk->alamat,
                    'dusun' => $penduduk->dusun,
                    'rt' => $penduduk->rt,
                    'rw' => $penduduk->rw,
                    'tempat_lahir' => $penduduk->tempat_lahir,
                    'tanggal_lahir' => $penduduk->tanggal_lahir,
                    'jenis_kelamin' => $penduduk->jenis_kelamin,
                    'agama' => $penduduk->agama,
                    'status_kawin' => $penduduk->status_kawin,
                    'pekerjaan' => $penduduk->pekerjaan,
                    'pendidikan_terakhir' => $penduduk->pendidikan_terakhir,
                    'no_wa' => $penduduk->no_wa,
                    'foto' => $penduduk->foto
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Warga Profile Error', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat memuat profil'
            ], 500);
        }
    }

    /**
     * Update profile warga
     */
    public function updateProfile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'no_wa' => 'nullable|string|max:15',
            'alamat' => 'nullable|string',
            'foto' => 'nullable|image|mimes:jpg,jpeg,png|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = $request->user();
            $penduduk = Penduduk::where('nik', $user->nik)->first();

            if (!$penduduk) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data penduduk tidak ditemukan'
                ], 404);
            }

            $updateData = [];

            if ($request->no_wa) {
                $updateData['no_wa'] = $request->no_wa;
            }

            if ($request->alamat) {
                $updateData['alamat'] = $request->alamat;
            }

            // Handle photo upload
            if ($request->hasFile('foto')) {
                $foto = $request->file('foto');
                $filename = time() . '_' . $foto->getClientOriginalName();
                $path = $foto->storeAs('penduduk', $filename, 'public');
                $updateData['foto'] = $path;
            }

            $penduduk->update($updateData);

            return response()->json([
                'success' => true,
                'message' => 'Profil berhasil diperbarui',
                'data' => $penduduk->fresh()
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Warga Update Profile Error', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat memperbarui profil'
            ], 500);
        }
    }

    /**
     * Get notifications for warga
     */
    public function notifications(Request $request)
    {
        try {
            $user = $request->user();
            $penduduk = Penduduk::where('nik', $user->nik)->first();

            // Get notifications from various sources
            $notifications = collect();

            // Surat notifications
            $suratNotifications = PelayananSurat::where('nik', $penduduk->nik)
                ->where('status', '!=', 'Menunggu')
                ->orderBy('updated_at', 'desc')
                ->limit(10)
                ->get()
                ->map(function ($surat) {
                    return [
                        'id' => 'surat_' . $surat->id,
                        'type' => 'surat',
                        'title' => 'Status Surat Diperbarui',
                        'message' => "Surat {$surat->jenis_surat} dengan nomor antrian {$surat->nomor_antrian} sekarang berstatus: {$surat->status}",
                        'date' => $surat->updated_at,
                        'read' => false,
                        'data' => [
                            'surat_id' => $surat->id,
                            'nomor_antrian' => $surat->nomor_antrian,
                            'status' => $surat->status
                        ]
                    ];
                });

            $notifications = $notifications->merge($suratNotifications);

            // Sort by date
            $notifications = $notifications->sortByDesc('date')->values();

            return response()->json([
                'success' => true,
                'data' => $notifications
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Warga Notifications Error', [
                'user_id' => $request->user()->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat memuat notifikasi'
            ], 500);
        }
    }

    /**
     * Login for portal warga
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nik' => 'required|string|size:16',
            'password' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $nik = trim($request->nik);
            $password = trim($request->password);
            
            // Log login attempt (without password)
            Log::info('Portal Login Attempt', [
                'nik' => $nik,
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'timestamp' => now()
            ]);
            
            $penduduk = Penduduk::where('nik', $nik)->first();

            if (!$penduduk) {
                Log::warning('Portal Login Failed - NIK Not Found', [
                    'nik' => $nik,
                    'ip' => $request->ip()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'NIK tidak terdaftar dalam sistem'
                ], 401);
            }

            // Check if penduduk has password
            if (!$penduduk->hasPassword()) {
                Log::warning('Portal Login Failed - No Password Set', [
                    'nik' => $nik,
                    'nama' => $penduduk->nama
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Password belum diatur. Silakan hubungi perangkat desa untuk mengatur password.'
                ], 401);
            }

            // Check password
            if (!$penduduk->checkPassword($password)) {
                Log::warning('Portal Login Failed - Wrong Password', [
                    'nik' => $nik,
                    'nama' => $penduduk->nama,
                    'ip' => $request->ip()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'NIK atau password salah'
                ], 401);
            }

            // Update last login
            $penduduk->update([
                'last_login_at' => now(),
                'login_count' => ($penduduk->login_count ?? 0) + 1
            ]);

            // Generate unique token with NIK
            $token = 'portal_' . $penduduk->nik . '_' . time() . '_' . Str::random(32);
            
            $userData = [
                'nik' => $penduduk->nik,
                'nama' => $penduduk->nama,
                'alamat' => $penduduk->alamat,
                'foto' => $penduduk->foto,
                'login_time' => now()->toISOString()
            ];

            Log::info('Portal Login Success', [
                'nik' => $penduduk->nik,
                'nama' => $penduduk->nama,
                'token' => substr($token, 0, 20) . '...',
                'ip' => $request->ip()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Login berhasil',
                'token' => $token,
                'user' => $userData,
                'must_change_password' => $penduduk->mustChangePassword()
            ]);

        } catch (\Exception $e) {
            Log::error('Portal Login Error', [
                'nik' => $request->nik,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan sistem'
            ], 500);
        }
    }

    public function getUserData(Request $request)
    {
        try {
            $nik = $request->input('nik');
            
            if (!$nik) {
                return response()->json([
                    'success' => false,
                    'message' => 'NIK is required'
                ], 400);
            }
            
            // Get user data from penduduk table
            $penduduk = Penduduk::where('nik', $nik)->first();
            
            if (!$penduduk) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data penduduk tidak ditemukan'
                ], 404);
            }
            
            return response()->json([
                'success' => true,
                'data' => [
                    'nama' => $penduduk->nama,
                    'nik' => $penduduk->nik,
                    'tempat_lahir' => $penduduk->tempat_lahir,
                    'tanggal_lahir' => $penduduk->tanggal_lahir ? $penduduk->tanggal_lahir->format('d F Y') : null,
                    'jenis_kelamin' => $penduduk->jenis_kelamin,
                    'agama' => $penduduk->agama,
                    'status_perkawinan' => $penduduk->status_perkawinan,
                    'pekerjaan' => $penduduk->pekerjaan,
                    'pendidikan' => $penduduk->pendidikan,
                    'alamat' => $penduduk->alamat,
                    'rt' => $penduduk->rt,
                    'rw' => $penduduk->rw,
                    'dusun' => $penduduk->dusun,
                    'kelurahan' => $penduduk->kelurahan,
                    'kecamatan' => $penduduk->kecamatan,
                    'kabupaten' => $penduduk->kabupaten,
                    'provinsi' => $penduduk->provinsi,
                    'no_hp' => $penduduk->no_hp,
                    'email' => $penduduk->email ?? null,
                    'foto' => $penduduk->foto ?? null
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }
    
    public function getFamilyData(Request $request)
    {
        try {
            $nik = $request->input('nik');
            
            if (!$nik) {
                return response()->json([
                    'success' => false,
                    'message' => 'NIK is required'
                ], 400);
            }
            
            // Get user's no_kk
            $penduduk = Penduduk::where('nik', $nik)->first();
            
            if (!$penduduk || !$penduduk->no_kk) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data keluarga tidak ditemukan'
                ], 404);
            }
            
            // Get all family members with the same no_kk
            $anggotaKeluarga = Penduduk::where('no_kk', $penduduk->no_kk)->get();
            
            $familyData = [];
            foreach ($anggotaKeluarga as $anggota) {
                // Determine relationship based on position or age (simplified logic)
                $hubungan = $this->determineRelationship($anggota, $anggotaKeluarga, $penduduk);
                
                $familyData[] = [
                    'nama' => $anggota->nama,
                    'nik' => $anggota->nik,
                    'hubungan' => $hubungan,
                    'tempat_lahir' => $anggota->tempat_lahir,
                    'tanggal_lahir' => $anggota->tanggal_lahir ? $anggota->tanggal_lahir->format('d F Y') : null,
                    'jenis_kelamin' => $anggota->jenis_kelamin,
                    'agama' => $anggota->agama,
                    'pendidikan' => $anggota->pendidikan,
                    'pekerjaan' => $anggota->pekerjaan,
                    'status' => 'Aktif', // Default status
                    'foto' => $anggota->foto ?? null
                ];
            }
            
            return response()->json([
                'success' => true,
                'data' => $familyData
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }
    
    private function determineRelationship($anggota, $allMembers, $currentUser)
    {
        // If it's the current user
        if ($anggota->nik === $currentUser->nik) {
            return 'Kepala Keluarga'; // or based on actual data if available
        }
        
        // Simple logic based on age and gender
        $age = $anggota->tanggal_lahir ? now()->diffInYears($anggota->tanggal_lahir) : 0;
        
        if ($age < 18) {
            return 'Anak';
        } elseif ($anggota->jenis_kelamin === 'P' && $age >= 18) {
            return 'Istri';
        } else {
            return 'Suami';
        }
    }

    public function getStatistics(Request $request)
    {
        try {
            $nik = $request->input('nik');
            
            if (!$nik) {
                return response()->json([
                    'success' => false,
                    'message' => 'NIK is required'
                ], 400);
            }
            
            // Get statistics from database
            $totalSurat = PelayananSurat::where('nik', $nik)->count();
            $suratProses = PelayananSurat::where('nik', $nik)
                ->whereIn('status', ['Menunggu', 'Diproses'])
                ->count();
            $suratSelesai = PelayananSurat::where('nik', $nik)
                ->where('status', 'Selesai')
                ->count();
            
            return response()->json([
                'success' => true,
                'data' => [
                    'total_surat' => $totalSurat,
                    'surat_proses' => $suratProses,
                    'surat_selesai' => $suratSelesai
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }
} 