<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ProdukUmkm;
use Illuminate\Http\Request;

class ProdukController extends Controller
{
    public function index(Request $request)
    {
        $query = ProdukUmkm::with(['umkm'])
                           ->aktif();
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_produk', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhereHas('umkm', function($qq) use ($search) {
                      $qq->where('nama_umkm', 'like', "%{$search}%");
                  });
            });
        }
        
        // Filter by category
        if ($request->filled('kategori')) {
            $query->where('kategori_produk', $request->kategori);
        }
        
        // Filter by UMKM
        if ($request->filled('umkm_id')) {
            $query->where('umkm_id', $request->umkm_id);
        }
        
        // Filter by featured
        if ($request->filled('featured') && $request->featured == '1') {
            $query->featured();
        }
        
        // Filter by price range
        if ($request->filled('harga_min')) {
            $query->where('harga', '>=', $request->harga_min);
        }
        
        if ($request->filled('harga_max')) {
            $query->where('harga', '<=', $request->harga_max);
        }
        
        // Sort
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        
        if (in_array($sortBy, ['nama_produk', 'harga', 'rating_rata_rata', 'total_terjual', 'created_at'])) {
            $query->orderBy($sortBy, $sortOrder);
        }
        
        $produk = $query->paginate(20);
        
        return response()->json([
            'success' => true,
            'data' => $produk,
            'message' => 'Data produk berhasil diambil'
        ]);
    }
    
    public function show($id)
    {
        $produk = ProdukUmkm::with(['umkm', 'reviews.user'])
                            ->aktif()
                            ->findOrFail($id);
        
        return response()->json([
            'success' => true,
            'data' => $produk,
            'message' => 'Detail produk berhasil diambil'
        ]);
    }
    
    public function featured()
    {
        $produk = ProdukUmkm::with(['umkm'])
                            ->aktif()
                            ->featured()
                            ->orderBy('rating_rata_rata', 'desc')
                            ->limit(10)
                            ->get();
        
        return response()->json([
            'success' => true,
            'data' => $produk,
            'message' => 'Produk unggulan berhasil diambil'
        ]);
    }
    
    public function bestseller()
    {
        $produk = ProdukUmkm::with(['umkm'])
                            ->aktif()
                            ->orderBy('total_terjual', 'desc')
                            ->limit(10)
                            ->get();
        
        return response()->json([
            'success' => true,
            'data' => $produk,
            'message' => 'Produk terlaris berhasil diambil'
        ]);
    }
    
    public function categories()
    {
        $categories = [
            'makanan' => 'Makanan',
            'minuman' => 'Minuman',
            'kerajinan_tangan' => 'Kerajinan Tangan',
            'tekstil' => 'Tekstil',
            'elektronik' => 'Elektronik',
            'kosmetik' => 'Kosmetik',
            'pertanian' => 'Produk Pertanian',
            'perikanan' => 'Produk Perikanan',
            'peternakan' => 'Produk Peternakan',
            'olahan_makanan' => 'Olahan Makanan',
            'furniture' => 'Furniture',
            'aksesoris' => 'Aksesoris',
            'lainnya' => 'Lainnya'
        ];
        
        return response()->json([
            'success' => true,
            'data' => $categories,
            'message' => 'Kategori produk berhasil diambil'
        ]);
    }
    
    public function byUmkm($umkmId)
    {
        $produk = ProdukUmkm::with(['umkm'])
                            ->where('umkm_id', $umkmId)
                            ->aktif()
                            ->orderBy('created_at', 'desc')
                            ->paginate(20);
        
        return response()->json([
            'success' => true,
            'data' => $produk,
            'message' => 'Produk UMKM berhasil diambil'
        ]);
    }
} 