<?php

namespace App\Http\Controllers;

use App\Models\Bantuan;
use App\Models\Penduduk;
use App\Services\WhatsAppService;
use Illuminate\Http\Request;

class BantuanController extends Controller
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    public function index(Request $request)
    {
        $query = Bantuan::with('penduduk');

        // Filter berdasarkan jenis bantuan
        if ($request->jenis_bantuan) {
            $query->byJenis($request->jenis_bantuan);
        }

        // Filter berdasarkan periode
        if ($request->periode) {
            $query->byPeriode($request->periode);
        }

        // Filter berdasarkan status
        if ($request->status) {
            $query->where('status', $request->status);
        }

        // Filter berdasarkan tanggal distribusi
        if ($request->tanggal_dari && $request->tanggal_sampai) {
            $query->whereBetween('tanggal_distribusi', [$request->tanggal_dari, $request->tanggal_sampai]);
        }

        // Pencarian
        if ($request->search) {
            $query->whereHas('penduduk', function($q) use ($request) {
                $q->where('nama', 'like', '%' . $request->search . '%')
                  ->orWhere('nik', 'like', '%' . $request->search . '%');
            });
        }

        $bantuan = $query->orderBy('created_at', 'desc')->paginate(20);

        // Data untuk filter
        $jenisBantuanList = Bantuan::distinct()->pluck('jenis_bantuan');
        $periodeList = Bantuan::distinct()->pluck('periode')->filter()->sort()->values();
        
        // Get penduduk data for the add modal
        $penduduk = Penduduk::orderBy('nama')->get();
        
        // Calculate statistics
        $stats = [
            'total_bantuan' => Bantuan::count(),
            'total_penerima' => Bantuan::where('status', 'Diterima')->distinct('nik')->count(),
            'bulan_ini' => Bantuan::whereMonth('created_at', now()->month)->count(),
            'menunggu_verifikasi' => Bantuan::where('status', 'Diajukan')->count(),
        ];
        
        // Chart data for different bantuan types
        $chart_data = [
            'blt' => Bantuan::where('jenis_bantuan', 'LIKE', '%BLT%')->count(),
            'pkh' => Bantuan::where('jenis_bantuan', 'PKH')->count(),
            'bpnt' => Bantuan::where('jenis_bantuan', 'BPNT')->count(),
            'sembako' => Bantuan::where('jenis_bantuan', 'LIKE', '%Sembako%')->orWhere('jenis_bantuan', 'Bansos Pangan')->count(),
            'bsu' => Bantuan::where('jenis_bantuan', 'LIKE', '%BSU%')->count(),
            'lainnya' => Bantuan::where('jenis_bantuan', 'Lainnya')->count(),
        ];

        return view('admin.bantuan.index', compact('bantuan', 'jenisBantuanList', 'periodeList', 'penduduk', 'stats', 'chart_data'));
    }

    public function create()
    {
        $penduduk = Penduduk::orderBy('nama')->get();
        return view('admin.bantuan.create', compact('penduduk'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'jenis_bantuan' => 'required|string',
            'deskripsi' => 'nullable|string',
            'periode' => 'required|string',
            'nominal' => 'nullable|numeric|min:0',
            'status' => 'required|in:Diajukan,Diverifikasi,Diterima,Ditolak',
            'tanggal_distribusi' => 'nullable|date',
            'keterangan' => 'nullable|string'
        ]);

        $bantuan = Bantuan::create($validated);

        // Kirim notifikasi WA jika status Diterima
        if ($validated['status'] === 'Diterima') {
            $this->sendApprovalNotification($bantuan);
        }

        return redirect()->route('admin.bantuan.index')
                        ->with('success', 'Data bantuan berhasil ditambahkan');
    }

    public function show(Bantuan $bantuan)
    {
        $bantuan->load('penduduk');
        return view('admin.bantuan.show', compact('bantuan'));
    }

    public function edit(Bantuan $bantuan)
    {
        $penduduk = Penduduk::orderBy('nama')->get();
        return view('admin.bantuan.edit', compact('bantuan', 'penduduk'));
    }

    public function update(Request $request, Bantuan $bantuan)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'jenis_bantuan' => 'required|string',
            'deskripsi' => 'nullable|string',
            'periode' => 'required|string',
            'nominal' => 'nullable|numeric|min:0',
            'status' => 'required|in:Diajukan,Diverifikasi,Diterima,Ditolak',
            'tanggal_distribusi' => 'nullable|date',
            'keterangan' => 'nullable|string'
        ]);

        $oldStatus = $bantuan->status;
        $bantuan->update($validated);

        // Kirim notifikasi WA jika status berubah ke Diterima
        if ($oldStatus !== 'Diterima' && $validated['status'] === 'Diterima') {
            $this->sendApprovalNotification($bantuan);
        }

        return redirect()->route('admin.bantuan.index')
                        ->with('success', 'Data bantuan berhasil diperbarui');
    }

    public function destroy(Bantuan $bantuan)
    {
        $bantuan->delete();
        
        return redirect()->route('admin.bantuan.index')
                        ->with('success', 'Data bantuan berhasil dihapus');
    }

    public function distribusi(Request $request, Bantuan $bantuan)
    {
        $request->validate([
            'tanggal_distribusi' => 'required|date',
            'keterangan' => 'nullable|string'
        ]);

        $bantuan->update([
            'status' => 'Diterima',
            'tanggal_distribusi' => $request->tanggal_distribusi,
            'keterangan' => $request->keterangan
        ]);

        // Kirim notifikasi WA
        $this->sendApprovalNotification($bantuan);

        return response()->json([
            'success' => true,
            'message' => 'Bantuan berhasil didistribusikan'
        ]);
    }

    private function sendApprovalNotification(Bantuan $bantuan)
    {
        $penduduk = $bantuan->penduduk;
        
        if ($penduduk->no_wa) {
            $message = $this->whatsappService->templateBantuanDiterima(
                $penduduk->nama,
                $bantuan->jenis_bantuan,
                $bantuan->periode
            );

            $this->whatsappService->sendMessage(
                $penduduk->no_wa,
                $message,
                'bantuan_diterima',
                $bantuan->id,
                Bantuan::class
            );
        }
    }

    // Import data bantuan dari Excel
    public function importForm()
    {
        return view('admin.bantuan.import');
    }

    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:5120'
        ]);

        try {
            // Implementasi import Excel akan ditambahkan jika diperlukan
            // Excel::import(new BantuanImport, $request->file('file'));
            
            return redirect()->route('admin.bantuan.index')
                            ->with('success', 'Data bantuan berhasil diimport');
        } catch (\Exception $e) {
            return redirect()->back()
                            ->with('error', 'Gagal import data: ' . $e->getMessage());
        }
    }

    // Laporan bantuan per periode
    public function laporan(Request $request)
    {
        $query = Bantuan::with('penduduk');

        if ($request->periode) {
            $query->where('periode', $request->periode);
        }

        if ($request->jenis_bantuan) {
            $query->where('jenis_bantuan', $request->jenis_bantuan);
        }

        $bantuan = $query->where('status', 'Diterima')->get();

        $statistik = [
            'total_penerima' => $bantuan->count(),
            'total_nominal' => $bantuan->sum('nominal'),
            'per_jenis' => $bantuan->groupBy('jenis_bantuan')->map->count(),
            'per_dusun' => $bantuan->groupBy(function($item) {
                return $item->penduduk->dusun ?? 'Tidak Diketahui';
            })->map->count()
        ];

        return view('admin.bantuan.laporan', compact('bantuan', 'statistik'));
    }

    // API Methods untuk Flutter
    public function apiIndex(Request $request)
    {
        $nik = $request->user()->nik ?? $request->nik;
        
        $bantuan = Bantuan::where('nik', $nik)
                         ->orderBy('created_at', 'desc')
                         ->get()
                         ->map(function($item) {
                             return [
                                 'id' => $item->id,
                                 'jenis_bantuan' => $item->jenis_bantuan,
                                 'deskripsi' => $item->deskripsi,
                                 'periode' => $item->periode,
                                 'nominal' => $item->nominal,
                                 'nominal_format' => $item->nominal_format,
                                 'status' => $item->status,
                                 'keterangan' => $item->keterangan,
                                 'tanggal_distribusi' => $item->tanggal_distribusi ? $item->tanggal_distribusi->format('d/m/Y') : null,
                                 'tanggal_pengajuan' => $item->created_at->format('d/m/Y')
                             ];
                         });

        return response()->json([
            'success' => true,
            'data' => $bantuan
        ]);
    }

    public function apiStore(Request $request)
    {
        $validated = $request->validate([
            'nik' => 'required|exists:penduduk,nik',
            'jenis_bantuan' => 'required|string',
            'deskripsi' => 'nullable|string',
            'periode' => 'required|string'
        ]);

        $validated['status'] = 'Diajukan';

        $bantuan = Bantuan::create($validated);

        return response()->json([
            'success' => true,
            'message' => 'Pengajuan bantuan berhasil dikirim',
            'data' => [
                'id' => $bantuan->id,
                'jenis_bantuan' => $bantuan->jenis_bantuan,
                'periode' => $bantuan->periode,
                'status' => $bantuan->status,
                'tanggal_pengajuan' => $bantuan->created_at->format('d/m/Y')
            ]
        ]);
    }

    // API untuk melihat jenis bantuan yang tersedia
    public function apiJenisBantuan()
    {
        $jenisBantuan = [
            'PKH',
            'Bansos Pangan',
            'BLT Dana Desa',
            'Kartu Prakerja',
            'Bantuan Pendidikan',
            'Bantuan Kesehatan',
            'Bantuan Perumahan',
            'BPNT',
            'Lainnya'
        ];

        return response()->json([
            'success' => true,
            'data' => $jenisBantuan
        ]);
    }
} 
