<?php

namespace App\Http\Controllers;

use App\Models\InventarisDesa;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class InventarisDesaController extends Controller
{
    public function index(Request $request)
    {
        $query = InventarisDesa::with('creator');

        if ($request->filled('kategori')) {
            $query->byKategori($request->kategori);
        }

        if ($request->filled('kondisi')) {
            $query->byKondisi($request->kondisi);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_barang', 'like', "%{$search}%")
                  ->orWhere('kode_inventaris', 'like', "%{$search}%");
            });
        }

        $inventaris = $query->latest()->paginate(15);
        
        return view('admin.inventaris-desa.index', compact('inventaris'));
    }

    public function create()
    {
        $kategori = [
            'Tanah', 'Bangunan', 'Kendaraan', 'Elektronik', 'Furniture',
            'Alat Tulis Kantor', 'Alat Kesehatan', 'Alat Pertanian'
        ];
        
        return view('admin.inventaris-desa.create', compact('kategori'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nama_barang' => 'required|string|max:255',
            'kategori' => 'required|string|max:100',
            'harga_perolehan' => 'required|numeric|min:0',
            'kondisi' => 'required|string',
            'lokasi_penyimpanan' => 'required|string|max:255'
        ]);

        $data = $request->all();
        $data['created_by'] = Auth::id();
        $data['kode_inventaris'] = InventarisDesa::generateKodeInventaris($request->kategori);
        $data['nilai_sekarang'] = $request->harga_perolehan;

        $inventaris = InventarisDesa::create($data);

        AuditLog::logDataChange('create', $inventaris, null, $data);

        return redirect()->route('admin.inventaris-desa.index')
                        ->with('success', 'Data inventaris berhasil ditambahkan');
    }

    public function show(InventarisDesa $inventarisDesa)
    {
        return view('admin.inventaris-desa.show', compact('inventarisDesa'));
    }

    public function edit(InventarisDesa $inventarisDesa)
    {
        $kategori = [
            'Tanah', 'Bangunan', 'Kendaraan', 'Elektronik', 'Furniture',
            'Alat Tulis Kantor', 'Alat Kesehatan', 'Alat Pertanian'
        ];
        
        return view('admin.inventaris-desa.edit', compact('inventarisDesa', 'kategori'));
    }

    public function update(Request $request, InventarisDesa $inventarisDesa)
    {
        $request->validate([
            'nama_barang' => 'required|string|max:255',
            'kategori' => 'required|string|max:100',
            'harga_perolehan' => 'required|numeric|min:0',
            'kondisi' => 'required|string',
            'lokasi_penyimpanan' => 'required|string|max:255'
        ]);

        $oldData = $inventarisDesa->toArray();
        $inventarisDesa->update($request->all());

        AuditLog::logDataChange('update', $inventarisDesa, $oldData, $request->all());

        return redirect()->route('admin.inventaris-desa.index')
                        ->with('success', 'Data inventaris berhasil diperbarui');
    }

    public function destroy(InventarisDesa $inventarisDesa)
    {
        $oldData = $inventarisDesa->toArray();
        $inventarisDesa->delete();

        AuditLog::logDataChange('delete', $inventarisDesa, $oldData, null);

        return redirect()->route('admin.inventaris-desa.index')
                        ->with('success', 'Data inventaris berhasil dihapus');
    }

    public function qrCode(InventarisDesa $inventarisDesa)
    {
        $qrCode = $inventarisDesa->generateQrCode();
        
        return view('admin.inventaris-desa.qr-code', compact('inventarisDesa', 'qrCode'));
    }

    public function statistik()
    {
        $stats = [
            'total_nilai' => InventarisDesa::getTotalNilai(),
            'kondisi' => InventarisDesa::getStatistikKondisi(),
            'per_kategori' => InventarisDesa::selectRaw('kategori, COUNT(*) as total')
                                           ->groupBy('kategori')
                                           ->get()
        ];
        
        return view('admin.inventaris-desa.statistik', compact('stats'));
    }
} 