<?php

namespace App\Http\Controllers;

use App\Models\KegiatanPembangunan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class KegiatanPembangunanController extends Controller
{
    public function index(Request $request)
    {
        $query = KegiatanPembangunan::with('user');

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('jenis_kegiatan')) {
            $query->byJenis($request->jenis_kegiatan);
        }

        $kegiatan = $query->orderBy('tanggal_mulai', 'desc')->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $kegiatan
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nama_kegiatan' => 'required|string|max:255',
            'deskripsi' => 'required|string',
            'lokasi' => 'required|string|max:255',
            'jenis_kegiatan' => 'required|string',
            'sumber_dana' => 'required|string',
            'total_anggaran' => 'required|numeric|min:0',
            'tanggal_mulai' => 'required|date',
            'tanggal_selesai_rencana' => 'required|date',
            'penanggung_jawab' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $validator->validated();
        $data['user_id'] = Auth::id();
        $data['sisa_anggaran'] = $data['total_anggaran'];

        $kegiatan = KegiatanPembangunan::create($data);

        return response()->json([
            'success' => true,
            'message' => 'Kegiatan pembangunan berhasil ditambahkan',
            'data' => $kegiatan
        ], 201);
    }

    public function show($id)
    {
        $kegiatan = KegiatanPembangunan::with('user')->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $kegiatan
        ]);
    }

    public function update(Request $request, $id)
    {
        $kegiatan = KegiatanPembangunan::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'nama_kegiatan' => 'required|string|max:255',
            'deskripsi' => 'required|string',
            'lokasi' => 'required|string|max:255',
            'alamat_lengkap' => 'required|string',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'jenis_kegiatan' => 'required|in:fisik,non_fisik,infrastruktur,pemberdayaan,kesehatan,pendidikan,ekonomi',
            'sumber_dana' => 'required|in:dana_desa,add,pad,apbd_kabupaten,apbd_provinsi,apbn,swadaya,hibah,csr',
            'total_anggaran' => 'required|numeric|min:0',
            'tanggal_mulai' => 'required|date',
            'tanggal_selesai_rencana' => 'required|date|after:tanggal_mulai',
            'tanggal_selesai_aktual' => 'nullable|date',
            'pelaksana_kegiatan' => 'nullable|string|max:255',
            'penanggung_jawab' => 'required|string|max:255',
            'manfaat_kegiatan' => 'nullable|string',
            'kendala' => 'nullable|string',
            'status' => 'required|in:perencanaan,berjalan,selesai,tertunda,dibatalkan'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $validator->validated();

        // Recalculate sisa anggaran if total anggaran changed
        if ($data['total_anggaran'] != $kegiatan->total_anggaran) {
            $data['sisa_anggaran'] = $data['total_anggaran'] - $kegiatan->realisasi_anggaran;
        }

        $kegiatan->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Kegiatan pembangunan berhasil diperbarui',
            'data' => $kegiatan->load('user')
        ]);
    }

    public function destroy($id)
    {
        $kegiatan = KegiatanPembangunan::findOrFail($id);

        // Delete documentation files if exist
        if ($kegiatan->dokumentasi) {
            foreach ($kegiatan->dokumentasi as $file) {
                if (Storage::exists('public/' . $file)) {
                    Storage::delete('public/' . $file);
                }
            }
        }

        $kegiatan->delete();

        return response()->json([
            'success' => true,
            'message' => 'Kegiatan pembangunan berhasil dihapus'
        ]);
    }

    public function updateProgres(Request $request, $id)
    {
        $kegiatan = KegiatanPembangunan::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'progres_fisik' => 'nullable|integer|min:0|max:100',
            'progres_keuangan' => 'nullable|integer|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $kegiatan->updateProgres(
            $request->input('progres_fisik'),
            $request->input('progres_keuangan')
        );

        return response()->json([
            'success' => true,
            'message' => 'Progres kegiatan berhasil diperbarui'
        ]);
    }

    public function selesaikan(Request $request, $id)
    {
        $kegiatan = KegiatanPembangunan::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'tanggal_selesai' => 'nullable|date',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $kegiatan->selesaikan($request->input('tanggal_selesai'));

        return response()->json([
            'success' => true,
            'message' => 'Kegiatan berhasil diselesaikan',
            'data' => $kegiatan->fresh()
        ]);
    }

    public function uploadDokumentasi(Request $request, $id)
    {
        $kegiatan = KegiatanPembangunan::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'dokumentasi' => 'required|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $file = $request->file('dokumentasi');
        $filename = time() . '_' . $file->getClientOriginalName();
        $path = $file->storeAs('public/kegiatan/dokumentasi', $filename);
        
        $kegiatan->addDokumentasi('kegiatan/dokumentasi/' . $filename);

        return response()->json([
            'success' => true,
            'message' => 'Dokumentasi berhasil diunggah',
            'data' => $kegiatan->fresh()
        ]);
    }

    public function statistik()
    {
        $stats = KegiatanPembangunan::getStatistikProgres();

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }
} 