<?php

namespace App\Http\Controllers;

use App\Models\Keluarga;
use App\Models\AnggotaKeluarga;
use App\Models\Penduduk;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class KeluargaController extends Controller
{
    /**
     * Display a listing of families
     */
    public function index(Request $request)
    {
        $query = Keluarga::with(['kepalaKeluarga', 'anggotaKeluarga.penduduk', 'anggota.penduduk']);

        // Search functionality
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('no_kk', 'like', '%' . $search . '%')
                  ->orWhereHas('kepalaKeluarga', function($q2) use ($search) {
                      $q2->where('nama', 'like', '%' . $search . '%');
                  });
            });
        }

        // Filter by dusun
        if ($request->has('dusun') && $request->dusun) {
            $query->where('dusun', $request->dusun);
        }

        // Filter by RT
        if ($request->has('rt') && $request->rt) {
            $query->where('rt', $request->rt);
        }

        // Filter by RW
        if ($request->has('rw') && $request->rw) {
            $query->where('rw', $request->rw);
        }

        $keluarga = $query->orderBy('created_at', 'desc')->paginate(15);
        
        // Get filter options
        $dusunList = Keluarga::distinct('dusun')->pluck('dusun');
        $rtList = Keluarga::distinct('rt')->pluck('rt');
        $rwList = Keluarga::distinct('rw')->pluck('rw');

        // Get statistics
        $stats = [
            'total_keluarga' => Keluarga::count(),
            'total_anggota' => AnggotaKeluarga::count(),
            'kepala_keluarga_laki' => Keluarga::whereHas('kepalaKeluarga', function($q) {
                $q->where('jenis_kelamin', 'L');
            })->count(),
            'kepala_keluarga_perempuan' => Keluarga::whereHas('kepalaKeluarga', function($q) {
                $q->where('jenis_kelamin', 'P');
            })->count(),
        ];

        if ($request->ajax()) {
            return response()->json([
                'html' => view('admin.keluarga.table', compact('keluarga'))->render(),
                'pagination' => $keluarga->links()->render()
            ]);
        }

        return view('admin.keluarga.index', compact('keluarga', 'dusunList', 'rtList', 'rwList', 'stats'));
    }

    /**
     * Show the form for creating new family
     */
    public function create()
    {
        // Get penduduk yang belum menjadi kepala keluarga
        $availablePenduduk = Penduduk::whereNotIn('nik', function($query) {
            $query->select('kepala_keluarga_nik')->from('keluarga');
        })->orderBy('nama')->get();

        return view('admin.keluarga.create', compact('availablePenduduk'));
    }

    /**
     * Store new family
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'no_kk' => 'required|string|unique:keluarga,no_kk|max:16',
            'kepala_keluarga_nik' => 'required|string|exists:penduduk,nik',
            'alamat' => 'required|string|max:255',
            'dusun' => 'nullable|string|max:100',
            'rt' => 'nullable|string|max:3',
            'rw' => 'nullable|string|max:3',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'anggota' => 'nullable|array',
            'anggota.*.nik' => 'required|string|exists:penduduk,nik',
            'anggota.*.hubungan' => 'required|string|max:50'
        ]);

        DB::beginTransaction();
        try {
            // Create family record
            $keluarga = Keluarga::create($validated);

            // Add kepala keluarga as first member
            AnggotaKeluarga::create([
                'no_kk' => $validated['no_kk'],
                'nik' => $validated['kepala_keluarga_nik'],
                'status_keluarga' => 'Kepala Keluarga'
            ]);

            // Add other family members
            if (isset($validated['anggota'])) {
                foreach ($validated['anggota'] as $anggota) {
                    // Skip if same as kepala keluarga
                    if ($anggota['nik'] !== $validated['kepala_keluarga_nik']) {
                        AnggotaKeluarga::create([
                            'no_kk' => $validated['no_kk'],
                            'nik' => $anggota['nik'],
                            'status_keluarga' => $anggota['hubungan']
                        ]);
                    }
                }
            }

            DB::commit();

            return redirect()->route('admin.keluarga.index')
                           ->with('success', 'Data keluarga berhasil ditambahkan!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()
                        ->with('error', 'Gagal menyimpan data keluarga: ' . $e->getMessage());
        }
    }

    /**
     * Show specific family
     */
    public function show(Keluarga $keluarga)
    {
        $keluarga->load(['kepalaKeluarga', 'anggotaKeluarga.penduduk', 'anggota.penduduk']);
        
        // Get family statistics with null checks
        $anggotaKeluarga = $keluarga->anggotaKeluarga ?? collect();
        
        $stats = [
            'total_anggota' => $anggotaKeluarga->count(),
            'laki_laki' => $anggotaKeluarga->filter(function($anggota) {
                return $anggota->penduduk && $anggota->penduduk->jenis_kelamin === 'L';
            })->count(),
            'perempuan' => $anggotaKeluarga->filter(function($anggota) {
                return $anggota->penduduk && $anggota->penduduk->jenis_kelamin === 'P';
            })->count(),
            'anak' => $anggotaKeluarga->filter(function($anggota) {
                return $anggota->penduduk && $anggota->penduduk->umur < 18;
            })->count(),
            'dewasa' => $anggotaKeluarga->filter(function($anggota) {
                return $anggota->penduduk && $anggota->penduduk->umur >= 18;
            })->count()
        ];

        return view('admin.keluarga.show', compact('keluarga', 'stats'));
    }

    /**
     * Show the form for editing family
     */
    public function edit(Keluarga $keluarga)
    {
        $keluarga->load(['anggotaKeluarga.penduduk', 'anggota.penduduk', 'kepalaKeluarga']);
        
        // Get available penduduk for dropdown
        $penduduk = Penduduk::orderBy('nama')->get();

        return view('admin.keluarga.edit', compact('keluarga', 'penduduk'));
    }

    /**
     * Update family
     */
    public function update(Request $request, Keluarga $keluarga)
    {
        $validated = $request->validate([
            'no_kk' => 'required|string|max:16|unique:keluarga,no_kk,' . $keluarga->id,
            'kepala_keluarga_nik' => 'required|string|exists:penduduk,nik',
            'alamat' => 'required|string|max:255',
            'dusun' => 'nullable|string|max:100',
            'rt' => 'nullable|string|max:3',
            'rw' => 'nullable|string|max:3',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'anggota' => 'nullable|array',
            'anggota.*.nik' => 'required|string|exists:penduduk,nik',
            'anggota.*.hubungan' => 'required|string|max:50'
        ]);

        DB::beginTransaction();
        try {
            // Update family record
            $keluarga->update($validated);

            // Delete existing family members
            $keluarga->anggotaKeluarga()->delete();

            // Add kepala keluarga as first member
            AnggotaKeluarga::create([
                'no_kk' => $validated['no_kk'],
                'nik' => $validated['kepala_keluarga_nik'],
                'status_keluarga' => 'Kepala Keluarga'
            ]);

            // Add other family members
            if (isset($validated['anggota'])) {
                foreach ($validated['anggota'] as $anggota) {
                    // Skip if same as kepala keluarga
                    if ($anggota['nik'] !== $validated['kepala_keluarga_nik']) {
                        AnggotaKeluarga::create([
                            'no_kk' => $validated['no_kk'],
                            'nik' => $anggota['nik'],
                            'status_keluarga' => $anggota['hubungan']
                        ]);
                    }
                }
            }

            DB::commit();

            return redirect()->route('admin.keluarga.index')
                           ->with('success', 'Data keluarga berhasil diperbarui!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()
                        ->with('error', 'Gagal memperbarui data keluarga: ' . $e->getMessage());
        }
    }

    /**
     * Delete family
     */
    public function destroy(Keluarga $keluarga)
    {
        DB::beginTransaction();
        try {
            // Delete family members first
            $keluarga->anggotaKeluarga()->delete();
            
            // Delete family record
            $keluarga->delete();

            DB::commit();

            return redirect()->route('admin.keluarga.index')
                           ->with('success', 'Data keluarga berhasil dihapus!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Gagal menghapus data keluarga: ' . $e->getMessage());
        }
    }

    /**
     * Get coordinates from address using reverse geocoding
     */
    public function getCoordinates(Request $request)
    {
        $alamat = trim($request->input('alamat'));
        
        if (empty($alamat)) {
            return response()->json(['error' => 'Alamat tidak boleh kosong'], 400);
        }

        try {
            $url = "https://nominatim.openstreetmap.org/search";
            $params = [
                'q' => $alamat,
                'format' => 'json',
                'limit' => 1,
                'countrycodes' => 'id'
            ];

            $context = stream_context_create([
                'http' => [
                    'method' => 'GET',
                    'header' => 'User-Agent: Aplikasi-Desa/1.0'
                ]
            ]);

            $response = file_get_contents($url . '?' . http_build_query($params), false, $context);
            $data = json_decode($response, true);

            if (!empty($data)) {
                return response()->json([
                    'latitude' => floatval($data[0]['lat']),
                    'longitude' => floatval($data[0]['lon']),
                    'display_name' => $data[0]['display_name']
                ]);
            }

            return response()->json(['error' => 'Koordinat tidak ditemukan'], 404);

        } catch (\Exception $e) {
            return response()->json(['error' => 'Gagal mendapatkan koordinat'], 500);
        }
    }

    /**
     * Get available penduduk for family member selection
     */
    public function getAvailablePenduduk(Request $request)
    {
        $keluargaId = $request->get('keluarga_id');
        $search = $request->get('search');

        $query = Penduduk::query();

        // Exclude current family members if editing
        if ($keluargaId) {
            $keluarga = Keluarga::find($keluargaId);
            if ($keluarga) {
                $existingMembers = AnggotaKeluarga::where('no_kk', $keluarga->no_kk)
                                               ->pluck('nik')
                                               ->toArray();
                $query->whereNotIn('nik', $existingMembers);
            }
        }

        // Search by name or NIK
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('nama', 'like', '%' . $search . '%')
                  ->orWhere('nik', 'like', '%' . $search . '%');
            });
        }

        $penduduk = $query->orderBy('nama')->limit(10)->get();

        return response()->json($penduduk->map(function($p) {
            return [
                'nik' => $p->nik,
                'nama' => $p->nama,
                'jenis_kelamin' => $p->jenis_kelamin,
                'umur' => $p->umur,
                'alamat' => $p->alamat_lengkap
            ];
        }));
    }

    /**
     * Get family statistics
     */
    public function statistics()
    {
        $stats = [
            'total_keluarga' => Keluarga::count(),
            'total_anggota' => AnggotaKeluarga::count(),
            'rata_rata_anggota' => round(AnggotaKeluarga::count() / max(Keluarga::count(), 1), 2),
            'dusun_stats' => Keluarga::selectRaw('dusun, COUNT(*) as count')
                                   ->groupBy('dusun')
                                   ->orderBy('count', 'desc')
                                   ->get(),
            'hubungan_stats' => AnggotaKeluarga::selectRaw('status_keluarga, COUNT(*) as count')
                                             ->groupBy('status_keluarga')
                                             ->orderBy('count', 'desc')
                                             ->get()
        ];

        return response()->json($stats);
    }

    /**
     * Generic export method that handles different formats
     */
    public function export(Request $request, $format = 'csv')
    {
        switch (strtolower($format)) {
            case 'excel':
                return $this->exportExcel($request);
            case 'pdf':
                return $this->exportPdf($request);
            case 'csv':
            default:
                return $this->exportCsv($request);
        }
    }

    /**
     * Export family data to Excel
     */
    public function exportExcel(Request $request)
    {
        $query = Keluarga::with(['kepalaKeluarga', 'anggotaKeluarga.penduduk']);

        // Apply filters
        if ($request->dusun) {
            $query->where('dusun', $request->dusun);
        }
        if ($request->rt) {
            $query->where('rt', $request->rt);
        }
        if ($request->rw) {
            $query->where('rw', $request->rw);
        }

        $keluarga = $query->get();

        $headers = [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="data_keluarga_' . date('Y-m-d') . '.xlsx"',
        ];

        $callback = function() use ($keluarga) {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // CSV headers
            fputcsv($file, [
                'No KK',
                'Kepala Keluarga',
                'NIK Kepala Keluarga',
                'Alamat',
                'Dusun',
                'RT',
                'RW',
                'Jumlah Anggota',
                'Anggota Laki-laki',
                'Anggota Perempuan',
                'Dibuat Tanggal'
            ]);

            foreach ($keluarga as $k) {
                $lakiLaki = $k->anggotaKeluarga->filter(function($anggota) {
                    return $anggota->penduduk && $anggota->penduduk->jenis_kelamin === 'L';
                })->count();
                
                $perempuan = $k->anggotaKeluarga->filter(function($anggota) {
                    return $anggota->penduduk && $anggota->penduduk->jenis_kelamin === 'P';
                })->count();

                fputcsv($file, [
                    $k->no_kk,
                    $k->kepalaKeluarga->nama ?? '-',
                    $k->kepalaKeluarga->nik ?? '-',
                    $k->alamat,
                    $k->dusun,
                    $k->rt,
                    $k->rw,
                    $k->anggotaKeluarga->count(),
                    $lakiLaki,
                    $perempuan,
                    $k->created_at->format('Y-m-d')
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Export family data to PDF
     */
    public function exportPdf(Request $request)
    {
        $query = Keluarga::with(['kepalaKeluarga', 'anggotaKeluarga.penduduk']);

        // Apply filters
        if ($request->dusun) {
            $query->where('dusun', $request->dusun);
        }
        if ($request->rt) {
            $query->where('rt', $request->rt);
        }
        if ($request->rw) {
            $query->where('rw', $request->rw);
        }

        $keluarga = $query->paginate(50); // Limit for PDF

        return view('admin.keluarga.export-pdf', compact('keluarga'));
    }

    /**
     * Export family data to CSV
     */
    public function exportCsv(Request $request)
    {
        $query = Keluarga::with(['kepalaKeluarga', 'anggotaKeluarga.penduduk']);

        // Apply filters
        if ($request->dusun) {
            $query->where('dusun', $request->dusun);
        }
        if ($request->rt) {
            $query->where('rt', $request->rt);
        }
        if ($request->rw) {
            $query->where('rw', $request->rw);
        }

        $keluarga = $query->get();

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="data_keluarga_' . date('Y-m-d') . '.csv"',
        ];

        $callback = function() use ($keluarga) {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // CSV headers
            fputcsv($file, [
                'No KK',
                'Kepala Keluarga',
                'NIK Kepala Keluarga',
                'Alamat',
                'Dusun',
                'RT',
                'RW',
                'Jumlah Anggota'
            ]);

            foreach ($keluarga as $k) {
                fputcsv($file, [
                    $k->no_kk,
                    $k->kepalaKeluarga->nama ?? '-',
                    $k->kepalaKeluarga->nik ?? '-',
                    $k->alamat,
                    $k->dusun,
                    $k->rt,
                    $k->rw,
                    $k->anggotaKeluarga->count()
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Import Excel data
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:5120', // 5MB max
        ]);

        try {
            $file = $request->file('file');
            $path = $file->store('temp');
            $fullPath = storage_path('app/' . $path);
            
            // Read the file based on extension
            $extension = $file->getClientOriginalExtension();
            if (in_array($extension, ['xlsx', 'xls'])) {
                $data = $this->readExcelFile($fullPath);
            } else {
                $data = $this->readCsvFile($fullPath);
            }
            
            $imported = 0;
            $errors = [];
            
            foreach ($data as $row => $item) {
                try {
                    // Validate required fields
                    if (empty($item['no_kk']) || empty($item['kepala_keluarga_nik'])) {
                        $errors[] = "Baris " . ($row + 2) . ": No KK dan NIK Kepala Keluarga harus diisi";
                        continue;
                    }

                    // Check if KK already exists
                    if (Keluarga::where('no_kk', $item['no_kk'])->exists()) {
                        $errors[] = "Baris " . ($row + 2) . ": No KK {$item['no_kk']} sudah terdaftar";
                        continue;
                    }

                    // Check if kepala keluarga exists
                    $kepalaKeluarga = Penduduk::where('nik', $item['kepala_keluarga_nik'])->first();
                    if (!$kepalaKeluarga) {
                        $errors[] = "Baris " . ($row + 2) . ": NIK Kepala Keluarga {$item['kepala_keluarga_nik']} tidak ditemukan di data penduduk";
                        continue;
                    }

                    // Check if kepala keluarga already has family
                    if (Keluarga::where('kepala_keluarga_nik', $item['kepala_keluarga_nik'])->exists()) {
                        $errors[] = "Baris " . ($row + 2) . ": NIK {$item['kepala_keluarga_nik']} sudah menjadi kepala keluarga di KK lain";
                        continue;
                    }

                    DB::beginTransaction();

                    // Create keluarga record
                    $keluarga = Keluarga::create([
                        'no_kk' => $item['no_kk'],
                        'kepala_keluarga_nik' => $item['kepala_keluarga_nik'],
                        'alamat' => $item['alamat'] ?? '',
                        'dusun' => $item['dusun'] ?? null,
                        'rt' => $item['rt'] ?? null,
                        'rw' => $item['rw'] ?? null,
                        'latitude' => is_numeric($item['latitude'] ?? null) ? $item['latitude'] : null,
                        'longitude' => is_numeric($item['longitude'] ?? null) ? $item['longitude'] : null,
                    ]);

                    // Add kepala keluarga as first member
                    AnggotaKeluarga::create([
                        'no_kk' => $item['no_kk'],
                        'nik' => $item['kepala_keluarga_nik'],
                        'status_keluarga' => 'Kepala Keluarga'
                    ]);

                    DB::commit();
                    $imported++;
                    
                } catch (\Exception $e) {
                    DB::rollBack();
                    $errors[] = "Baris " . ($row + 2) . ": " . $e->getMessage();
                }
            }
            
            // Clean up temp file
            if (file_exists($fullPath)) {
                unlink($fullPath);
            }
            
            $message = "Berhasil import {$imported} data keluarga";
            if (count($errors) > 0) {
                $message .= ". " . count($errors) . " data gagal diimport.";
            }
            
            return redirect()->route('admin.keluarga.index')
                           ->with('success', $message)
                           ->with('import_errors', $errors);
                           
        } catch (\Exception $e) {
            return redirect()->back()
                           ->with('error', 'Gagal import data: ' . $e->getMessage());
        }
    }

    /**
     * Download Excel template
     */
    public function downloadTemplate()
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="template_data_keluarga.csv"',
        ];

        $callback = function() {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // CSV headers
            fputcsv($file, [
                'No KK',
                'NIK Kepala Keluarga',
                'Alamat',
                'Dusun',
                'RT',
                'RW',
                'Latitude',
                'Longitude'
            ]);

            // Sample data
            fputcsv($file, [
                '3201012201850001',
                '3201012201850001',
                'Jl. Merdeka No. 123',
                'Dusun Mekar',
                '01',
                '05',
                '-6.595038',
                '106.816635'
            ]);

            fputcsv($file, [
                '3201012205880002',
                '3201012205880002',
                'Jl. Kemerdekaan No. 456',
                'Dusun Sari',
                '02',
                '03',
                '-6.596000',
                '106.817000'
            ]);

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Read Excel file and return data array
     */
    private function readExcelFile($filePath)
    {
        try {
            // Simple CSV reading since we're just doing basic import
            return $this->readCsvFile($filePath);
        } catch (\Exception $e) {
            throw new \Exception("Error reading Excel file: " . $e->getMessage());
        }
    }

    /**
     * Read CSV file and return data array
     */
    private function readCsvFile($filePath)
    {
        $data = [];
        $handle = fopen($filePath, 'r');
        
        if ($handle === false) {
            throw new \Exception("Cannot open file");
        }

        // Read header row
        $headers = fgetcsv($handle);
        if ($headers === false) {
            fclose($handle);
            throw new \Exception("Cannot read header row");
        }

        // Normalize headers
        $headers = array_map(function($header) {
            return $this->normalizeHeader($header);
        }, $headers);

        // Read data rows
        $rowIndex = 0;
        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) === count($headers)) {
                $data[$rowIndex] = array_combine($headers, $row);
                $rowIndex++;
            }
        }

        fclose($handle);
        return $data;
    }

    /**
     * Normalize header names for consistent mapping
     */
    private function normalizeHeader($header)
    {
        $header = strtolower(trim($header));
        $header = str_replace([' ', '_', '-'], '_', $header);
        
        $mappings = [
            'no_kk' => 'no_kk',
            'nokk' => 'no_kk',
            'nomor_kk' => 'no_kk',
            'nomor_kartu_keluarga' => 'no_kk',
            'nik_kepala_keluarga' => 'kepala_keluarga_nik',
            'kepala_keluarga_nik' => 'kepala_keluarga_nik',
            'nik_kepala' => 'kepala_keluarga_nik',
            'kepala_nik' => 'kepala_keluarga_nik',
            'alamat' => 'alamat',
            'dusun' => 'dusun',
            'rt' => 'rt',
            'rw' => 'rw',
            'latitude' => 'latitude',
            'longitude' => 'longitude',
            'lat' => 'latitude',
            'lng' => 'longitude',
            'long' => 'longitude'
        ];

        return $mappings[$header] ?? $header;
    }

    // API Methods for mobile app

    /**
     * API: Get families list
     */
    public function apiIndex(Request $request)
    {
        $query = Keluarga::with(['kepalaKeluarga:nik,nama', 'anggotaKeluarga:id,no_kk,nik,status_keluarga']);

        // Search
        if ($request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('no_kk', 'like', '%' . $search . '%')
                  ->orWhereHas('kepalaKeluarga', function($q2) use ($search) {
                      $q2->where('nama', 'like', '%' . $search . '%');
                  });
            });
        }

        // Filters
        if ($request->dusun) {
            $query->where('dusun', $request->dusun);
        }

        $keluarga = $query->orderBy('created_at', 'desc')
                         ->paginate($request->get('per_page', 15));

        return response()->json([
            'success' => true,
            'data' => $keluarga->items(),
            'pagination' => [
                'current_page' => $keluarga->currentPage(),
                'last_page' => $keluarga->lastPage(),
                'per_page' => $keluarga->perPage(),
                'total' => $keluarga->total()
            ]
        ]);
    }

    /**
     * API: Get family detail
     */
    public function apiShow($id)
    {
        $keluarga = Keluarga::with(['kepalaKeluarga', 'anggotaKeluarga.penduduk'])
                           ->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $keluarga
        ]);
    }
} 
