<?php

namespace App\Http\Controllers;

use App\Models\Penduduk;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PohonKeluargaController extends Controller
{
    /**
     * Display family tree index page
     */
    public function index(Request $request)
    {
        $query = Penduduk::with(['ayah', 'ibu']);

        // Search functionality
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('nama', 'like', '%' . $request->search . '%')
                  ->orWhere('nik', 'like', '%' . $request->search . '%');
            });
        }

        // Filter by dusun
        if ($request->dusun) {
            $query->where('dusun', $request->dusun);
        }

        $penduduk = $query->paginate(20);

        // Get statistics
        $stats = [
            'total_penduduk' => Penduduk::count(),
            'with_parents' => Penduduk::where(function($q) {
                $q->whereNotNull('ayah_nik')->orWhereNotNull('ibu_nik');
            })->count(),
            'generasi_pertama' => Penduduk::whereNull('ayah_nik')->whereNull('ibu_nik')->count(),
            'total_keluarga_besar' => $this->getTotalKeluargaBesar()
        ];

        // Get dusun list for filter
        $dusunList = Penduduk::distinct()->pluck('dusun')->filter();

        return view('admin.pohon-keluarga.index', compact('penduduk', 'stats', 'dusunList'));
    }

    /**
     * Show detailed family tree for specific person
     */
    public function show($nik)
    {
        $penduduk = Penduduk::with(['ayah', 'ibu'])->where('nik', $nik)->firstOrFail();
        
        // Get complete family tree (descendants)
        $pohonKeturunan = $penduduk->getPohonKeluarga(4);
        
        // Get family lineage (ancestors)
        $garisKeturunan = $penduduk->getGarisKeturunan(4);
        
        // Get family members summary
        $keluargaSummary = [
            'orangtua' => $penduduk->orangTua(),
            'saudara' => $penduduk->saudara(),
            'anak' => $penduduk->getAllChildren(),
            'cucu' => $penduduk->cucu(),
            'kakek_nenek' => $penduduk->kakekNenek()
        ];

        return view('admin.pohon-keluarga.show', compact('penduduk', 'pohonKeturunan', 'garisKeturunan', 'keluargaSummary'));
    }

    /**
     * Show interactive family tree visualization
     */
    public function visualisasi($nik)
    {
        $penduduk = Penduduk::with(['ayah', 'ibu'])->where('nik', $nik)->firstOrFail();
        
        // Generate tree data for visualization
        $treeData = $this->generateTreeData($penduduk);
        
        return view('admin.pohon-keluarga.visualisasi', compact('penduduk', 'treeData'));
    }

    /**
     * Edit family relationships
     */
    public function edit($nik)
    {
        $penduduk = Penduduk::with(['ayah', 'ibu'])->where('nik', $nik)->firstOrFail();
        
        // Get potential parents (older people, different gender)
        $potentialAyah = Penduduk::where('jenis_kelamin', 'L')
                                ->where('nik', '!=', $penduduk->nik)
                                ->whereRaw('YEAR(tanggal_lahir) < YEAR(?)', [$penduduk->tanggal_lahir])
                                ->orderBy('nama')
                                ->get();
                                
        $potentialIbu = Penduduk::where('jenis_kelamin', 'P')
                               ->where('nik', '!=', $penduduk->nik)
                               ->whereRaw('YEAR(tanggal_lahir) < YEAR(?)', [$penduduk->tanggal_lahir])
                               ->orderBy('nama')
                               ->get();

        return view('admin.pohon-keluarga.edit', compact('penduduk', 'potentialAyah', 'potentialIbu'));
    }

    /**
     * Update family relationships
     */
    public function update(Request $request, $nik)
    {
        $request->validate([
            'ayah_nik' => 'nullable|exists:penduduk,nik',
            'ibu_nik' => 'nullable|exists:penduduk,nik',
            'status_keluarga' => 'nullable|string'
        ]);

        $penduduk = Penduduk::where('nik', $nik)->firstOrFail();
        
        // Validate that person is not setting themselves as parent
        if ($request->ayah_nik == $nik || $request->ibu_nik == $nik) {
            return back()->withErrors(['error' => 'Seseorang tidak dapat menjadi orang tua dari dirinya sendiri.']);
        }

        // Validate age logic (parents should be older)
        if ($request->ayah_nik) {
            $ayah = Penduduk::where('nik', $request->ayah_nik)->first();
            if ($ayah && $ayah->tanggal_lahir >= $penduduk->tanggal_lahir) {
                return back()->withErrors(['ayah_nik' => 'Ayah harus lahir sebelum anak.']);
            }
        }

        if ($request->ibu_nik) {
            $ibu = Penduduk::where('nik', $request->ibu_nik)->first();
            if ($ibu && $ibu->tanggal_lahir >= $penduduk->tanggal_lahir) {
                return back()->withErrors(['ibu_nik' => 'Ibu harus lahir sebelum anak.']);
            }
        }

        $penduduk->update([
            'ayah_nik' => $request->ayah_nik,
            'ibu_nik' => $request->ibu_nik,
            'status_keluarga' => $request->status_keluarga ?? $penduduk->status_keluarga
        ]);

        return redirect()->route('admin.pohon-keluarga.show', $nik)
                        ->with('success', 'Relasi keluarga berhasil diperbarui.');
    }

    /**
     * Get family tree API data for AJAX
     */
    public function apiTreeData($nik)
    {
        $penduduk = Penduduk::with(['ayah', 'ibu'])->where('nik', $nik)->firstOrFail();
        $treeData = $this->generateTreeData($penduduk);
        
        return response()->json($treeData);
    }

    /**
     * Search family members
     */
    public function search(Request $request)
    {
        $query = Penduduk::query();
        
        if ($request->q) {
            $query->where(function($q) use ($request) {
                $q->where('nama', 'like', '%' . $request->q . '%')
                  ->orWhere('nik', 'like', '%' . $request->q . '%');
            });
        }
        
        $results = $query->limit(10)->get(['nik', 'nama', 'tanggal_lahir', 'jenis_kelamin']);
        
        return response()->json($results);
    }

    /**
     * Get family statistics
     */
    public function statistik()
    {
        $stats = [
            'generasi' => $this->getGenerasiStats(),
            'keluarga_besar' => $this->getKeluargaBesarStats(),
            'distribusi_anak' => $this->getDistribusiAnakStats(),
            'umur_rata_rata' => $this->getUmurRataRataStats()
        ];

        return view('admin.pohon-keluarga.statistik', compact('stats'));
    }

    /**
     * Generate data for family tree visualization
     */
    private function generateTreeData($penduduk, $processedNiks = [], $depth = 0, $maxDepth = 3)
    {
        if ($depth > $maxDepth || in_array($penduduk->nik, $processedNiks)) {
            return null;
        }

        $processedNiks[] = $penduduk->nik;

        $nodeData = [
            'id' => $penduduk->nik,
            'name' => $penduduk->nama,
            'gender' => $penduduk->jenis_kelamin,
            'birth_date' => $penduduk->tanggal_lahir ? $penduduk->tanggal_lahir->format('d/m/Y') : null,
            'age' => $penduduk->umur,
            'status_keluarga' => $penduduk->status_keluarga,
            'no_kk' => $penduduk->no_kk,
            'children' => [],
            'parents' => []
        ];

        // Add parents
        if ($penduduk->ayah && !in_array($penduduk->ayah->nik, $processedNiks)) {
            $nodeData['parents']['father'] = $this->generateTreeData($penduduk->ayah, $processedNiks, $depth + 1, $maxDepth);
        }

        if ($penduduk->ibu && !in_array($penduduk->ibu->nik, $processedNiks)) {
            $nodeData['parents']['mother'] = $this->generateTreeData($penduduk->ibu, $processedNiks, $depth + 1, $maxDepth);
        }

        // Add children
        $children = $penduduk->getAllChildren();
        foreach ($children as $child) {
            if (!in_array($child->nik, $processedNiks)) {
                $childData = $this->generateTreeData($child, $processedNiks, $depth + 1, $maxDepth);
                if ($childData) {
                    $nodeData['children'][] = $childData;
                }
            }
        }

        return $nodeData;
    }

    /**
     * Get total number of extended families
     */
    private function getTotalKeluargaBesar()
    {
        // Find root ancestors (people with no parents)
        $roots = Penduduk::whereNull('ayah_nik')->whereNull('ibu_nik')->count();
        return $roots;
    }

    /**
     * Get generation statistics
     */
    private function getGenerasiStats()
    {
        $stats = [];
        
        // Count by generations (simplified)
        $stats['generasi_1'] = Penduduk::whereNull('ayah_nik')->whereNull('ibu_nik')->count();
        $stats['generasi_2'] = Penduduk::whereNotNull('ayah_nik')->orWhereNotNull('ibu_nik')
                                      ->whereDoesntHave('anakSebagaiAyah')
                                      ->whereDoesntHave('anakSebagaiIbu')->count();
        
        return $stats;
    }

    /**
     * Get extended family statistics
     */
    private function getKeluargaBesarStats()
    {
        return [
            'rata_rata_anak_per_keluarga' => round(Penduduk::whereNotNull('ayah_nik')->count() / max(Penduduk::count(), 1), 2),
            'keluarga_terbesar' => $this->getKeluargaTerbesar()
        ];
    }

    /**
     * Get largest family
     */
    private function getKeluargaTerbesar()
    {
        $result = DB::table('penduduk')
                   ->select('ayah_nik', DB::raw('COUNT(*) as jumlah_anak'))
                   ->whereNotNull('ayah_nik')
                   ->groupBy('ayah_nik')
                   ->orderBy('jumlah_anak', 'desc')
                   ->first();

        if ($result) {
            $ayah = Penduduk::where('nik', $result->ayah_nik)->first();
            return [
                'nama_ayah' => $ayah ? $ayah->nama : 'Tidak diketahui',
                'jumlah_anak' => $result->jumlah_anak
            ];
        }

        return null;
    }

    /**
     * Get child distribution statistics
     */
    private function getDistribusiAnakStats()
    {
        return DB::table('penduduk')
                ->select(DB::raw('COUNT(ayah_nik) as jumlah_anak, COUNT(*) as jumlah_keluarga'))
                ->whereNotNull('ayah_nik')
                ->groupBy('ayah_nik')
                ->get();
    }

    /**
     * Get average age statistics
     */
    private function getUmurRataRataStats()
    {
        return [
            'rata_rata_umur_ayah' => Penduduk::whereHas('anakSebagaiAyah')->avg(DB::raw('YEAR(CURDATE()) - YEAR(tanggal_lahir)')),
            'rata_rata_umur_ibu' => Penduduk::whereHas('anakSebagaiIbu')->avg(DB::raw('YEAR(CURDATE()) - YEAR(tanggal_lahir)'))
        ];
    }
}
