<?php

namespace App\Http\Controllers;

use App\Models\Polling;
use App\Models\PollingOpsi;
use App\Models\PollingVote;
use App\Models\Penduduk;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PollingController extends Controller
{
    public function index(Request $request)
    {
        $query = Polling::with(['opsi', 'votes']);

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter berdasarkan jenis
        if ($request->filled('jenis')) {
            $query->where('jenis_polling', $request->jenis);
        }

        // Pencarian
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('judul', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%");
            });
        }

        $polling = $query->latest()->paginate(10);
        
        return view('admin.polling.index', compact('polling'));
    }

    public function create()
    {
        $jenisPolling = Polling::JENIS_POLLING;
        $targetPeserta = Polling::TARGET_PESERTA;
        
        return view('admin.polling.create', compact('jenisPolling', 'targetPeserta'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'judul' => 'required|string|max:255',
            'deskripsi' => 'required|string',
            'jenis_polling' => 'required|in:Aspirasi,Voting,Survey,Polling',
            'target_peserta' => 'required|in:Semua Warga,RT Tertentu,RW Tertentu,Dusun Tertentu,Kelompok Usia',
            'filter_peserta' => 'nullable|array',
            'tanggal_mulai' => 'required|date|after_or_equal:today',
            'tanggal_selesai' => 'required|date|after:tanggal_mulai',
            'multiple_choice' => 'boolean',
            'anonymous' => 'boolean',
            'tampil_hasil' => 'boolean',
            'opsi' => 'required|array|min:2',
            'opsi.*.teks' => 'required|string|max:255',
            'opsi.*.deskripsi' => 'nullable|string|max:500'
        ]);

        DB::beginTransaction();
        try {
            $polling = Polling::create([
                'judul' => $request->judul,
                'deskripsi' => $request->deskripsi,
                'jenis_polling' => $request->jenis_polling,
                'target_peserta' => $request->target_peserta,
                'filter_peserta' => $request->filter_peserta,
                'tanggal_mulai' => $request->tanggal_mulai,
                'tanggal_selesai' => $request->tanggal_selesai,
                'multiple_choice' => $request->boolean('multiple_choice'),
                'anonymous' => $request->boolean('anonymous'),
                'tampil_hasil' => $request->boolean('tampil_hasil'),
                'status' => 'Draft',
                'created_by' => auth()->id()
            ]);

            // Buat opsi polling
            foreach ($request->opsi as $index => $opsiData) {
                PollingOpsi::create([
                    'polling_id' => $polling->id,
                    'teks_opsi' => $opsiData['teks'],
                    'deskripsi' => $opsiData['deskripsi'] ?? null,
                    'urutan' => $index + 1
                ]);
            }

            DB::commit();

            // Log activity
            AuditLog::logDataChange('create', $polling, null, $request->all());

            return redirect()->route('admin.polling.show', $polling)
                           ->with('success', 'Polling berhasil dibuat');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Gagal membuat polling: ' . $e->getMessage());
        }
    }

    public function show(Polling $polling)
    {
        $polling->load(['opsi.votes', 'votes.penduduk']);
        
        $stats = $polling->getStatistik();
        $targetPeserta = $polling->getTargetPeserta();
        
        return view('admin.polling.show', compact('polling', 'stats', 'targetPeserta'));
    }

    public function edit(Polling $polling)
    {
        if ($polling->status !== 'Draft') {
            return redirect()->back()
                           ->with('error', 'Hanya polling dengan status Draft yang bisa diedit');
        }

        $jenisPolling = Polling::JENIS_POLLING;
        $targetPeserta = Polling::TARGET_PESERTA;
        
        return view('admin.polling.edit', compact('polling', 'jenisPolling', 'targetPeserta'));
    }

    public function update(Request $request, Polling $polling)
    {
        if ($polling->status !== 'Draft') {
            return redirect()->back()
                           ->with('error', 'Hanya polling dengan status Draft yang bisa diedit');
        }

        $request->validate([
            'judul' => 'required|string|max:255',
            'deskripsi' => 'required|string',
            'jenis_polling' => 'required|in:Aspirasi,Voting,Survey,Polling',
            'target_peserta' => 'required|in:Semua Warga,RT Tertentu,RW Tertentu,Dusun Tertentu,Kelompok Usia',
            'filter_peserta' => 'nullable|array',
            'tanggal_mulai' => 'required|date|after_or_equal:today',
            'tanggal_selesai' => 'required|date|after:tanggal_mulai',
            'multiple_choice' => 'boolean',
            'anonymous' => 'boolean',
            'tampil_hasil' => 'boolean',
            'opsi' => 'required|array|min:2',
            'opsi.*.teks' => 'required|string|max:255',
            'opsi.*.deskripsi' => 'nullable|string|max:500'
        ]);

        $oldData = $polling->toArray();

        DB::beginTransaction();
        try {
            $polling->update([
                'judul' => $request->judul,
                'deskripsi' => $request->deskripsi,
                'jenis_polling' => $request->jenis_polling,
                'target_peserta' => $request->target_peserta,
                'filter_peserta' => $request->filter_peserta,
                'tanggal_mulai' => $request->tanggal_mulai,
                'tanggal_selesai' => $request->tanggal_selesai,
                'multiple_choice' => $request->boolean('multiple_choice'),
                'anonymous' => $request->boolean('anonymous'),
                'tampil_hasil' => $request->boolean('tampil_hasil')
            ]);

            // Hapus opsi lama dan buat yang baru
            $polling->opsi()->delete();
            foreach ($request->opsi as $index => $opsiData) {
                PollingOpsi::create([
                    'polling_id' => $polling->id,
                    'teks_opsi' => $opsiData['teks'],
                    'deskripsi' => $opsiData['deskripsi'] ?? null,
                    'urutan' => $index + 1
                ]);
            }

            DB::commit();

            // Log activity
            AuditLog::logDataChange('update', $polling, $oldData, $request->all());

            return redirect()->route('admin.polling.show', $polling)
                           ->with('success', 'Polling berhasil diperbarui');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Gagal memperbarui polling: ' . $e->getMessage());
        }
    }

    public function destroy(Polling $polling)
    {
        if ($polling->status === 'Aktif') {
            return redirect()->back()
                           ->with('error', 'Polling yang sedang aktif tidak bisa dihapus');
        }

        $oldData = $polling->toArray();

        DB::beginTransaction();
        try {
            $polling->votes()->delete();
            $polling->opsi()->delete();
            $polling->delete();

            DB::commit();

            // Log activity
            AuditLog::logDataChange('delete', $polling, $oldData, null);

            return redirect()->route('admin.polling.index')
                           ->with('success', 'Polling berhasil dihapus');

        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->back()
                           ->with('error', 'Gagal menghapus polling: ' . $e->getMessage());
        }
    }

    public function activate(Polling $polling)
    {
        if ($polling->status !== 'Draft') {
            return redirect()->back()
                           ->with('error', 'Hanya polling Draft yang bisa diaktifkan');
        }

        $polling->update(['status' => 'Aktif']);

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'ACTIVATE',
            'module' => 'Polling',
            'table_name' => 'polling',
            'record_id' => $polling->id,
            'description' => 'Activate polling: ' . $polling->judul
        ]);

        return redirect()->back()
                       ->with('success', 'Polling berhasil diaktifkan');
    }

    public function deactivate(Polling $polling)
    {
        if ($polling->status !== 'Aktif') {
            return redirect()->back()
                           ->with('error', 'Hanya polling aktif yang bisa dinonaktifkan');
        }

        $polling->update(['status' => 'Selesai']);

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'DEACTIVATE',
            'module' => 'Polling',
            'table_name' => 'polling',
            'record_id' => $polling->id,
            'description' => 'Deactivate polling: ' . $polling->judul
        ]);

        return redirect()->back()
                       ->with('success', 'Polling berhasil dinonaktifkan');
    }

    public function results(Polling $polling)
    {
        $stats = $polling->getStatistikLengkap();
        $votes = $polling->votes()->with(['penduduk', 'opsi'])->latest()->get();
        
        return view('admin.polling.results', compact('polling', 'stats', 'votes'));
    }

    public function export(Request $request, Polling $polling)
    {
        $request->validate([
            'format' => 'required|in:excel,pdf,csv'
        ]);

        // Log activity
        AuditLog::logActivity([
            'action_type' => 'EXPORT',
            'module' => 'Polling',
            'description' => "Export hasil polling: {$polling->judul} format {$request->format}",
            'metadata' => [
                'polling_id' => $polling->id,
                'format' => $request->format
            ]
        ]);

        // Implementation would depend on export library
        return redirect()->back()
                       ->with('info', 'Fitur export akan segera tersedia');
    }

    // Public methods untuk voting
    public function publicIndex()
    {
        $polling = Polling::aktif()
                         ->with(['opsi'])
                         ->get();
        
        return view('public.polling.index', compact('polling'));
    }

    public function publicShow(Polling $polling)
    {
        if ($polling->status !== 'Aktif') {
            abort(404);
        }

        // Check if user dapat vote
        $canVote = $polling->canUserVote(auth()->user()->nik ?? null);
        $hasVoted = false;
        
        if (auth()->check() && auth()->user()->nik) {
            $hasVoted = $polling->hasUserVoted(auth()->user()->nik);
        }

        $stats = $polling->tampil_hasil ? $polling->getStatistik() : null;
        
        return view('public.polling.show', compact('polling', 'canVote', 'hasVoted', 'stats'));
    }

    public function vote(Request $request, Polling $polling)
    {
        if ($polling->status !== 'Aktif') {
            return response()->json(['error' => 'Polling tidak aktif'], 400);
        }

        $nik = auth()->user()->nik ?? $request->nik;
        
        if (!$nik) {
            return response()->json(['error' => 'NIK diperlukan untuk voting'], 400);
        }

        if (!$polling->canUserVote($nik)) {
            return response()->json(['error' => 'Anda tidak memiliki hak untuk vote'], 403);
        }

        if ($polling->hasUserVoted($nik)) {
            return response()->json(['error' => 'Anda sudah melakukan voting'], 400);
        }

        $validation = $polling->multiple_choice ? 
            ['opsi_ids' => 'required|array|exists:polling_opsi,id'] :
            ['opsi_id' => 'required|exists:polling_opsi,id'];
            
        $request->validate($validation);

        DB::beginTransaction();
        try {
            if ($polling->multiple_choice) {
                foreach ($request->opsi_ids as $opsiId) {
                    PollingVote::create([
                        'polling_id' => $polling->id,
                        'opsi_id' => $opsiId,
                        'nik' => $polling->anonymous ? null : $nik,
                        'ip_address' => $request->ip(),
                        'user_agent' => $request->userAgent()
                    ]);
                }
            } else {
                PollingVote::create([
                    'polling_id' => $polling->id,
                    'opsi_id' => $request->opsi_id,
                    'nik' => $polling->anonymous ? null : $nik,
                    'ip_address' => $request->ip(),
                    'user_agent' => $request->userAgent()
                ]);
            }

            DB::commit();

            // Log activity
            AuditLog::logActivity([
                'action_type' => 'VOTE',
                'module' => 'Polling',
                'description' => "Vote pada polling: {$polling->judul}",
                'metadata' => [
                    'polling_id' => $polling->id,
                    'voter_nik' => $polling->anonymous ? null : $nik
                ]
            ]);

            return response()->json(['success' => 'Vote berhasil disimpan']);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['error' => 'Gagal menyimpan vote'], 500);
        }
    }
} 