<?php

namespace App\Http\Controllers;

use App\Models\WilayahAdministratif;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class WilayahAdministratifController extends Controller
{
    public function index(Request $request)
    {
        $query = WilayahAdministratif::with(['user', 'parent']);

        if ($request->has('tipe')) {
            $query->where('jenis_wilayah', $request->tipe);
        }

        if ($request->has('parent')) {
            $query->where('parent_id', $request->parent);
        }

        $wilayah = $query->orderBy('jenis_wilayah')->orderBy('kode_wilayah')->get();

        // Get statistics
        $stats = [
            'total_dusun' => WilayahAdministratif::where('jenis_wilayah', 'dusun')->count(),
            'total_rw' => WilayahAdministratif::where('jenis_wilayah', 'rw')->count(),
            'total_rt' => WilayahAdministratif::where('jenis_wilayah', 'rt')->count(),
            'total_kk' => WilayahAdministratif::sum('jumlah_kk'),
        ];

        // Get dusun list for parent options
        $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')->get();
        $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')->with('parent')->get();

        // If AJAX request, return JSON
        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'data' => $wilayah,
                'stats' => $stats
            ]);
        }

        // Return view for normal requests
        return view('admin.wilayah-administratif.index', compact('wilayah', 'stats', 'dusunList', 'rwList'));
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'jenis_wilayah' => 'required|in:dusun,rw,rt',
            'kode_wilayah' => 'required|string|max:20|unique:wilayah_administratif,kode_wilayah',
            'nama_wilayah' => 'required|string|max:255',
            'parent_id' => 'nullable|exists:wilayah_administratif,id',
            'kepala_wilayah' => 'nullable|string|max:255',
            'nomor_hp_kepala' => 'nullable|string|max:15',
            'jumlah_kk' => 'nullable|integer|min:0',
            'jumlah_penduduk' => 'nullable|integer|min:0',
            'luas_wilayah' => 'nullable|numeric|min:0',
            'batas_wilayah' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $validator->validated();
        $data['user_id'] = Auth::id();

        $wilayah = WilayahAdministratif::create($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Data wilayah administratif berhasil ditambahkan',
                'data' => $wilayah
            ], 201);
        }

        return redirect()->route('admin.wilayah-administratif.index')
                        ->with('success', 'Data wilayah administratif berhasil ditambahkan');
    }

    public function create()
    {
        $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')->get();
        $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')->with('parent')->get();
        
        return view('admin.wilayah-administratif.create', compact('dusunList', 'rwList'));
    }

    public function edit($id)
    {
        $wilayah = WilayahAdministratif::with(['parent'])->findOrFail($id);
        $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')->where('id', '!=', $id)->get();
        $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')->where('id', '!=', $id)->with('parent')->get();
        
        return view('admin.wilayah-administratif.edit', compact('wilayah', 'dusunList', 'rwList'));
    }

    public function show($id)
    {
        $wilayah = WilayahAdministratif::with(['user', 'parent', 'children'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $wilayah
        ]);
    }

    public function update(Request $request, $id)
    {
        $wilayah = WilayahAdministratif::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'jenis_wilayah' => 'required|in:dusun,rw,rt',
            'kode_wilayah' => 'required|string|max:20|unique:wilayah_administratif,kode_wilayah,' . $id,
            'nama_wilayah' => 'required|string|max:255',
            'parent_id' => 'nullable|exists:wilayah_administratif,id',
            'kepala_wilayah' => 'nullable|string|max:255',
            'nomor_hp_kepala' => 'nullable|string|max:15',
            'jumlah_kk' => 'nullable|integer|min:0',
            'jumlah_penduduk' => 'nullable|integer|min:0',
            'luas_wilayah' => 'nullable|numeric|min:0',
            'batas_wilayah' => 'nullable|string',
            'status' => 'required|in:aktif,nonaktif',
        ]);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $validator->validated();
        $wilayah->update($data);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Data wilayah administratif berhasil diperbarui',
                'data' => $wilayah
            ]);
        }

        return redirect()->route('admin.wilayah-administratif.index')
                        ->with('success', 'Data wilayah administratif berhasil diperbarui');
    }

    public function destroy($id)
    {
        $wilayah = WilayahAdministratif::findOrFail($id);

        // Check if has children
        if ($wilayah->children()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Tidak dapat menghapus wilayah yang masih memiliki wilayah di bawahnya'
            ], 400);
        }

        $wilayah->delete();

        return response()->json([
            'success' => true,
            'message' => 'Data wilayah administratif berhasil dihapus'
        ]);
    }

    public function byJenis($jenis)
    {
        $wilayah = WilayahAdministratif::getWilayahByJenis($jenis);

        return response()->json([
            'success' => true,
            'data' => $wilayah
        ]);
    }

    public function treeStructure()
    {
        $tree = WilayahAdministratif::getTreeStructure();

        return response()->json([
            'success' => true,
            'data' => $tree
        ]);
    }

    public function statistik()
    {
        $stats = WilayahAdministratif::getStatistikWilayah();

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

    public function updateStatistik($id)
    {
        $wilayah = WilayahAdministratif::findOrFail($id);
        $wilayah->updateStatistikPenduduk();

        return response()->json([
            'success' => true,
            'message' => 'Statistik penduduk berhasil diperbarui'
        ]);
    }

    public function dusunWithRwRt()
    {
        $dusun = WilayahAdministratif::getDusunWithRwRt();

        return response()->json([
            'success' => true,
            'data' => $dusun
        ]);
    }

    public function sync(Request $request)
    {
        try {
            // Logic to sync population data from penduduk table
            $wilayahList = WilayahAdministratif::all();
            
            foreach ($wilayahList as $wilayah) {
                // Update statistics based on penduduk data
                // This is a placeholder - you can implement actual sync logic
                $wilayah->updateStatistikPenduduk();
            }

            return response()->json([
                'success' => true,
                'message' => 'Data penduduk berhasil disinkronkan dengan wilayah administratif'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal melakukan sinkronisasi: ' . $e->getMessage()
            ], 500);
        }
    }
} 