<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class MutasiPenduduk extends Model
{
    use HasFactory;

    protected $fillable = [
        'nik',
        'jenis_mutasi',
        'tanggal_mutasi',
        'keterangan',
        'asal_daerah',
        'tujuan_daerah',
        'sebab_meninggal',
        'tempat_meninggal',
        'no_surat_keterangan',
        'data_lama',
        'admin_user',
        'catatan_admin'
    ];

    protected $casts = [
        'tanggal_mutasi' => 'date',
        'data_lama' => 'array'
    ];

    // Relationships
    public function penduduk()
    {
        return $this->belongsTo(Penduduk::class, 'nik', 'nik');
    }

    // Scopes
    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_mutasi', $jenis);
    }

    public function scopeByTahun($query, $tahun)
    {
        return $query->whereYear('tanggal_mutasi', $tahun);
    }

    public function scopeByBulan($query, $tahun, $bulan)
    {
        return $query->whereYear('tanggal_mutasi', $tahun)
                    ->whereMonth('tanggal_mutasi', $bulan);
    }

    // Accessors
    public function getJenisMutasiColorAttribute()
    {
        return match($this->jenis_mutasi) {
            'Lahir' => 'success',
            'Meninggal' => 'danger',
            'Pindah Masuk' => 'info',
            'Pindah Keluar' => 'warning',
            'Status Update' => 'secondary',
            default => 'primary'
        };
    }

    public function getJenisMutasiIconAttribute()
    {
        return match($this->jenis_mutasi) {
            'Lahir' => 'baby',
            'Meninggal' => 'cross',
            'Pindah Masuk' => 'arrow-right',
            'Pindah Keluar' => 'arrow-left',
            'Status Update' => 'edit',
            default => 'user'
        };
    }

    // Static methods untuk statistik
    public static function getStatistikBulanan($tahun = null)
    {
        $tahun = $tahun ?? date('Y');
        
        $data = self::selectRaw('MONTH(tanggal_mutasi) as bulan, jenis_mutasi, COUNT(*) as total')
                   ->whereYear('tanggal_mutasi', $tahun)
                   ->groupBy('bulan', 'jenis_mutasi')
                   ->get();

        $result = collect();
        for ($bulan = 1; $bulan <= 12; $bulan++) {
            $monthData = $data->where('bulan', $bulan);
            $byJenis = [];
            foreach ($monthData as $item) {
                $byJenis[$item->jenis_mutasi] = $item->total;
            }
            
            $result->push((object)[
                'bulan' => $bulan,
                'total' => $monthData->sum('total'),
                'by_jenis' => $byJenis
            ]);
        }
        
        return $result;
    }

    public static function getStatistikTahunan($startYear = null, $endYear = null)
    {
        $startYear = $startYear ?? (date('Y') - 4);
        $endYear = $endYear ?? date('Y');
        
        return self::selectRaw('YEAR(tanggal_mutasi) as tahun, COUNT(*) as total')
                   ->whereBetween('tanggal_mutasi', [$startYear.'-01-01', $endYear.'-12-31'])
                   ->groupBy('tahun')
                   ->orderBy('tahun', 'desc')
                   ->get();
    }
}
