<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class ObjekWisata extends Model
{
    use HasFactory;

    protected $table = 'objek_wisata';

    protected $fillable = [
        'nama_wisata',
        'slug',
        'deskripsi',
        'deskripsi_singkat',
        'kategori_wisata',
        'tags',
        'alamat',
        'latitude',
        'longitude',
        'cara_akses',
        'jarak_dari_pusat_desa',
        'kondisi_jalan',
        'transportasi_umum',
        'akses_kendaraan_roda_4',
        'akses_motor',
        'fasilitas',
        'ada_toilet',
        'ada_parkir',
        'ada_mushola',
        'ada_warung',
        'ada_penginapan',
        'ada_guide',
        'ada_wifi',
        'fasilitas_lainnya',
        'jam_operasional',
        'hari_tutup',
        'catatan_operasional',
        'musim_terbaik',
        'info_musim',
        'harga_tiket_dewasa',
        'harga_tiket_anak',
        'harga_tiket_rombongan',
        'minimal_rombongan',
        'harga_parkir_motor',
        'harga_parkir_mobil',
        'harga_guide',
        'catatan_harga',
        'foto_utama',
        'galeri_foto',
        'video_profil',
        'virtual_tour',
        'tingkat_kesulitan',
        'peringatan_keamanan',
        'aturan_berkunjung',
        'perlu_izin_khusus',
        'kontak_izin',
        'kapasitas_maksimal',
        'rating_rata_rata',
        'jumlah_review',
        'total_pengunjung',
        'pengunjung_bulan_ini',
        'view_count',
        'statistik_bulanan',
        'status',
        'is_featured',
        'is_recommended',
        'urutan_tampil',
        'tanggal_buka',
        'alasan_tutup',
        'pengelola',
        'nomor_telepon',
        'email',
        'website',
        'instagram',
        'facebook',
        'meta_title',
        'meta_description',
        'keyword',
        'user_id'
    ];

    protected $casts = [
        'hari_operasional' => 'array',
        'hari_libur' => 'array',
        'fasilitas_tersedia' => 'array',
        'aktivitas_tersedia' => 'array',
        'wahana_permainan' => 'array',
        'spot_foto_menarik' => 'array',
        'kuliner_khas' => 'array',
        'oleh_oleh_khas' => 'array',
        'penginapan_terdekat' => 'array',
        'galeri_foto' => 'array',
        'social_media' => 'array',
        'bahasa_guide' => 'array',
        'penghargaan_diterima' => 'array',
        'keyword_seo' => 'array',
        'perlengkapan_wajib' => 'array',
        'harga_tiket_dewasa' => 'decimal:2',
        'harga_tiket_anak' => 'decimal:2',
        'harga_tiket_wisatawan_asing' => 'decimal:2',
        'harga_parkir_motor' => 'decimal:2',
        'harga_parkir_mobil' => 'decimal:2',
        'harga_parkir_bus' => 'decimal:2',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'altitude' => 'decimal:2',
        'jarak_dari_pusat_kota_km' => 'decimal:2',
        'rating_rata_rata' => 'decimal:2',
        'pendapatan_bulan_ini' => 'decimal:2',
        'pendapatan_tahun_ini' => 'decimal:2',
        'pendapatan_keseluruhan' => 'decimal:2',
        'is_featured' => 'boolean',
        'is_recommended' => 'boolean',
        'guide_tersedia' => 'boolean',
        'sertifikat_halal' => 'boolean',
        'verified_at' => 'datetime',
    ];

    // Boot method untuk auto-generate slug
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($objek) {
            if (empty($objek->slug)) {
                $objek->slug = Str::slug($objek->nama_wisata);
                
                // Ensure unique slug
                $originalSlug = $objek->slug;
                $counter = 1;
                while (static::where('slug', $objek->slug)->exists()) {
                    $objek->slug = $originalSlug . '-' . $counter;
                    $counter++;
                }
            }
        });
    }

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function lastUpdatedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'last_updated_by');
    }

    public function verifiedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    public function tiketWisata(): HasMany
    {
        return $this->hasMany(TiketWisata::class);
    }

    public function transaksi()
    {
        return $this->morphMany(Transaksi::class, 'transactionable');
    }

    // Scopes
    public function scopeAktif($query)
    {
        return $query->where('status', 'aktif');
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopeRecommended($query)
    {
        return $query->where('is_recommended', true);
    }

    public function scopeByKategori($query, $kategori)
    {
        return $query->where('kategori_wisata', $kategori);
    }

    public function scopeVerified($query)
    {
        return $query->whereNotNull('verified_at');
    }

    public function scopeNearby($query, $latitude, $longitude, $radiusKm = 50)
    {
        return $query->selectRaw("
            *, 
            (6371 * acos(cos(radians(?)) * cos(radians(latitude)) * cos(radians(longitude) - radians(?)) + sin(radians(?)) * sin(radians(latitude)))) AS distance
        ", [$latitude, $longitude, $latitude])
        ->having('distance', '<', $radiusKm)
        ->orderBy('distance');
    }

    // Accessors & Mutators
    public function getFotoUtamaUrlAttribute()
    {
        if ($this->foto_utama) {
            return asset('storage/wisata/' . $this->foto_utama);
        }
        return asset('images/default-tourism.png');
    }

    public function getGaleriFotoUrlsAttribute()
    {
        if ($this->galeri_foto) {
            return collect($this->galeri_foto)->map(function ($foto) {
                return asset('storage/wisata/galeri/' . $foto);
            })->toArray();
        }
        return [];
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'aktif' => '<span class="badge bg-success">Aktif</span>',
            'nonaktif' => '<span class="badge bg-secondary">Non-aktif</span>',
            'maintenance' => '<span class="badge bg-warning">Maintenance</span>',
            'tutup_sementara' => '<span class="badge bg-danger">Tutup Sementara</span>',
            'pending_review' => '<span class="badge bg-info">Pending Review</span>',
        ];
        
        return $badges[$this->status] ?? '<span class="badge bg-secondary">Unknown</span>';
    }

    public function getKategoriLabelAttribute()
    {
        $labels = [
            'alam' => 'Wisata Alam',
            'budaya' => 'Wisata Budaya',
            'sejarah' => 'Wisata Sejarah',
            'religi' => 'Wisata Religi',
            'kuliner' => 'Wisata Kuliner',
            'adventure' => 'Wisata Adventure',
            'edukasi' => 'Wisata Edukasi',
            'agrowisata' => 'Agrowisata',
            'pantai' => 'Wisata Pantai',
            'gunung' => 'Wisata Gunung',
            'air_terjun' => 'Air Terjun',
            'danau' => 'Wisata Danau',
            'goa' => 'Wisata Goa',
            'taman' => 'Taman Wisata',
            'museum' => 'Museum',
            'candi' => 'Candi',
            'masjid' => 'Masjid Bersejarah',
            'gereja' => 'Gereja Bersejarah',
            'pura' => 'Pura',
            'vihara' => 'Vihara',
            'lainnya' => 'Lainnya'
        ];
        
        return $labels[$this->kategori_wisata] ?? 'Unknown';
    }

    public function getTingkatKesulitanLabelAttribute()
    {
        $labels = [
            'mudah' => '<span class="badge bg-success">Mudah</span>',
            'sedang' => '<span class="badge bg-warning">Sedang</span>',
            'sulit' => '<span class="badge bg-danger">Sulit</span>',
            'ekstrem' => '<span class="badge bg-dark">Ekstrem</span>',
        ];
        
        return $labels[$this->tingkat_kesulitan] ?? '<span class="badge bg-secondary">-</span>';
    }

    public function getJamOperasionalAttribute()
    {
        if ($this->jam_buka && $this->jam_tutup) {
            return $this->jam_buka . ' - ' . $this->jam_tutup;
        }
        return '24 Jam';
    }

    public function getCoordinatesAttribute()
    {
        if ($this->latitude && $this->longitude) {
            return $this->latitude . ',' . $this->longitude;
        }
        return null;
    }

    public function getHargaTiketFormattedAttribute()
    {
        if ($this->harga_tiket_dewasa > 0) {
            return 'Rp ' . number_format($this->harga_tiket_dewasa, 0, ',', '.');
        }
        return 'Gratis';
    }

    public function getJarakAttribute()
    {
        if ($this->jarak_dari_pusat_kota_km > 0) {
            return number_format($this->jarak_dari_pusat_kota_km, 1) . ' km';
        }
        return 'Pusat kota';
    }

    // Methods
    public function isOpen()
    {
        if (!$this->jam_buka || !$this->jam_tutup) {
            return true; // 24 jam
        }
        
        $now = now()->format('H:i');
        $jamBuka = $this->jam_buka;
        $jamTutup = $this->jam_tutup;
        
        // Handle overnight hours (e.g., 22:00 - 06:00)
        if ($jamTutup < $jamBuka) {
            return $now >= $jamBuka || $now <= $jamTutup;
        }
        
        return $now >= $jamBuka && $now <= $jamTutup;
    }

    public function isOperationalToday()
    {
        $today = strtolower(now()->format('l')); // monday, tuesday, etc.
        $hariIndonesia = [
            'monday' => 'senin',
            'tuesday' => 'selasa',
            'wednesday' => 'rabu',
            'thursday' => 'kamis',
            'friday' => 'jumat',
            'saturday' => 'sabtu',
            'sunday' => 'minggu'
        ];
        
        $hariIni = $hariIndonesia[$today] ?? $today;
        
        // Check if today is in operational days
        if ($this->hari_operasional && !in_array($hariIni, $this->hari_operasional)) {
            return false;
        }
        
        // Check if today is in holiday list
        if ($this->hari_libur && in_array($hariIni, $this->hari_libur)) {
            return false;
        }
        
        return true;
    }

    public function updateRating($rating, $increment = true)
    {
        if ($increment) {
            $totalRating = ($this->rating_rata_rata * $this->jumlah_review) + $rating;
            $newJumlahReview = $this->jumlah_review + 1;
            $newRating = $totalRating / $newJumlahReview;
            
            $this->update([
                'rating_rata_rata' => round($newRating, 2),
                'jumlah_review' => $newJumlahReview
            ]);
        }
    }

    public function incrementPengunjung($jumlah = 1)
    {
        $this->increment('total_pengunjung_bulan_ini', $jumlah);
        $this->increment('total_pengunjung_tahun_ini', $jumlah);
        $this->increment('total_pengunjung_keseluruhan', $jumlah);
    }

    public function addPendapatan($amount)
    {
        $this->increment('pendapatan_bulan_ini', $amount);
        $this->increment('pendapatan_tahun_ini', $amount);
        $this->increment('pendapatan_keseluruhan', $amount);
    }

    public function resetStatistikBulanan()
    {
        $this->update([
            'total_pengunjung_bulan_ini' => 0,
            'pendapatan_bulan_ini' => 0
        ]);
    }

    public function resetStatistikTahunan()
    {
        $this->update([
            'total_pengunjung_tahun_ini' => 0,
            'pendapatan_tahun_ini' => 0
        ]);
    }

    public function getDistance($latitude, $longitude)
    {
        if (!$this->latitude || !$this->longitude) {
            return null;
        }
        
        $earthRadius = 6371; // km
        
        $latFrom = deg2rad($this->latitude);
        $lonFrom = deg2rad($this->longitude);
        $latTo = deg2rad($latitude);
        $lonTo = deg2rad($longitude);
        
        $latDelta = $latTo - $latFrom;
        $lonDelta = $lonTo - $lonFrom;
        
        $a = sin($latDelta / 2) * sin($latDelta / 2) +
             cos($latFrom) * cos($latTo) *
             sin($lonDelta / 2) * sin($lonDelta / 2);
        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
        
        return round($earthRadius * $c, 2);
    }

    public function getGoogleMapsUrl()
    {
        if ($this->latitude && $this->longitude) {
            return "https://www.google.com/maps?q={$this->latitude},{$this->longitude}";
        }
        return null;
    }

    public function getWazeUrl()
    {
        if ($this->latitude && $this->longitude) {
            return "https://waze.com/ul?ll={$this->latitude},{$this->longitude}";
        }
        return null;
    }

    public function getTotalPendapatan()
    {
        return $this->transaksi()
            ->where('status', 'PAID')
            ->sum('amount');
    }

    public function getTotalTiketTerjual()
    {
        return $this->tiketWisata()
            ->where('status_tiket', 'paid')
            ->sum('jumlah_tiket');
    }

    public function getPopularityScore()
    {
        // Calculate popularity based on various factors
        $score = 0;
        
        // Rating weight (40%)
        $score += ($this->rating_rata_rata / 5) * 40;
        
        // Review count weight (20%)
        $reviewScore = min($this->jumlah_review / 100, 1) * 20;
        $score += $reviewScore;
        
        // Visitor count weight (30%)
        $visitorScore = min($this->total_pengunjung_keseluruhan / 10000, 1) * 30;
        $score += $visitorScore;
        
        // Featured/Recommended bonus (10%)
        if ($this->is_featured) $score += 5;
        if ($this->is_recommended) $score += 5;
        
        return round($score, 2);
    }
}
