<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class PelayananSurat extends Model
{
    use HasFactory;

    protected $table = 'pelayanan_surat';

    protected $fillable = [
        'nik',
        'jenis_surat',
        'keperluan',
        'tanggal_pengajuan',
        'status',
        'catatan',
        'file_pdf',
        'tanggal_selesai',
        'template_id',
        'nama_pemohon',
        'alamat',
        'no_telepon',
        'created_by',
        // Enhanced fields
        'nomor_surat',
        'template_surat',
        'qr_code',
        'tanda_tangan_digital',
        'penandatangan_nama',
        'penandatangan_jabatan',
        'penandatangan_nip',
        'file_surat',
        'catatan_internal',
        'biaya_admin',
        'validasi_qr'
    ];

    protected $casts = [
        'validasi_qr' => 'boolean',
        'tanggal_selesai' => 'date',
        'tanggal_pengajuan' => 'date'
    ];

    // Relationships
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function template()
    {
        return $this->belongsTo(TemplateSurat::class, 'template_id');
    }

    public function templateSurat()
    {
        return $this->belongsTo(TemplateSurat::class, 'template_surat');
    }

    public function penduduk()
    {
        return $this->belongsTo(Penduduk::class, 'nik', 'nik');
    }

    // Scope methods
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopePending($query)
    {
        return $query->where('status', 'Pending');
    }

    public function scopeDiproses($query)
    {
        return $query->where('status', 'Diproses');
    }

    public function scopeSelesai($query)
    {
        return $query->where('status', 'Selesai');
    }

    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_surat', $jenis);
    }

    public function scopeTahunIni($query)
    {
        return $query->whereYear('created_at', now()->year);
    }

    public function scopeBulanIni($query)
    {
        return $query->whereMonth('created_at', now()->month)
                    ->whereYear('created_at', now()->year);
    }

    // Method untuk generate nomor surat otomatis
    public function generateNomorSurat()
    {
        if ($this->nomor_surat) {
            return $this->nomor_surat;
        }

        $tahun = now()->year;
        $bulan = now()->format('m');
        $bulanRomawi = $this->getRomawi($bulan);
        
        // Counter berdasarkan jenis surat
        $counter = self::where('jenis_surat', $this->jenis_surat)
                      ->whereYear('created_at', $tahun)
                      ->whereNotNull('nomor_surat')
                      ->count() + 1;
        
        $kodeJenis = $this->getKodeSurat($this->jenis_surat);
        
        // Format: 001/SK-DESA/I/2025
        $format = str_pad($counter, 3, '0', STR_PAD_LEFT) . '/' . 
                  $kodeJenis . '/' . 
                  $bulanRomawi . '/' . 
                  $tahun;
        
        $this->nomor_surat = $format;
        $this->save();
        
        return $format;
    }

    // Method untuk generate QR Code untuk validasi
    public function generateQrCode()
    {
        if ($this->qr_code) {
            return $this->qr_code;
        }

        // Generate unique QR code data
        $qrData = json_encode([
            'nomor_surat' => $this->nomor_surat,
            'nik' => $this->nik,
            'nama' => $this->nama_pemohon,
            'jenis_surat' => $this->jenis_surat,
            'tanggal' => $this->created_at->format('Y-m-d'),
            'hash' => hash('sha256', $this->nomor_surat . $this->nik . $this->created_at)
        ]);
        
        $this->qr_code = base64_encode($qrData);
        $this->save();
        
        return $this->qr_code;
    }

    // Method untuk validasi QR Code
    public static function validateQrCode($qrCode)
    {
        try {
            $qrData = json_decode(base64_decode($qrCode), true);
            
            if (!$qrData || !isset($qrData['nomor_surat'])) {
                return false;
            }

            $surat = self::where('nomor_surat', $qrData['nomor_surat'])->first();
            
            if (!$surat) {
                return false;
            }

            // Verify hash
            $expectedHash = hash('sha256', $surat->nomor_surat . $surat->nik . $surat->created_at);
            
            return isset($qrData['hash']) && $qrData['hash'] === $expectedHash;
            
        } catch (\Exception $e) {
            return false;
        }
    }

    // Method untuk proses surat menggunakan template
    public function processWithTemplate($templateId, $variables = [])
    {
        $template = TemplateSurat::find($templateId);
        
        if (!$template) {
            return false;
        }

        // Auto-fill variables from pemohon data
        $defaultVariables = [
            'nama_pemohon' => $this->nama_pemohon,
            'nik' => $this->nik,
            'alamat' => $this->alamat,
            'keperluan' => $this->keperluan,
            'tanggal_surat' => now()->format('d F Y'),
            'nomor_surat' => $this->generateNomorSurat()
        ];

        // Get data from penduduk if available
        if ($this->penduduk) {
            $defaultVariables = array_merge($defaultVariables, [
                'tempat_lahir' => $this->penduduk->tempat_lahir,
                'tanggal_lahir' => $this->penduduk->tanggal_lahir?->format('d F Y'),
                'jenis_kelamin' => $this->penduduk->jenis_kelamin,
                'agama' => $this->penduduk->agama,
                'pekerjaan' => $this->penduduk->pekerjaan,
                'status_perkawinan' => $this->penduduk->status_perkawinan,
                'rt' => $this->penduduk->rt,
                'rw' => $this->penduduk->rw,
                'dusun' => $this->penduduk->dusun
            ]);
        }

        $allVariables = array_merge($defaultVariables, $variables);
        
        $this->template_surat = $templateId;
        $this->save();

        return $template->replaceVariables($allVariables);
    }

    // Method untuk mark as complete
    public function markAsComplete($penandatangan = null, $fileSurat = null)
    {
        $this->status = 'Selesai';
        $this->tanggal_selesai = now();
        
        if ($penandatangan) {
            $this->penandatangan_nama = $penandatangan['nama'] ?? null;
            $this->penandatangan_jabatan = $penandatangan['jabatan'] ?? null;
            $this->penandatangan_nip = $penandatangan['nip'] ?? null;
            $this->tanda_tangan_digital = $penandatangan['tanda_tangan'] ?? null;
        }
        
        if ($fileSurat) {
            $this->file_surat = $fileSurat;
        }
        
        // Generate QR code for validation
        $this->generateQrCode();
        
        $this->save();
        
        return $this;
    }

    // Method untuk generate PDF surat otomatis
    public function generatePDF()
    {
        if (!$this->template_surat && !$this->template_id) {
            throw new \Exception('Template surat belum dipilih');
        }

        $template = $this->template ?? $this->templateSurat ?? TemplateSurat::find($this->template_surat);
        
        if (!$template) {
            throw new \Exception('Template tidak ditemukan');
        }

        // Generate nomor surat jika belum ada
        if (!$this->nomor_surat) {
            $this->generateNomorSurat();
        }

        // Prepare data untuk template
        $data = $this->preparePdfData();
        
        // Generate content dari template
        $content = $template->replaceVariables($data);
        
        // Generate QR Code
        $this->generateQrCode();
        
        // Generate PDF menggunakan simple HTML to PDF
        $pdfContent = $this->createPdfFromContent($content, $data);
        
        // Save PDF file
        $fileName = $this->savePdfFile($pdfContent);
        
        return $fileName;
    }

    // Method untuk prepare data PDF
    private function preparePdfData()
    {
        $pendudukData = $this->penduduk ? [
            'nama' => $this->penduduk->nama,
            'nik' => $this->penduduk->nik,
            'alamat' => $this->penduduk->alamat,
            'tempat_lahir' => $this->penduduk->tempat_lahir,
            'tanggal_lahir' => $this->penduduk->tanggal_lahir?->format('d F Y'),
            'jenis_kelamin' => $this->penduduk->jenis_kelamin,
            'agama' => $this->penduduk->agama,
            'pekerjaan' => $this->penduduk->pekerjaan,
            'status_perkawinan' => $this->penduduk->status_perkawinan,
            'rt' => $this->penduduk->rt,
            'rw' => $this->penduduk->rw,
            'dusun' => $this->penduduk->dusun,
            'no_kk' => $this->penduduk->no_kk,
        ] : [];

        // Data dari setting desa
        $setting = \App\Models\Setting::first();
        $desaData = $setting ? [
            'nama_desa' => $setting->nama_desa ?? 'Desa',
            'nama_kecamatan' => $setting->nama_kecamatan ?? 'Kecamatan',
            'nama_kabupaten' => $setting->nama_kabupaten ?? 'Kabupaten',
            'nama_provinsi' => $setting->nama_provinsi ?? 'Provinsi',
            'kode_pos' => $setting->kode_pos ?? '',
            'alamat_kantor' => $setting->alamat_kantor ?? '',
            'telepon' => $setting->telepon ?? '',
            'email' => $setting->email ?? '',
            'website' => $setting->website ?? '',
        ] : [
            'nama_desa' => 'Desa',
            'nama_kecamatan' => 'Kecamatan', 
            'nama_kabupaten' => 'Kabupaten',
            'nama_provinsi' => 'Provinsi'
        ];

        return array_merge($pendudukData, $desaData, [
            'nomor_surat' => $this->nomor_surat,
            'jenis_surat' => $this->jenis_surat,
            'keperluan' => $this->keperluan,
            'tanggal_surat' => now()->format('d F Y'),
            'tanggal_pengajuan' => $this->tanggal_pengajuan->format('d F Y'),
            'nama_pemohon' => $this->nama_pemohon ?? ($this->penduduk->nama ?? ''),
            'alamat_pemohon' => $this->alamat ?? ($this->penduduk->alamat ?? ''),
            'no_telepon' => $this->no_telepon ?? ($this->penduduk->no_wa ?? ''),
            'qr_code_data' => $this->qr_code,
            'penandatangan_nama' => $this->penandatangan_nama ?? 'Kepala Desa',
            'penandatangan_jabatan' => $this->penandatangan_jabatan ?? 'Kepala Desa',
            'penandatangan_nip' => $this->penandatangan_nip ?? '',
        ]);
    }

    // Method untuk create PDF dari content
    private function createPdfFromContent($content, $data)
    {
        // Untuk sementara, kita buat implementasi sederhana
        // Bisa diupgrade ke DomPDF/TCPDF nanti jika diperlukan
        return $this->createSimplePdf($content, $data);
    }

    // Method untuk build HTML PDF
    private function buildPdfHtml($content, $data)
    {
        $qrCodeImage = '';
        if ($this->qr_code) {
            // Generate QR code image menggunakan QRCodeService
            try {
                $qrService = app(\App\Services\QRCodeService::class);
                $qrImageBase64 = $qrService->generateSuratQRCode($this);
                
                if ($qrImageBase64) {
                    $qrCodeImage = '<div style="text-align: center; margin: 20px 0;">
                        <p style="font-size: 10px;">QR Code untuk validasi surat</p>
                        <img src="' . $qrImageBase64 . '" style="width: 100px; height: 100px;" />
                    </div>';
                }
            } catch (\Exception $e) {
                $qrCodeImage = '<div style="text-align: center; margin: 20px 0;">
                    <p style="font-size: 10px;">QR Code untuk validasi surat</p>
                    <div style="border: 1px solid #000; width: 100px; height: 100px; margin: 0 auto; display: flex; align-items: center; justify-content: center;">
                        QR CODE
                    </div>
                </div>';
            }
        }

        return '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="utf-8">
            <title>' . $data['jenis_surat'] . '</title>
            <style>
                body { 
                    font-family: "Times New Roman", serif; 
                    font-size: 12pt; 
                    line-height: 1.5;
                    margin: 2cm;
                }
                .header { 
                    text-align: center; 
                    border-bottom: 3px solid #000; 
                    padding-bottom: 20px; 
                    margin-bottom: 30px; 
                }
                .header h1 { 
                    font-size: 18pt; 
                    font-weight: bold; 
                    margin: 5px 0; 
                }
                .header h2 { 
                    font-size: 16pt; 
                    font-weight: bold; 
                    margin: 5px 0; 
                }
                .header p { 
                    font-size: 10pt; 
                    margin: 2px 0; 
                }
                .content { 
                    text-align: justify; 
                    margin: 30px 0; 
                }
                .signature { 
                    margin-top: 50px; 
                    float: right; 
                    text-align: center; 
                    width: 200px; 
                }
                .signature-space { 
                    height: 80px; 
                }
                .footer { 
                    position: fixed; 
                    bottom: 2cm; 
                    left: 2cm; 
                    right: 2cm; 
                    font-size: 8pt; 
                    text-align: center; 
                }
                .nomor-surat {
                    text-align: center;
                    margin: 20px 0;
                    font-weight: bold;
                }
            </style>
        </head>
        <body>
            <div class="header">
                <h1>PEMERINTAH ' . strtoupper($data['nama_kabupaten'] ?? 'KABUPATEN') . '</h1>
                <h1>KECAMATAN ' . strtoupper($data['nama_kecamatan'] ?? 'KECAMATAN') . '</h1>
                <h2>' . strtoupper($data['nama_desa'] ?? 'DESA') . '</h2>
                <p>' . ($data['alamat_kantor'] ?? 'Alamat Kantor Desa') . '</p>
                <p>Telp: ' . ($data['telepon'] ?? '-') . ' | Email: ' . ($data['email'] ?? '-') . '</p>
            </div>
            
            <div class="nomor-surat">
                <u>' . strtoupper($data['jenis_surat']) . '</u><br>
                Nomor: ' . $data['nomor_surat'] . '
            </div>
            
            <div class="content">
                ' . nl2br($content) . '
            </div>
            
            <div class="signature">
                <p>' . ($data['nama_desa'] ?? 'Desa') . ', ' . $data['tanggal_surat'] . '</p>
                <p>' . $data['penandatangan_jabatan'] . '</p>
                <div class="signature-space"></div>
                <p><u>' . $data['penandatangan_nama'] . '</u></p>
                ' . ($data['penandatangan_nip'] ? '<p>NIP: ' . $data['penandatangan_nip'] . '</p>' : '') . '
            </div>
            
            ' . $qrCodeImage . '
            
            <div class="footer">
                <p>Dokumen ini digenerate otomatis oleh Sistem Informasi Desa pada ' . now()->format('d F Y H:i:s') . '</p>
            </div>
        </body>
        </html>';
    }

    // Method untuk create simple PDF
    private function createSimplePdf($content, $data)
    {
        // Implementasi sederhana - menyimpan sebagai HTML yang bisa diprint ke PDF
        $html = $this->buildPdfHtml($content, $data);
        
        // Untuk development, kita save sebagai HTML dulu
        // Nanti bisa diupgrade ke PDF library yang proper
        return $html;
    }

    // Method untuk save PDF file
    private function savePdfFile($pdfContent)
    {
        $fileName = 'surat_' . $this->id . '_' . time() . '.html';
        $filePath = 'public/surat/' . $fileName;
        
        // Ensure directory exists
        if (!file_exists(storage_path('app/public/surat'))) {
            mkdir(storage_path('app/public/surat'), 0755, true);
        }
        
        // Save HTML file (nanti bisa diconvert ke PDF)
        Storage::put($filePath, $pdfContent);
        
        // Update database
        $this->file_pdf = $fileName;
        $this->save();
        
        return $fileName;
    }

    // Helper methods
    private function getKodeSurat($jenisSurat)
    {
        return match($jenisSurat) {
            'Surat Keterangan Domisili' => 'SKD-DESA',
            'Surat Keterangan Kelahiran' => 'SKL-DESA',
            'Surat Keterangan Kematian' => 'SKM-DESA',
            'Surat Keterangan Usaha' => 'SKU-DESA',
            'Surat Keterangan Penghasilan' => 'SKP-DESA',
            'Surat Keterangan Catatan Kepolisian' => 'SKCK-DESA',
            'Surat Pengantar Nikah' => 'SPN-DESA',
            'Surat Keterangan Tidak Mampu' => 'SKTM-DESA',
            'Surat Rekomendasi' => 'REK-DESA',
            default => 'SK-DESA'
        };
    }

    private function getRomawi($bulan)
    {
        $romawi = [
            1 => 'I', 2 => 'II', 3 => 'III', 4 => 'IV',
            5 => 'V', 6 => 'VI', 7 => 'VII', 8 => 'VIII',
            9 => 'IX', 10 => 'X', 11 => 'XI', 12 => 'XII'
        ];
        
        return $romawi[(int)$bulan] ?? 'I';
    }

    // Attribute untuk status badge
    public function getStatusBadgeColorAttribute()
    {
        return match($this->status) {
            'Pending' => 'warning',
            'Diproses' => 'info',
            'Selesai' => 'success',
            'Ditolak' => 'danger',
            default => 'secondary'
        };
    }

    // Method untuk get statistik pelayanan
    public static function getStatistikPelayanan($period = 'month')
    {
        $query = self::query();
        
        switch ($period) {
            case 'today':
                $query->whereDate('created_at', today());
                break;
            case 'week':
                $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
                break;
            case 'month':
                $query->bulanIni();
                break;
            case 'year':
                $query->tahunIni();
                break;
        }

        return [
            'total_permohonan' => $query->count(),
            'pending' => $query->where('status', 'Pending')->count(),
            'diproses' => $query->where('status', 'Diproses')->count(),
            'selesai' => $query->where('status', 'Selesai')->count(),
            'ditolak' => $query->where('status', 'Ditolak')->count(),
            'by_jenis' => $query->selectRaw('jenis_surat, COUNT(*) as total')
                               ->groupBy('jenis_surat')
                               ->pluck('total', 'jenis_surat')
        ];
    }
} 
